/**
 * Dashboard JavaScript - Enhanced for API Management System
 */

// Dashboard Management System
window.Dashboard = {
    charts: {},
    
    // Initialize all dashboard components
    init: function() {
        this.initializeCharts();
        this.setupRefreshHandlers();
        this.startRealTimeUpdates();
    },

    // Initialize all charts
    initializeCharts: function() {
        this.initRequestsChart();
        this.initSuccessRateChart();
        this.initResponseTimeChart();
    },

    // Initialize Request Volume Chart
    initRequestsChart: function() {
        const ctx = document.getElementById('requestsChart');
        if (!ctx) return;

        const gradient = ctx.getContext('2d').createLinearGradient(0, 0, 0, 400);
        gradient.addColorStop(0, 'rgba(75, 192, 192, 0.6)');
        gradient.addColorStop(1, 'rgba(75, 192, 192, 0.1)');

        this.charts.requests = new Chart(ctx, {
            type: 'line',
            data: {
                labels: ['Sen', 'Sel', 'Rab', 'Kam', 'Jum', 'Sab', 'Min'],
                datasets: [{
                    label: 'Permintaan API',
                    data: [120, 190, 300, 500, 200, 300, 450],
                    borderColor: 'rgb(75, 192, 192)',
                    backgroundColor: gradient,
                    borderWidth: 3,
                    tension: 0.4,
                    fill: true,
                    pointBackgroundColor: 'rgb(75, 192, 192)',
                    pointBorderColor: '#ffffff',
                    pointBorderWidth: 2,
                    pointRadius: 5,
                    pointHoverRadius: 8
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                animation: {
                    duration: 2000,
                    easing: 'easeInOutQuart'
                },
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0,0,0,0.05)'
                        },
                        ticks: {
                            color: '#666'
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        },
                        ticks: {
                            color: '#666'
                        }
                    }
                }
            }
        });
    },

    // Initialize Success Rate Chart
    initSuccessRateChart: function() {
        const ctx = document.getElementById('successChart');
        if (!ctx) return;

        const successRate = window.dashboardData ? window.dashboardData.success_rate : 95;
        const failureRate = 100 - successRate;

        this.charts.success = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: ['Berhasil', 'Gagal'],
                datasets: [{
                    data: [successRate, failureRate],
                    backgroundColor: [
                        '#28a745',
                        '#dc3545'
                    ],
                    borderWidth: 0,
                    hoverOffset: 10
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                animation: {
                    animateRotate: true,
                    duration: 2000,
                    easing: 'easeInOutQuart'
                },
                plugins: {
                    legend: {
                        display: false
                    }
                },
                cutout: '70%'
            }
        });
    },

    // Initialize Response Time Chart
    initResponseTimeChart: function() {
        const ctx = document.getElementById('responseTimeChart');
        if (!ctx) return;

        this.charts.responseTime = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: ['API 1', 'API 2', 'API 3', 'API 4', 'API 5'],
                datasets: [{
                    label: 'Waktu Respons (ms)',
                    data: [120, 85, 200, 150, 90],
                    backgroundColor: [
                        '#FF6384', '#36A2EB', '#FFCE56', '#4BC0C0', '#9966FF'
                    ],
                    borderColor: [
                        '#FF6384', '#36A2EB', '#FFCE56', '#4BC0C0', '#9966FF'
                    ],
                    borderWidth: 2,
                    borderRadius: 8,
                    borderSkipped: false,
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                animation: {
                    duration: 2000,
                    easing: 'easeInOutQuart'
                },
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0,0,0,0.05)'
                        },
                        ticks: {
                            color: '#666'
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        },
                        ticks: {
                            color: '#666'
                        }
                    }
                }
            }
        });
    },

    // Setup refresh functionality
    setupRefreshHandlers: function() {
        // Setup refresh button if exists
        const refreshBtn = document.querySelector('#refreshDashboard');
        if (refreshBtn) {
            refreshBtn.addEventListener('click', () => {
                this.refreshDashboard(refreshBtn);
            });
        }
    },

    // Refresh dashboard
    refreshDashboard: function(button) {
        if (button) {
            const originalHtml = button.innerHTML;
            button.disabled = true;
            button.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Memperbarui...';

            setTimeout(() => {
                this.refreshAllCharts();
                this.refreshStats();

                if (window.Toast) {
                    Toast.success('Dashboard berhasil diperbarui!');
                }

                button.disabled = false;
                button.innerHTML = originalHtml;
            }, 2000);
        }
    },

    // Refresh all charts
    refreshAllCharts: function() {
        Object.keys(this.charts).forEach(chartKey => {
            const chart = this.charts[chartKey];
            chart.data.datasets[0].data = chart.data.datasets[0].data.map(() => 
                Math.floor(Math.random() * 400) + 100
            );
            chart.update('active');
        });
    },

    // Refresh statistics
    refreshStats: function() {
        const statElements = document.querySelectorAll('[data-stat-value]');
        statElements.forEach(element => {
            const currentValue = parseInt(element.dataset.statValue);
            const newValue = Math.floor(currentValue * (0.9 + Math.random() * 0.2));
            
            element.style.transform = 'scale(1.1)';
            setTimeout(() => {
                this.animateCounter(element, newValue);
                element.style.transform = 'scale(1)';
                element.dataset.statValue = newValue;
            }, 200);
        });
    },

    // Animate counter
    animateCounter: function(element, target, duration = 1000) {
        const start = parseInt(element.textContent.replace(/[^\d]/g, '')) || 0;
        const increment = (target - start) / (duration / 16);
        let current = start;
        
        const timer = setInterval(() => {
            current += increment;
            if ((increment > 0 && current >= target) || (increment < 0 && current <= target)) {
                current = target;
                clearInterval(timer);
            }
            
            const formattedValue = Math.floor(current).toLocaleString();
            const originalText = element.textContent;
            if (originalText.includes('ms')) {
                element.textContent = formattedValue + 'ms';
            } else if (originalText.includes('%')) {
                element.textContent = formattedValue + '%';
            } else {
                element.textContent = formattedValue;
            }
        }, 16);
    },

    // Start real-time updates
    startRealTimeUpdates: function() {
        setInterval(() => {
            const statElements = document.querySelectorAll('[data-stat]');
            statElements.forEach(element => {
                element.style.transform = 'scale(1.05)';
                setTimeout(() => {
                    element.style.transform = 'scale(1)';
                }, 200);
            });
        }, 30000);
    }
};

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    if (typeof Chart !== 'undefined') {
        setTimeout(() => {
            window.Dashboard.init();
        }, 500);
    }
}); 