@extends('layouts.app')

@section('title', 'Edit API Client')

@section('breadcrumb')
    <li class="breadcrumb-item">
        <a href="{{ route('api-clients.index') }}">API Clients</a>
    </li>
    <li class="breadcrumb-item">
        <a href="{{ route('api-clients.show', $apiClient) }}">{{ $apiClient->name }}</a>
    </li>
    <li class="breadcrumb-item active">Edit</li>
@endsection

@section('content')
<div class="container-fluid">
    <!-- Alert Messages -->
    <div id="alert-container">
        @if(session('success'))
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="fas fa-check-circle me-2"></i>{{ session('success') }}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        @endif
        
        @if(session('error'))
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="fas fa-exclamation-circle me-2"></i>{{ session('error') }}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        @endif
        
        @if($errors->any())
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <strong>Please fix the following errors:</strong>
                <ul class="mb-0 mt-2">
                    @foreach($errors->all() as $error)
                        <li>{{ $error }}</li>
                    @endforeach
                </ul>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        @endif
    </div>

    <div class="row">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h1 class="h3 mb-0 text-gray-800">
                    <i class="fas fa-edit me-2"></i>Edit API Client: {{ $apiClient->name }}
                </h1>
                <div>
                    <a href="{{ route('api-clients.show', $apiClient) }}" class="btn btn-info">
                        <i class="fas fa-eye me-2"></i>View
                    </a>
                    <a href="{{ route('api-clients.index') }}" class="btn btn-secondary">
                        <i class="fas fa-arrow-left me-2"></i>Back to List
                    </a>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-lg-8">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-user-cog me-2"></i>Update API Client Information
                    </h6>
                </div>
                <div class="card-body">
                    <form action="{{ route('api-clients.update', $apiClient) }}" method="POST" id="editForm" novalidate>
                        @csrf
                        @method('PUT')
                        
                        <!-- Basic Information -->
                        <div class="row">
                            <div class="col-md-12">
                                <div class="form-group mb-3">
                                    <label for="name" class="form-label">Client Name <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control @error('name') is-invalid @enderror" 
                                           id="name" name="name" value="{{ old('name', $apiClient->name) }}" required
                                           placeholder="Enter a descriptive name for this API client">
                                    @error('name')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="form-group mb-3">
                            <label for="description" class="form-label">Description</label>
                            <textarea class="form-control @error('description') is-invalid @enderror" 
                                      id="description" name="description" rows="3" 
                                      placeholder="Brief description of this API client and its purpose">{{ old('description', $apiClient->description) }}</textarea>
                            @error('description')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Configuration -->
                        <hr class="my-4">
                        <h5 class="mb-3">
                            <i class="fas fa-cog me-2"></i>Client Configuration
                        </h5>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="rate_limit" class="form-label">Rate Limit (requests per minute)</label>
                                    <input type="number" class="form-control @error('rate_limit') is-invalid @enderror" 
                                           id="rate_limit" name="rate_limit" value="{{ old('rate_limit', $apiClient->rate_limit) }}" 
                                           min="1" placeholder="100">
                                    <small class="form-text text-muted">Maximum number of requests per minute allowed for this client</small>
                                    @error('rate_limit')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <div class="form-check">
                                        <input type="checkbox" class="form-check-input" id="is_active" name="is_active" 
                                               value="1" {{ old('is_active', $apiClient->is_active) ? 'checked' : '' }}>
                                        <label class="form-check-label" for="is_active">
                                            Active Client
                                        </label>
                                    </div>
                                    <small class="form-text text-muted">Inactive clients cannot make API requests</small>
                                </div>
                            </div>
                        </div>

                        <!-- Security -->
                        <hr class="my-4">
                        <h5 class="mb-3">
                            <i class="fas fa-shield-alt me-2"></i>Security Settings
                        </h5>

                        <div class="form-group mb-3">
                            <label for="allowed_ips" class="form-label">Allowed IP Addresses</label>
                            <input type="text" class="form-control @error('allowed_ips') is-invalid @enderror" 
                                   id="allowed_ips" name="allowed_ips" 
                                   value="{{ old('allowed_ips', $apiClient->allowed_ips ? implode(', ', $apiClient->allowed_ips) : '') }}" 
                                   placeholder="192.168.1.1, 10.0.0.1, 203.0.113.0/24">
                            <small class="form-text text-muted">
                                Comma-separated list of IP addresses or CIDR blocks. Leave empty to allow all IPs.
                            </small>
                            @error('allowed_ips')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Current Credentials Info -->
                        <hr class="my-4">
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle me-2"></i>
                            <strong>Note:</strong> Client ID and Client Secret cannot be changed after creation. 
                            If you need new credentials, you'll need to create a new API client.
                        </div>

                        <!-- Form Actions -->
                        <div class="d-flex justify-content-between">
                            <div>
                                <a href="{{ route('api-clients.show', $apiClient) }}" class="btn btn-secondary">
                                    <i class="fas fa-times me-2"></i>Cancel
                                </a>
                            </div>
                            <div>
                                <button type="button" class="btn btn-danger me-2" onclick="confirmDelete()">
                                    <i class="fas fa-trash me-2"></i>Delete Client
                                </button>
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save me-2"></i>Update API Client
                                </button>
                            </div>
                        </div>
                    </form>

                    <!-- Hidden Delete Form -->
                    <form id="deleteForm" action="{{ route('api-clients.destroy', $apiClient) }}" method="POST" style="display: none;">
                        @csrf
                        @method('DELETE')
                    </form>
                </div>
            </div>
        </div>

        <!-- Current Client Info -->
        <div class="col-lg-4">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-info">
                        <i class="fas fa-info-circle me-2"></i>Current Information
                    </h6>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label class="form-label fw-bold">Client ID:</label>
                        <p class="font-monospace small text-gray-800">{{ $apiClient->client_id }}</p>
                    </div>

                    <div class="mb-3">
                        <label class="form-label fw-bold">Created:</label>
                        <p class="text-gray-800">{{ $apiClient->created_at->format('M d, Y H:i:s') }}</p>
                    </div>

                    <div class="mb-3">
                        <label class="form-label fw-bold">Last Used:</label>
                        <p class="text-gray-800">
                            {{ $apiClient->last_used_at ? $apiClient->last_used_at->format('M d, Y H:i:s') : 'Never' }}
                        </p>
                    </div>

                    <div class="mb-3">
                        <label class="form-label fw-bold">Total Tokens:</label>
                        <p class="text-gray-800">{{ $apiClient->tokens()->count() }}</p>
                    </div>
                </div>
            </div>

            <!-- Help Card -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-warning">
                        <i class="fas fa-exclamation-triangle me-2"></i>Important Notes
                    </h6>
                </div>
                <div class="card-body">
                    <p class="small text-muted mb-3">
                        <strong>Changing Status:</strong> Deactivating a client will immediately prevent all API requests using this client.
                    </p>

                    <p class="small text-muted mb-3">
                        <strong>Rate Limits:</strong> Changes to rate limits take effect immediately and apply to all active tokens.
                    </p>

                    <p class="small text-muted mb-0">
                        <strong>IP Restrictions:</strong> Adding IP restrictions will immediately block requests from non-allowed IPs.
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Form validation
    const form = document.getElementById('editForm');
    
    form.addEventListener('submit', function(e) {
        // Basic client-side validation
        const name = document.getElementById('name').value.trim();
        
        if (!name) {
            e.preventDefault();
            alert('Please enter a client name.');
            document.getElementById('name').focus();
            return false;
        }
        
        // Validate IP addresses if provided
        const allowedIps = document.getElementById('allowed_ips').value.trim();
        if (allowedIps) {
            const ips = allowedIps.split(',').map(ip => ip.trim());
            const ipPattern = /^(\d{1,3}\.){3}\d{1,3}(\/\d{1,2})?$/;
            
            for (let ip of ips) {
                if (!ipPattern.test(ip)) {
                    e.preventDefault();
                    alert('Invalid IP address format: ' + ip);
                    document.getElementById('allowed_ips').focus();
                    return false;
                }
            }
        }
    });
});

function confirmDelete() {
    if (confirm('Are you sure you want to delete this API client? This action cannot be undone and will revoke all associated tokens.')) {
        document.getElementById('deleteForm').submit();
    }
}
</script>
@endsection 