@extends('layouts.app')

@section('title', 'Create API Receiver')

@section('content')
<div class="container-fluid px-4">
    <!-- Error Messages -->
    @if($errors->any())
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-triangle me-2"></i>
            <strong>Please fix the following errors:</strong>
            <ul class="mb-0 mt-2">
                @foreach($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif
    
    @if(session('error'))
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-circle me-2"></i>{{ session('error') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    <!-- Page Header -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <div>
            <h1 class="h3 mb-0 text-gray-800">
                <i class="fas fa-plus-circle me-2 text-primary"></i>Create API Receiver
            </h1>
            <p class="text-muted mt-2">Configure a new API receiver to accept incoming data</p>
        </div>
        <a href="{{ route('api-receivers.index') }}" class="btn btn-outline-secondary">
            <i class="fas fa-arrow-left me-2"></i>Back to List
        </a>
    </div>

    <div class="row">
        <!-- Main Form -->
        <div class="col-12">
            <div class="card shadow-sm border-0 mb-4">
                <div class="card-header bg-gradient-primary text-white py-3">
                    <h6 class="m-0 font-weight-bold">
                        <i class="fas fa-cog me-2"></i>Receiver Configuration
                    </h6>
                </div>
                <div class="card-body p-4">
                    <form action="{{ route('api-receivers.store') }}" method="POST" id="receiverForm">
                        @csrf
                        
                        <!-- Basic Information Section -->
                        <div class="section-header mb-4">
                            <h5 class="text-primary mb-3">
                                <i class="fas fa-info-circle me-2"></i>Basic Information
                            </h5>
                        </div>
                        
                        <div class="row g-3">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="name" class="form-label fw-bold">
                                        Receiver Name <span class="text-danger">*</span>
                                    </label>
                                    <input type="text" class="form-control form-control-lg @error('name') is-invalid @enderror" 
                                           id="name" name="name" value="{{ old('name') }}" required 
                                           placeholder="e.g., Customer Data Receiver" autocomplete="off">
                                    <div class="form-text">Give your receiver a descriptive name</div>
                                    @error('name')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="endpoint" class="form-label fw-bold">
                                        Endpoint <span class="text-danger">*</span>
                                    </label>
                                    <div class="input-group input-group-lg">
                                        <span class="input-group-text bg-light">/api/in/</span>
                                        <input type="text" class="form-control @error('endpoint') is-invalid @enderror" 
                                               id="endpoint" name="endpoint" value="{{ old('endpoint') }}" required
                                               placeholder="customer-data" autocomplete="off">
                                        <button type="button" class="btn btn-outline-primary" onclick="generateEndpoint()" title="Generate from name">
                                            <i class="fas fa-magic"></i>
                                        </button>
                                    </div>
                                    <div class="form-text">Unique identifier for this receiver endpoint</div>
                                    @error('endpoint')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row g-3 mt-2">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="target_table" class="form-label fw-bold">
                                        Target Table <span class="text-danger">*</span>
                                    </label>
                                    <select class="form-select form-select-lg @error('target_table') is-invalid @enderror" 
                                            id="target_table" name="target_table" required>
                                        <option value="">Select Target Table</option>
                                        @if(!empty($tables) && is_array($tables))
                                            @foreach($tables as $table)
                                                <option value="{{ $table }}" {{ old('target_table') == $table ? 'selected' : '' }}>{{ $table }}</option>
                                            @endforeach
                                        @else
                                            <option value="" disabled>No tables found</option>
                                        @endif
                                    </select>
                                    <div class="form-text">Database table to store received data</div>
                                    @error('target_table')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="allowed_methods" class="form-label fw-bold">
                                        Allowed Methods <span class="text-danger">*</span>
                                    </label>
                                    <select class="form-select form-select-lg @error('allowed_methods') is-invalid @enderror" 
                                            id="allowed_methods" name="allowed_methods" required>
                                        <option value="">Select Methods</option>
                                        <option value="GET" {{ old('allowed_methods') == 'GET' ? 'selected' : '' }}>GET Only</option>
                                        <option value="POST" {{ old('allowed_methods') == 'POST' ? 'selected' : '' }}>POST Only</option>
                                        <option value="GET,POST" {{ old('allowed_methods', 'GET,POST') == 'GET,POST' ? 'selected' : '' }}>GET + POST</option>
                                        <option value="PUT" {{ old('allowed_methods') == 'PUT' ? 'selected' : '' }}>PUT Only</option>
                                        <option value="DELETE" {{ old('allowed_methods') == 'DELETE' ? 'selected' : '' }}>DELETE Only</option>
                                        <option value="GET,POST,PUT,DELETE" {{ old('allowed_methods') == 'GET,POST,PUT,DELETE' ? 'selected' : '' }}>All Methods</option>
                                    </select>
                                    <div class="form-text">HTTP methods this receiver will accept</div>
                                    @error('allowed_methods')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row g-3 mt-2">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="api_client_id" class="form-label fw-bold">API Client (Optional)</label>
                                    <select class="form-select form-select-lg @error('api_client_id') is-invalid @enderror" 
                                            id="api_client_id" name="api_client_id" onchange="syncAuthFromClient()">
                                        <option value="">No Client Association</option>
                                        @foreach($apiClients as $client)
                                            <option value="{{ $client->id }}" {{ old('api_client_id') == $client->id ? 'selected' : '' }}>
                                                {{ $client->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                    <div class="form-text">Associate with an API client for tracking</div>
                                    @error('api_client_id')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="response_format" class="form-label fw-bold">Response Format</label>
                                    <select class="form-select form-select-lg @error('response_format') is-invalid @enderror" 
                                            id="response_format" name="response_format">
                                        <option value="json" {{ old('response_format', 'json') == 'json' ? 'selected' : '' }}>JSON</option>
                                        <option value="xml" {{ old('response_format') == 'xml' ? 'selected' : '' }}>XML</option>
                                        <option value="plain" {{ old('response_format') == 'plain' ? 'selected' : '' }}>Plain Text</option>
                                    </select>
                                    <div class="form-text">Format for response data</div>
                                    @error('response_format')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <!-- Authentication Section -->
                        <hr class="my-4">
                        <div class="section-header mb-4">
                            <h5 class="text-primary mb-3">
                                <i class="fas fa-shield-alt me-2"></i>Authentication & Security
                            </h5>
                        </div>
                        
                        <div class="row g-3">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="auth_type" class="form-label fw-bold">Authentication Type</label>
                                    <select class="form-select form-select-lg @error('auth_type') is-invalid @enderror" 
                                            id="auth_type" name="auth_type" onchange="toggleAuthFields()">
                                        <option value="none" {{ old('auth_type', 'none') == 'none' ? 'selected' : '' }}>No Authentication</option>
                                        <option value="api_key" {{ old('auth_type') == 'api_key' ? 'selected' : '' }}>API Key</option>
                                        <option value="bearer_token" {{ old('auth_type') == 'bearer_token' ? 'selected' : '' }}>Bearer Token</option>
                                        <option value="basic_auth" {{ old('auth_type') == 'basic_auth' ? 'selected' : '' }}>Basic Auth</option>
                                    </select>
                                    <div class="form-text">Choose how to authenticate incoming requests</div>
                                    @error('auth_type')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-group d-none" id="auth_token_field">
                                    <label for="auth_token" class="form-label fw-bold">Authentication Token</label>
                                    <div class="input-group input-group-lg">
                                        <input type="text" class="form-control @error('auth_token') is-invalid @enderror" 
                                               id="auth_token" name="auth_token" value="{{ old('auth_token') }}"
                                               placeholder="Enter your token" autocomplete="off">
                                        <button type="button" class="btn btn-outline-secondary" onclick="generateToken()" title="Generate Token">
                                            <i class="fas fa-key"></i>
                                        </button>
                                    </div>
                                    <div class="form-text">Token for API authentication</div>
                                    @error('auth_token')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row g-3 mt-2" id="basic_auth_fields" style="display: none;">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="auth_username" class="form-label fw-bold">Username</label>
                                    <input type="text" class="form-control form-control-lg @error('auth_username') is-invalid @enderror" 
                                           id="auth_username" name="auth_username" value="{{ old('auth_username') }}"
                                           placeholder="Enter username" autocomplete="off">
                                    <div class="form-text">Username for Basic Authentication</div>
                                    @error('auth_username')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="auth_password" class="form-label fw-bold">Password</label>
                                    <div class="input-group input-group-lg">
                                        <input type="password" class="form-control @error('auth_password') is-invalid @enderror" 
                                               id="auth_password" name="auth_password" value="{{ old('auth_password') }}"
                                               placeholder="Enter password" autocomplete="off">
                                        <button type="button" class="btn btn-outline-secondary" onclick="togglePassword()" title="Show/Hide Password">
                                            <i class="fas fa-eye" id="password_toggle_icon"></i>
                                        </button>
                                    </div>
                                    <div class="form-text">Password for Basic Authentication</div>
                                    @error('auth_password')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row g-3 mt-2">
                            <div class="col-md-12">
                                <div class="form-group">
                                    <label for="allowed_ips" class="form-label fw-bold">Allowed IP Addresses</label>
                                    <input type="text" class="form-control" id="allowed_ips" name="allowed_ips" 
                                           value="{{ old('allowed_ips') }}" placeholder="192.168.1.1, 10.0.0.0/8, 172.16.0.0/12">
                                    <div class="form-text">Comma-separated list of allowed IP addresses or ranges (leave empty for all IPs)</div>
                                    @error('allowed_ips')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <!-- Data Processing Section -->
                        <hr class="my-4">
                        <div class="section-header mb-4">
                            <h5 class="text-primary mb-3">
                                <i class="fas fa-database me-2"></i>Data Processing
                            </h5>
                        </div>
                        
                        <div class="row g-3">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="duplicate_handling" class="form-label fw-bold">Duplicate Handling</label>
                                    <select class="form-select" id="duplicate_handling" name="duplicate_handling">
                                        <option value="allow" {{ old('duplicate_handling') == 'allow' ? 'selected' : '' }}>Allow Duplicates</option>
                                        <option value="skip" {{ old('duplicate_handling', 'skip') == 'skip' ? 'selected' : '' }}>Skip Duplicates</option>
                                        <option value="update" {{ old('duplicate_handling') == 'update' ? 'selected' : '' }}>Update Existing</option>
                                        <option value="reject" {{ old('duplicate_handling') == 'reject' ? 'selected' : '' }}>Reject Duplicates</option>
                                    </select>
                                    <div class="form-text">How to handle duplicate data</div>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="unique_fields" class="form-label fw-bold">Unique Fields</label>
                                    <input type="text" class="form-control" id="unique_fields" name="unique_fields" 
                                           placeholder="id,email,phone" value="{{ old('unique_fields', 'id') }}">
                                    <div class="form-text">Fields to check for duplicates (comma-separated)</div>
                                </div>
                            </div>
                        </div>

                        <div class="row g-3 mt-2">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="data_filter" class="form-label fw-bold">Data Filter</label>
                                    <select class="form-select" id="data_filter" name="data_filter">
                                        <option value="all" {{ old('data_filter') == 'all' ? 'selected' : '' }}>All Data</option>
                                        <option value="latest_only" {{ old('data_filter', 'latest_only') == 'latest_only' ? 'selected' : '' }}>Latest Only</option>
                                        <option value="today_only" {{ old('data_filter') == 'today_only' ? 'selected' : '' }}>Today Only</option>
                                        <option value="last_24h" {{ old('data_filter') == 'last_24h' ? 'selected' : '' }}>Last 24 Hours</option>
                                        <option value="last_week" {{ old('data_filter') == 'last_week' ? 'selected' : '' }}>Last Week</option>
                                        <option value="new" {{ old('data_filter') == 'new' ? 'selected' : '' }}>New Records Only</option>
                                        <option value="updated" {{ old('data_filter') == 'updated' ? 'selected' : '' }}>Updated Records Only</option>
                                    </select>
                                    <div class="form-text">Filter data based on criteria</div>
                                </div>
                            </div>
                        </div>

                        <div class="row g-3 mt-2">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="max_records_per_request" class="form-label fw-bold">Max Records Per Request</label>
                                    <select class="form-select" id="max_records_per_request" name="max_records_per_request">
                                        <option value="10" {{ old('max_records_per_request') == '10' ? 'selected' : '' }}>10 Records</option>
                                        <option value="25" {{ old('max_records_per_request') == '25' ? 'selected' : '' }}>25 Records</option>
                                        <option value="50" {{ old('max_records_per_request', '50') == '50' ? 'selected' : '' }}>50 Records</option>
                                        <option value="100" {{ old('max_records_per_request') == '100' ? 'selected' : '' }}>100 Records</option>
                                        <option value="500" {{ old('max_records_per_request') == '500' ? 'selected' : '' }}>500 Records</option>
                                        <option value="unlimited" {{ old('max_records_per_request') == 'unlimited' ? 'selected' : '' }}>Unlimited</option>
                                    </select>
                                    <div class="form-text">Maximum records allowed per request</div>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="rate_limit" class="form-label fw-bold">Rate Limit (requests/hour)</label>
                                    <select class="form-select" id="rate_limit" name="rate_limit">
                                        <option value="" {{ old('rate_limit') == '' ? 'selected' : '' }}>No Limit</option>
                                        <option value="10" {{ old('rate_limit') == '10' ? 'selected' : '' }}>10 requests/hour</option>
                                        <option value="60" {{ old('rate_limit', '60') == '60' ? 'selected' : '' }}>60 requests/hour</option>
                                        <option value="100" {{ old('rate_limit') == '100' ? 'selected' : '' }}>100 requests/hour</option>
                                        <option value="500" {{ old('rate_limit') == '500' ? 'selected' : '' }}>500 requests/hour</option>
                                        <option value="1000" {{ old('rate_limit') == '1000' ? 'selected' : '' }}>1000 requests/hour</option>
                                    </select>
                                    <div class="form-text">Limit requests per hour to prevent abuse</div>
                                </div>
                            </div>
                        </div>

                        <!-- Field Mapping Section -->
                        <hr class="my-4">
                        <div class="section-header mb-4">
                            <h5 class="text-primary mb-3">
                                <i class="fas fa-exchange-alt me-2"></i>Field Mapping
                            </h5>
                        </div>
                        
                        <div class="row g-3">
                            <div class="col-md-12">
                                <div class="form-group">
                                    <label for="field_mapping" class="form-label fw-bold">Field Mapping (JSON)</label>
                                    <div class="input-group">
                                        <textarea class="form-control" id="field_mapping" name="field_mapping" rows="4" 
                                                 placeholder='{"request_field": "database_column", "user_name": "name", "user_email": "email"}'>{{ old('field_mapping') }}</textarea>
                                        <button type="button" class="btn btn-outline-secondary" onclick="generateFieldMapping()">
                                            <i class="fas fa-magic"></i> Sample
                                        </button>
                                    </div>
                                    <div class="form-text">Map incoming request fields to database columns (leave empty for direct mapping)</div>
                                    @error('field_mapping')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <!-- Sample Data Section -->
                        <hr class="my-4">
                        <div class="section-header mb-4">
                            <h5 class="text-primary mb-3">
                                <i class="fas fa-file-code me-2"></i>Sample Data & Documentation
                            </h5>
                        </div>
                        
                        <div class="row g-3">
                            <div class="col-md-12">
                                <div class="form-group">
                                    <label for="sample_data" class="form-label fw-bold">Sample Data (JSON)</label>
                                    <div class="input-group">
                                        <textarea class="form-control" id="sample_data" name="sample_data" rows="6" 
                                                 placeholder='{"name": "John Doe", "email": "john@example.com", "age": 30}'>{{ old('sample_data') }}</textarea>
                                        <button type="button" class="btn btn-outline-secondary" onclick="generateSampleData()">
                                            <i class="fas fa-magic"></i> Sample
                                        </button>
                                    </div>
                                    <div class="form-text">Example data structure for documentation purposes</div>
                                    @error('sample_data')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <!-- Settings Section -->
                        <hr class="my-4">
                        <div class="section-header mb-4">
                            <h5 class="text-primary mb-3">
                                <i class="fas fa-cogs me-2"></i>Settings & Options
                            </h5>
                        </div>
                        
                        <div class="row g-3">
                            <div class="col-md-6">
                                <div class="form-check form-switch mb-3">
                                    <input type="hidden" name="auto_create_table" value="0">
                                    <input type="checkbox" class="form-check-input" id="auto_create_table" name="auto_create_table" value="1"
                                           {{ old('auto_create_table') ? 'checked' : '' }}>
                                    <label class="form-check-label fw-bold" for="auto_create_table">
                                        Auto Create Table
                                    </label>
                                    <div class="form-text">Automatically create table if it doesn't exist</div>
                                </div>

                                <div class="form-check form-switch mb-3">
                                    <input type="hidden" name="is_active" value="0">
                                    <input type="checkbox" class="form-check-input" id="is_active" name="is_active" value="1"
                                           {{ old('is_active', true) ? 'checked' : '' }}>
                                    <label class="form-check-label fw-bold" for="is_active">
                                        Active Receiver
                                    </label>
                                    <div class="form-text">Enable this receiver to accept incoming data</div>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-check form-switch mb-3">
                                    <input type="hidden" name="is_bidirectional" value="0">
                                    <input type="checkbox" class="form-check-input" id="is_bidirectional" name="is_bidirectional" value="1"
                                           {{ old('is_bidirectional') ? 'checked' : '' }} onchange="toggleBidirectionalFields()">
                                    <label class="form-check-label fw-bold" for="is_bidirectional">
                                        Enable Bidirectional
                                    </label>
                                    <div class="form-text">Enable two-way communication support</div>
                                </div>
                            </div>
                        </div>

                        <!-- Bidirectional Settings -->
                        <div id="bidirectionalSettings" style="display: none;" class="mt-4">
                            <div class="card border-primary">
                                <div class="card-header bg-primary text-white">
                                    <h6 class="mb-0"><i class="fas fa-exchange-alt me-2"></i>Bidirectional Settings</h6>
                                </div>
                                <div class="card-body">
                                    <div class="row g-3">
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label for="callback_url" class="form-label fw-bold">Callback URL</label>
                                                <input type="url" class="form-control" id="callback_url" name="callback_url" 
                                                       value="{{ old('callback_url') }}" placeholder="https://example.com/callback">
                                                <div class="form-text">URL to send callbacks to</div>
                                            </div>
                                        </div>

                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label for="callback_method" class="form-label fw-bold">Callback Method</label>
                                                <select class="form-select" id="callback_method" name="callback_method">
                                                    <option value="POST" {{ old('callback_method') == 'POST' ? 'selected' : '' }}>POST</option>
                                                    <option value="PUT" {{ old('callback_method') == 'PUT' ? 'selected' : '' }}>PUT</option>
                                                    <option value="PATCH" {{ old('callback_method') == 'PATCH' ? 'selected' : '' }}>PATCH</option>
                                                </select>
                                                <div class="form-text">HTTP method for callbacks</div>
                                            </div>
                                        </div>

                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label for="response_format" class="form-label fw-bold">Response Format</label>
                                                <select class="form-select" id="response_format" name="response_format">
                                                    <option value="json" {{ old('response_format', 'json') == 'json' ? 'selected' : '' }}>JSON</option>
                                                    <option value="xml" {{ old('response_format') == 'xml' ? 'selected' : '' }}>XML</option>
                                                    <option value="plain" {{ old('response_format') == 'plain' ? 'selected' : '' }}>Plain Text</option>
                                                </select>
                                                <div class="form-text">Format for response data</div>
                                            </div>
                                        </div>

                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label for="retry_attempts" class="form-label fw-bold">Retry Attempts</label>
                                                <input type="number" class="form-control" id="retry_attempts" name="retry_attempts" 
                                                       value="{{ old('retry_attempts', 3) }}" min="0" max="10">
                                                <div class="form-text">Number of retry attempts (0-10)</div>
                                            </div>
                                        </div>

                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label for="retry_delay" class="form-label fw-bold">Retry Delay (seconds)</label>
                                                <input type="number" class="form-control" id="retry_delay" name="retry_delay" 
                                                       value="{{ old('retry_delay', 60) }}" min="5" max="3600">
                                                <div class="form-text">Delay between retry attempts (5-3600 seconds)</div>
                                            </div>
                                        </div>

                                        <div class="col-md-6">
                                            <div class="form-check form-switch">
                                                <input type="hidden" name="auto_acknowledge" value="0">
                                                <input type="checkbox" class="form-check-input" id="auto_acknowledge" name="auto_acknowledge" value="1"
                                                       {{ old('auto_acknowledge', true) ? 'checked' : '' }}>
                                                <label class="form-check-label fw-bold" for="auto_acknowledge">
                                                    Auto Acknowledge
                                                </label>
                                                <div class="form-text">Automatically acknowledge received data</div>
                                            </div>
                                        </div>

                                        <div class="col-md-12">
                                            <div class="form-group">
                                                <label for="success_response_template" class="form-label fw-bold">Success Response Template</label>
                                                <textarea class="form-control" id="success_response_template" name="success_response_template" rows="3" 
                                                         placeholder='{"status": "success", "message": "Data received successfully", "data": {data}}'>{{ old('success_response_template') }}</textarea>
                                                <div class="form-text">Template for success response (use {data} for dynamic content)</div>
                                            </div>
                                        </div>

                                        <div class="col-md-12">
                                            <div class="form-group">
                                                <label for="error_response_template" class="form-label fw-bold">Error Response Template</label>
                                                <textarea class="form-control" id="error_response_template" name="error_response_template" rows="3" 
                                                         placeholder='{"status": "error", "message": "{message}", "code": {code}}'>{{ old('error_response_template') }}</textarea>
                                                <div class="form-text">Template for error response (use {message} and {code} for dynamic content)</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Scheduling Configuration -->
                        <hr class="my-4">
                        <div class="section-header mb-4">
                            <h5 class="text-primary mb-3">
                                <i class="fas fa-clock me-2"></i>Scheduling & Processing
                            </h5>
                        </div>
                        
                        <div class="row g-3">
                            <div class="col-md-6">
                                <div class="form-check form-switch mb-3">
                                    <input type="hidden" name="enable_scheduling" value="0">
                                    <input type="checkbox" class="form-check-input" id="enable_scheduling" name="enable_scheduling" value="1"
                                           {{ old('enable_scheduling') ? 'checked' : '' }} onchange="toggleSchedulingFields()">
                                    <label class="form-check-label fw-bold" for="enable_scheduling">
                                        Enable Scheduling
                                    </label>
                                    <div class="form-text">Enable automated data processing</div>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-group" id="schedule_type_field" style="display: none;">
                                    <label for="schedule_type" class="form-label fw-bold">Schedule Type</label>
                                    <select class="form-select" id="schedule_type" name="schedule_type">
                                        <option value="manual" {{ old('schedule_type') == 'manual' ? 'selected' : '' }}>Manual</option>
                                        <option value="interval" {{ old('schedule_type') == 'interval' ? 'selected' : '' }}>Interval</option>
                                        <option value="cron" {{ old('schedule_type') == 'cron' ? 'selected' : '' }}>Cron Expression</option>
                                    </select>
                                    <div class="form-text">How to schedule data processing</div>
                                </div>
                            </div>
                        </div>

                        <div class="row g-3 mt-2" id="scheduling_details" style="display: none;">
                            <div class="col-md-6">
                                <div class="form-group" id="schedule_interval_field" style="display: none;">
                                    <label for="schedule_interval" class="form-label fw-bold">Schedule Interval (minutes)</label>
                                    <input type="number" class="form-control" id="schedule_interval" name="schedule_interval" 
                                           value="{{ old('schedule_interval', 60) }}" min="1" max="1440">
                                    <div class="form-text">Interval between processing runs (1-1440 minutes)</div>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-group" id="schedule_time_field" style="display: none;">
                                    <label for="schedule_time" class="form-label fw-bold">Schedule Time</label>
                                    <input type="time" class="form-control" id="schedule_time" name="schedule_time" 
                                           value="{{ old('schedule_time', '00:00') }}">
                                    <div class="form-text">Time to run scheduled processing</div>
                                </div>
                            </div>
                        </div>

                        <!-- Description -->
                        <hr class="my-4">
                        <div class="form-group">
                            <label for="description" class="form-label fw-bold">Description</label>
                            <textarea class="form-control" id="description" name="description" rows="3" 
                                     placeholder="Optional description for this receiver">{{ old('description') }}</textarea>
                            <div class="form-text">Add a description to help identify this receiver</div>
                        </div>

                        <!-- Submit Buttons -->
                        <hr class="my-4">
                        <div class="text-center">
                            <div class="row justify-content-center g-3">
                                <div class="col-12">
                                    <button type="submit" class="btn btn-success btn-lg me-3" id="saveBtn">
                                        <i class="fas fa-save me-2"></i>Save API Receiver
                                    </button>
                                    <button type="button" class="btn btn-primary btn-lg me-3" id="altSubmitBtn">
                                        <i class="fas fa-rocket me-2"></i>Submit Now
                                    </button>
                                    <button type="button" class="btn btn-info btn-lg me-3" id="simpleSubmitBtn">
                                        <i class="fas fa-paper-plane me-2"></i>Simple Submit
                                    </button>
                                </div>
                                <div class="col-12">
                                    <a href="{{ route('api-receivers.index') }}" class="btn btn-secondary btn-lg me-3">
                                        <i class="fas fa-times me-2"></i>Cancel
                                    </a>
                                    <button type="button" class="btn btn-warning btn-lg me-3" onclick="testReceiver()">
                                        <i class="fas fa-vial me-2"></i>Test Receiver
                                    </button>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Auto-generate endpoint from name
function generateEndpoint() {
    const name = document.getElementById('name');
    const endpoint = document.getElementById('endpoint');
    
    if (name && endpoint && name.value) {
        const generated = name.value
            .toLowerCase()
            .replace(/[^a-z0-9]/g, '-')
            .replace(/-+/g, '-')
            .replace(/^-|-$/g, '');
        endpoint.value = generated;
    }
}

// Generate authentication token
function generateToken() {
    const authToken = document.getElementById('auth_token');
    if (authToken) {
        const token = 'token_' + Math.random().toString(36).substr(2, 9) + '_' + Date.now();
        authToken.value = token;
    }
}

// Toggle authentication fields based on auth type
function toggleAuthFields() {
    const authType = document.getElementById('auth_type');
    const authTokenField = document.getElementById('auth_token_field');
    const basicAuthFields = document.getElementById('basic_auth_fields');
    
    if (!authType || !authTokenField || !basicAuthFields) return;
    
    // Hide all auth fields first
    authTokenField.classList.add('d-none');
    basicAuthFields.style.display = 'none';
    
    // Show relevant fields based on auth type
    if (authType.value === 'api_key' || authType.value === 'bearer_token') {
        authTokenField.classList.remove('d-none');
    } else if (authType.value === 'basic_auth') {
        basicAuthFields.style.display = 'block';
    }
}

// Toggle password visibility
function togglePassword() {
    const passwordField = document.getElementById('auth_password');
    const toggleIcon = document.getElementById('password_toggle_icon');
    
    if (!passwordField || !toggleIcon) return;
    
    if (passwordField.type === 'password') {
        passwordField.type = 'text';
        toggleIcon.classList.remove('fa-eye');
        toggleIcon.classList.add('fa-eye-slash');
    } else {
        passwordField.type = 'password';
        toggleIcon.classList.remove('fa-eye-slash');
        toggleIcon.classList.add('fa-eye');
    }
}

// Toggle bidirectional fields
function toggleBidirectionalFields() {
    const isBidirectional = document.getElementById('is_bidirectional');
    const bidirectionalSettings = document.getElementById('bidirectionalSettings');
    
    if (!isBidirectional || !bidirectionalSettings) return;
    
    if (isBidirectional.checked) {
        bidirectionalSettings.style.display = 'block';
    } else {
        bidirectionalSettings.style.display = 'none';
    }
}

// Toggle scheduling fields
function toggleSchedulingFields() {
    const enableScheduling = document.getElementById('enable_scheduling');
    const scheduleTypeField = document.getElementById('schedule_type_field');
    const schedulingDetails = document.getElementById('scheduling_details');
    
    if (!enableScheduling || !scheduleTypeField || !schedulingDetails) return;
    
    if (enableScheduling.checked) {
        scheduleTypeField.style.display = 'block';
        schedulingDetails.style.display = 'block';
        updateScheduleFields();
    } else {
        scheduleTypeField.style.display = 'none';
        schedulingDetails.style.display = 'none';
    }
}

// Update schedule fields based on schedule type
function updateScheduleFields() {
    const scheduleType = document.getElementById('schedule_type');
    const scheduleIntervalField = document.getElementById('schedule_interval_field');
    const scheduleTimeField = document.getElementById('schedule_time_field');
    
    if (!scheduleType || !scheduleIntervalField || !scheduleTimeField) return;
    
    // Hide all fields first
    scheduleIntervalField.style.display = 'none';
    scheduleTimeField.style.display = 'none';
    
    // Show relevant fields based on type
    if (scheduleType.value === 'interval') {
        scheduleIntervalField.style.display = 'block';
    } else if (scheduleType.value === 'cron') {
        scheduleTimeField.style.display = 'block';
    }
}

// Auto-sync auth from API Client
function syncAuthFromClient() {
    const apiClientId = document.getElementById('api_client_id');
    const authType = document.getElementById('auth_type');
    const authToken = document.getElementById('auth_token');
    const authUsername = document.getElementById('auth_username');
    const authPassword = document.getElementById('auth_password');
    
    console.log('syncAuthFromClient called');
    console.log('apiClientId:', apiClientId);
    console.log('apiClientId.value:', apiClientId?.value);
    
    if (!apiClientId || !apiClientId.value) {
        console.log('No API Client selected');
        return;
    }
    
    const url = `/api-clients/${apiClientId.value}/credentials`;
    console.log('Fetching from URL:', url);
    
    // Fetch client data via AJAX
    fetch(url, {
        method: 'GET',
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
        }
    })
        .then(response => {
            console.log('Response status:', response.status);
            console.log('Response headers:', response.headers);
            return response.json();
        })
        .then(data => {
            console.log('Response data:', data);
            if (data.success && data.client) {
                const client = data.client;
                console.log('Client data:', client);
                
                // Auto-fill based on auth type
                if (authType.value === 'basic_auth') {
                    if (authUsername) authUsername.value = client.usertoken || '';
                    if (authPassword) authPassword.value = client.passtoken || '';
                    console.log('Filled basic auth fields');
                } else if (authType.value === 'bearer_token') {
                    if (authToken) authToken.value = client.client_secret || '';
                    console.log('Filled bearer token field');
                } else if (authType.value === 'api_key') {
                    if (authToken) authToken.value = client.client_id || '';
                    console.log('Filled API key field');
                }
                
                // Show success message
                if (typeof showNotification === 'function') {
                    showNotification('Auth credentials auto-filled from API Client', 'success');
                } else {
                    console.log('Auth credentials auto-filled from API Client');
                }
            } else {
                console.log('No client data in response');
            }
        })
        .catch(error => {
            console.error('Error fetching client credentials:', error);
        });
}


// Generate sample validation rules
function generateSampleValidationRules() {
    const validationRules = document.getElementById('validation_rules');
    if (validationRules) {
        const sampleRules = {
            "id": {
                "required": true,
                "type": "integer"
            },
            "name": {
                "required": false,
                "type": "string",
                "max_length": 100
            },
            "status": {
                "required": false,
                "type": "string"
            },
            "created_at": {
                "required": false,
                "type": "date"
            }
        };
        
        validationRules.value = JSON.stringify(sampleRules, null, 2);
    }
}

// Generate sample data
function generateSampleData() {
    const sampleData = document.getElementById('sample_data');
    if (sampleData) {
        const sample = {
            "id": 1,
            "name": "John Doe",
            "email": "john@example.com",
            "phone": "+1234567890",
            "status": "active",
            "created_at": "2024-01-15T10:30:00Z",
            "metadata": {
                "source": "web_form",
                "campaign": "summer_2024"
            }
        };
        
        sampleData.value = JSON.stringify(sample, null, 2);
    }
}

// Generate field mapping
function generateFieldMapping() {
    const fieldMapping = document.getElementById('field_mapping');
    if (fieldMapping) {
        const mapping = {
            "user_name": "name",
            "user_email": "email",
            "user_phone": "phone",
            "user_status": "status",
            "created_date": "created_at",
            "source_system": "source"
        };
        
        fieldMapping.value = JSON.stringify(mapping, null, 2);
    }
}

// Test receiver function
function testReceiver() {
    const endpoint = document.getElementById('endpoint');
    if (endpoint && endpoint.value) {
        const testUrl = `${window.location.origin}/api/in/${endpoint.value}`;
        window.open(testUrl, '_blank');
    } else {
        alert('Please enter an endpoint first.');
    }
}

// Add event listeners when page loads
document.addEventListener('DOMContentLoaded', function() {
    try {
        // Add change listeners with null checks
        const allowedMethodsEl = document.getElementById('allowed_methods');
        const authTypeEl = document.getElementById('auth_type');
        const apiClientIdEl = document.getElementById('api_client_id');
        const formEl = document.querySelector('form');
        const altSubmitBtn = document.getElementById('altSubmitBtn');
        const simpleSubmitBtn = document.getElementById('simpleSubmitBtn');
        
        if (allowedMethodsEl) {
            allowedMethodsEl.addEventListener('change', function() {
                updateFormStatus();
            });
        }
        
        if (authTypeEl) {
            authTypeEl.addEventListener('change', function() {
                toggleAuthFields();
                updateFormStatus();
            });
        }
        
        if (apiClientIdEl) {
            apiClientIdEl.addEventListener('change', function() {
                updateFormStatus();
            });
        }
        
        // Add event listener for schedule type
        const scheduleTypeEl = document.getElementById('schedule_type');
        if (scheduleTypeEl) {
            scheduleTypeEl.addEventListener('change', function() {
                updateScheduleFields();
            });
        }
        
        if (formEl) {
            formEl.addEventListener('submit', function(e) {
                // No validation for GET-only specific fields
            });
        }
        
        if (altSubmitBtn) {
            altSubmitBtn.addEventListener('click', function() {
                const form = document.getElementById('receiverForm');
                if (form) form.submit();
            });
        }
        
        if (simpleSubmitBtn) {
            simpleSubmitBtn.addEventListener('click', function() {
                const form = document.getElementById('receiverForm');
                if (form) form.submit();
            });
        }
        
        // Add input event listeners for real-time validation
        const requiredInputs = ['name', 'endpoint', 'target_table', 'allowed_methods'];
        requiredInputs.forEach(inputId => {
            const input = document.getElementById(inputId);
            if (input) {
                input.addEventListener('input', updateFormStatus);
                input.addEventListener('blur', updateFormStatus);
            }
        });
        
        // Initialize
        toggleAuthFields();
        toggleBidirectionalFields();
        toggleSchedulingFields();
        updateFormStatus();
        
    } catch (error) {
        console.error('Error initializing form:', error);
    }
});

// Update form status
function updateFormStatus() {
    const statusContainer = document.getElementById('form-status');
    if (!statusContainer) return;
    
    const name = document.getElementById('name')?.value || '';
    const endpoint = document.getElementById('endpoint')?.value || '';
    const targetTable = document.getElementById('target_table')?.value || '';
    const allowedMethods = document.getElementById('allowed_methods')?.value || '';
    
    const requiredFields = [name, endpoint, targetTable, allowedMethods];
    const filledFields = requiredFields.filter(field => field.trim() !== '').length;
    
    let statusHtml = '';
    
    // Form readiness
    if (filledFields === 4) {
        statusHtml += '<div class="d-flex align-items-center mb-2">';
        statusHtml += '<i class="fas fa-check-circle text-success me-2"></i>';
        statusHtml += '<span class="text-success fw-bold">Form is ready to submit</span>';
        statusHtml += '</div>';
    } else {
        statusHtml += '<div class="d-flex align-items-center mb-2">';
        statusHtml += '<i class="fas fa-circle text-muted me-2"></i>';
        statusHtml += '<span class="text-muted">Form is incomplete</span>';
        statusHtml += '</div>';
    }
    
    // Required fields count
    statusHtml += '<div class="d-flex align-items-center mb-2">';
    statusHtml += '<i class="fas fa-circle text-muted me-2"></i>';
    statusHtml += `<span class="text-muted">Required fields: ${filledFields}/4</span>`;
    statusHtml += '</div>';
    
    // Validation status
    if (filledFields === 4) {
        statusHtml += '<div class="d-flex align-items-center">';
        statusHtml += '<i class="fas fa-check-circle text-success me-2"></i>';
        statusHtml += '<span class="text-success">Validation: Ready</span>';
        statusHtml += '</div>';
    } else {
        statusHtml += '<div class="d-flex align-items-center">';
        statusHtml += '<i class="fas fa-exclamation-circle text-warning me-2"></i>';
        statusHtml += '<span class="text-warning">Validation: Pending</span>';
        statusHtml += '</div>';
    }
    
    statusContainer.innerHTML = statusHtml;
}
</script>

<style>
/* Enhanced Card Styling */
.card {
    border: none;
    box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    border-radius: 0.75rem;
    transition: all 0.3s ease;
}

.card:hover {
    box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    transform: translateY(-2px);
}

.card-header {
    border-radius: 0.75rem 0.75rem 0 0 !important;
    border-bottom: none;
}

/* Gradient Backgrounds */
.bg-gradient-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
}

.bg-gradient-success {
    background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
}

.bg-gradient-info {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
}

.bg-gradient-warning {
    background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
}

/* Form Styling */
.form-control, .form-select {
    border-radius: 0.5rem;
    border: 1px solid #e3e6f0;
    padding: 0.75rem 1rem;
    font-size: 0.95rem;
    transition: all 0.3s ease;
}

.form-control:focus, .form-select:focus {
    border-color: #667eea;
    box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
}

.form-control-lg, .form-select-lg {
    padding: 1rem 1.25rem;
    font-size: 1.1rem;
}

.form-label {
    font-weight: 600;
    color: #5a5c69;
    margin-bottom: 0.5rem;
}

.form-text {
    font-size: 0.875rem;
    color: #858796;
    margin-top: 0.25rem;
}

/* Button Styling */
.btn {
    border-radius: 0.5rem;
    font-weight: 600;
    padding: 0.75rem 1.5rem;
    transition: all 0.3s ease;
    border: none;
}

.btn-lg {
    padding: 1rem 2rem;
    font-size: 1.1rem;
}

.btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
}

.btn-success {
    background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
    color: white;
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.btn-info {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.btn-warning {
    background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
    color: white;
}

.btn-secondary {
    background: linear-gradient(135deg, #6c757d 0%, #495057 100%);
    color: white;
}

.btn-outline-secondary {
    border: 2px solid #6c757d;
    color: #6c757d;
    background: transparent;
}

.btn-outline-secondary:hover {
    background: #6c757d;
    color: white;
}

/* Badge Styling */
.badge {
    font-size: 0.75rem;
    padding: 0.5rem 0.75rem;
    border-radius: 0.375rem;
    font-weight: 600;
}

.fs-6 {
    font-size: 0.875rem !important;
}

/* Section Headers */
.section-header h5 {
    color: #667eea;
    font-weight: 700;
    border-bottom: 2px solid #e3e6f0;
    padding-bottom: 0.5rem;
}

/* Form Switch Styling */
.form-check-input {
    width: 3rem;
    height: 1.5rem;
    border-radius: 1rem;
    border: 2px solid #e3e6f0;
    background-color: #f8f9fc;
    transition: all 0.3s ease;
}

.form-check-input:checked {
    background-color: #667eea;
    border-color: #667eea;
}

.form-check-input:focus {
    box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
}

/* Input Group Styling */
.input-group-text {
    background-color: #f8f9fc;
    border: 1px solid #e3e6f0;
    color: #6c757d;
    font-weight: 600;
}

/* Code Styling */
code {
    background-color: #f8f9fc;
    color: #e83e8c;
    padding: 0.25rem 0.5rem;
    border-radius: 0.25rem;
    font-size: 0.875rem;
}

/* Alert Styling */
.alert {
    border: none;
    border-radius: 0.75rem;
    padding: 1rem 1.25rem;
}

.alert-danger {
    background: linear-gradient(135deg, #ff6b6b 0%, #ee5a52 100%);
    color: white;
}

.alert-success {
    background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
    color: white;
}


/* Status Indicators */
.text-muted {
    color: #858796 !important;
}

/* Responsive Design */
@media (max-width: 768px) {
    .btn-lg {
        padding: 0.75rem 1.5rem;
        font-size: 1rem;
    }
    
    .card-body {
        padding: 1.5rem !important;
    }
    
    .form-control-lg, .form-select-lg {
        padding: 0.75rem 1rem;
        font-size: 1rem;
    }
}

/* Animation */
@keyframes fadeIn {
    from { opacity: 0; transform: translateY(20px); }
    to { opacity: 1; transform: translateY(0); }
}

.card {
    animation: fadeIn 0.5s ease-out;
}

/* Hover Effects */
.form-group:hover .form-label {
    color: #667eea;
}

.input-group:hover .input-group-text {
    background-color: #e3e6f0;
}

/* Custom Scrollbar */
::-webkit-scrollbar {
    width: 8px;
}

::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 4px;
}

::-webkit-scrollbar-thumb {
    background: #667eea;
    border-radius: 4px;
}

::-webkit-scrollbar-thumb:hover {
    background: #5a6fd8;
}
</style>
@endsection