@extends('layouts.app')

@section('title', 'POST-Only API Receivers')

@section('content')
<div class="container-fluid px-4">
    <!-- Page Header -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <div>
            <h1 class="h3 mb-0 text-gray-800">
                <i class="fas fa-upload me-2 text-success"></i>POST-Only API Receivers
            </h1>
            <p class="text-muted mt-2">Manage API endpoints that only accept POST requests for data ingestion</p>
        </div>
        <a href="{{ route('post-only-receivers.create') }}" class="btn btn-success">
            <i class="fas fa-plus me-2"></i>Create POST-Only Receiver
        </a>
    </div>

    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-start border-success border-4 shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col me-2">
                            <div class="text-xs fw-bold text-success text-uppercase mb-1">
                                POST-Only Receivers
                            </div>
                            <div class="h5 mb-0 fw-bold text-gray-800">
                                {{ $stats['total_post_receivers'] }}
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-upload fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-start border-primary border-4 shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col me-2">
                            <div class="text-xs fw-bold text-primary text-uppercase mb-1">
                                Active Receivers
                            </div>
                            <div class="h5 mb-0 fw-bold text-gray-800">
                                {{ $stats['active_post_receivers'] }}
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-check-circle fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-start border-info border-4 shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col me-2">
                            <div class="text-xs fw-bold text-info text-uppercase mb-1">
                                Total Requests
                            </div>
                            <div class="h5 mb-0 fw-bold text-gray-800">
                                {{ number_format($stats['total_requests']) }}
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-chart-line fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-start border-warning border-4 shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col me-2">
                            <div class="text-xs fw-bold text-warning text-uppercase mb-1">
                                Recent Activity
                            </div>
                            <div class="h5 mb-0 fw-bold text-gray-800">
                                {{ $stats['recent_requests'] }}
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-clock fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Success Message -->
    @if(session('success'))
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle me-2"></i>{{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    <!-- Error Message -->
    @if(session('error'))
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-circle me-2"></i>{{ session('error') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    <!-- Receivers Table -->
    <div class="card shadow-sm border-0">
        <div class="card-header bg-gradient-success text-white py-3">
            <h6 class="m-0 font-weight-bold">
                <i class="fas fa-list me-2"></i>POST-Only Receivers List
            </h6>
        </div>
        <div class="card-body p-0">
            @if($api_receivers->count() > 0)
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th class="px-4 py-3">
                                    <i class="fas fa-tag me-2"></i>Name
                                </th>
                                <th class="px-4 py-3">
                                    <i class="fas fa-link me-2"></i>Endpoint
                                </th>
                                <th class="px-4 py-3">
                                    <i class="fas fa-database me-2"></i>Target Table
                                </th>
                                <th class="px-4 py-3">
                                    <i class="fas fa-shield-alt me-2"></i>Auth
                                </th>
                                <th class="px-4 py-3">
                                    <i class="fas fa-chart-bar me-2"></i>Stats
                                </th>
                                <th class="px-4 py-3">
                                    <i class="fas fa-toggle-on me-2"></i>Status
                                </th>
                                <th class="px-4 py-3">
                                    <i class="fas fa-cogs me-2"></i>Actions
                                </th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($api_receivers as $receiver)
                                <tr>
                                    <td class="px-4 py-3">
                                        <div class="d-flex align-items-center">
                                            <div class="avatar-sm bg-success bg-opacity-10 rounded-circle d-flex align-items-center justify-content-center me-3">
                                                <i class="fas fa-upload text-success"></i>
                                            </div>
                                            <div>
                                                <h6 class="mb-0 fw-bold">{{ $receiver->name }}</h6>
                                                @if($receiver->description)
                                                    <small class="text-muted">{{ Str::limit($receiver->description, 50) }}</small>
                                                @endif
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-4 py-3">
                                        <div class="d-flex align-items-center">
                                            <code class="bg-light px-2 py-1 rounded">{{ $receiver->endpoint }}</code>
                                            <button class="btn btn-sm btn-outline-secondary ms-2" onclick="copyToClipboard('{{ url('/api/in/' . $receiver->endpoint) }}')" title="Copy URL">
                                                <i class="fas fa-copy"></i>
                                            </button>
                                        </div>
                                    </td>
                                    <td class="px-4 py-3">
                                        <span class="badge bg-info">{{ $receiver->target_table }}</span>
                                    </td>
                                    <td class="px-4 py-3">
                                        @if($receiver->auth_type === 'none')
                                            <span class="badge bg-warning">None</span>
                                        @elseif($receiver->auth_type === 'api_key')
                                            <span class="badge bg-primary">API Key</span>
                                        @elseif($receiver->auth_type === 'bearer_token')
                                            <span class="badge bg-success">Bearer</span>
                                        @else
                                            <span class="badge bg-secondary">{{ $receiver->auth_type }}</span>
                                        @endif
                                    </td>
                                    <td class="px-4 py-3">
                                        <div class="d-flex flex-column">
                                            <small class="text-muted">Received: {{ number_format($receiver->total_received) }}</small>
                                            @if($receiver->is_bidirectional)
                                                <small class="text-muted">Callbacks: {{ number_format($receiver->total_callbacks_sent) }}</small>
                                            @endif
                                        </div>
                                    </td>
                                    <td class="px-4 py-3">
                                        @if($receiver->is_active)
                                            <span class="badge bg-success">
                                                <i class="fas fa-check-circle me-1"></i>Active
                                            </span>
                                        @else
                                            <span class="badge bg-secondary">
                                                <i class="fas fa-pause-circle me-1"></i>Inactive
                                            </span>
                                        @endif
                                    </td>
                                    <td class="px-4 py-3">
                                        <div class="btn-group" role="group">
                                            <a href="{{ route('api-receivers.show', $receiver) }}" class="btn btn-sm btn-outline-primary" title="View Details">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                            <a href="{{ route('api-receivers.edit', $receiver) }}" class="btn btn-sm btn-outline-warning" title="Edit">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                            <a href="{{ route('api-receivers.logs', $receiver) }}" class="btn btn-sm btn-outline-info" title="View Logs">
                                                <i class="fas fa-list-alt"></i>
                                            </a>
                                            <button type="button" class="btn btn-sm btn-outline-success" onclick="testReceiver('{{ $receiver->endpoint }}')" title="Test Receiver">
                                                <i class="fas fa-vial"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            @else
                <div class="text-center py-5">
                    <div class="mb-4">
                        <i class="fas fa-upload fa-4x text-muted"></i>
                    </div>
                    <h5 class="text-muted mb-3">No POST-Only Receivers Found</h5>
                    <p class="text-muted mb-4">Create your first POST-Only receiver to start accepting data via POST requests.</p>
                    <a href="{{ route('post-only-receivers.create') }}" class="btn btn-success btn-lg">
                        <i class="fas fa-plus me-2"></i>Create First Receiver
                    </a>
                </div>
            @endif
        </div>
        
        @if($api_receivers->count() > 0)
            <div class="card-footer bg-light">
                <div class="d-flex justify-content-between align-items-center">
                    <div class="text-muted">
                        Showing {{ $api_receivers->firstItem() }} to {{ $api_receivers->lastItem() }} of {{ $api_receivers->total() }} receivers
                    </div>
                    <div>
                        {{ $api_receivers->links() }}
                    </div>
                </div>
            </div>
        @endif
    </div>
</div>

<script>
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(function() {
        // Show success message
        const toast = document.createElement('div');
        toast.className = 'position-fixed top-0 end-0 p-3';
        toast.style.zIndex = '9999';
        toast.innerHTML = `
            <div class="toast show" role="alert">
                <div class="toast-header bg-success text-white">
                    <i class="fas fa-check-circle me-2"></i>
                    <strong class="me-auto">Copied!</strong>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="toast"></button>
                </div>
                <div class="toast-body">
                    Endpoint URL copied to clipboard
                </div>
            </div>
        `;
        document.body.appendChild(toast);
        
        setTimeout(() => {
            toast.remove();
        }, 3000);
    });
}

function testReceiver(endpoint) {
    const url = `${window.location.origin}/api/in/${endpoint}`;
    const testData = {
        test: true,
        timestamp: new Date().toISOString(),
        message: 'Test request from POST-Only Receivers page'
    };
    
    fetch(url, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(testData)
    })
    .then(response => response.json())
    .then(data => {
        alert(`Test successful!\nResponse: ${JSON.stringify(data, null, 2)}`);
    })
    .catch(error => {
        alert(`Test failed: ${error.message}`);
    });
}
</script>
@endsection 