@extends('layouts.app')

@section('title', 'API Receiver Details')

@section('breadcrumb')
    <li class="breadcrumb-item"><a href="{{ route('api-receivers.index') }}">API Receivers</a></li>
    <li class="breadcrumb-item active">{{ $apiReceiver->name }}</li>
@endsection

@section('content')
<div class="container-fluid">
    <!-- Page Heading -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h3 mb-0 text-gray-800">
            <i class="fas fa-inbox me-2"></i>{{ $apiReceiver->name }}
        </h1>
        <div class="btn-group">
            <a href="{{ route('api-receivers.edit', $apiReceiver) }}" class="btn btn-primary">
                <i class="fas fa-edit me-2"></i>Edit
            </a>
            <a href="{{ route('api-receivers.index') }}" class="btn btn-secondary">
                <i class="fas fa-arrow-left me-2"></i>Back to List
            </a>
        </div>
    </div>

    <!-- Status Alert -->
    @if(!$apiReceiver->is_active)
        <div class="alert alert-warning" role="alert">
            <i class="fas fa-exclamation-triangle me-2"></i>
            This API Receiver is currently inactive and will not receive data.
        </div>
    @endif

    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-success shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                Total Received
                            </div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                {{ number_format($apiReceiver->total_received ?? 0) }}
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-download fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-info shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">
                                Success Rate
                            </div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                @php
                                    $totalLogs = $apiReceiver->receiveLogs()->count();
                                    $successLogs = $apiReceiver->receiveLogs()->where('status_code', '>=', 200)->where('status_code', '<', 300)->count();
                                    $successRate = $totalLogs > 0 ? round(($successLogs / $totalLogs) * 100, 1) : 0;
                                @endphp
                                {{ $successRate }}%
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-chart-line fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-warning shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">
                                Today's Received
                            </div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                {{ $apiReceiver->receiveLogs()->whereDate('created_at', today())->count() }}
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-calendar-day fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-primary shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">
                                Last Received
                            </div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                @if($apiReceiver->last_received_at)
                                    @php
                                        $lastReceived = $apiReceiver->last_received_at;
                                        if (is_string($lastReceived)) {
                                            $lastReceived = \Carbon\Carbon::parse($lastReceived);
                                        }
                                    @endphp
                                    {{ $lastReceived->diffForHumans() }}
                                @else
                                    Never
                                @endif
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-clock fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <!-- Receiver Details -->
        <div class="col-lg-8">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-info-circle me-2"></i>Receiver Details
                    </h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="font-weight-bold text-gray-700">Name:</label>
                                <div class="bg-light p-2 rounded">{{ $apiReceiver->name }}</div>
                            </div>
                            <div class="mb-3">
                                <label class="font-weight-bold text-gray-700">Endpoint URL:</label>
                                <div>
                                    <code class="bg-light p-2 rounded d-block">{{ url('/api/in/' . $apiReceiver->endpoint) }}</code>
                                </div>
                            </div>
                            <div class="mb-3">
                                <label class="font-weight-bold text-gray-700">Target Table:</label>
                                <div><code class="bg-light p-2 rounded">{{ $apiReceiver->target_table }}</code></div>
                            </div>
                            <div class="mb-3">
                                <label class="font-weight-bold text-gray-700">Allowed Methods:</label>
                                <div>
                                    @php
                                        $methods = is_array($apiReceiver->allowed_methods) ? $apiReceiver->allowed_methods : explode(',', $apiReceiver->allowed_methods);
                                    @endphp
                                    @foreach($methods as $method)
                                        <span class="badge bg-{{ $method == 'GET' ? 'success' : ($method == 'POST' ? 'primary' : 'warning') }} me-1">
                                            {{ trim($method) }}
                                        </span>
                                    @endforeach
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="font-weight-bold text-gray-700">Status:</label>
                                <div>
                                    <span class="badge bg-{{ $apiReceiver->is_active ? 'success' : 'secondary' }}">
                                        {{ $apiReceiver->is_active ? 'Active' : 'Inactive' }}
                                    </span>
                                </div>
                            </div>
                            <div class="mb-3">
                                <label class="font-weight-bold text-gray-700">Authentication:</label>
                                <div>
                                    <span class="badge bg-{{ $apiReceiver->auth_type == 'none' ? 'warning' : 'success' }}">
                                        {{ ucfirst(str_replace('_', ' ', $apiReceiver->auth_type)) }}
                                    </span>
                                </div>
                            </div>
                            @if($apiReceiver->rate_limit)
                                <div class="mb-3">
                                    <label class="font-weight-bold text-gray-700">Rate Limit:</label>
                                    <div class="bg-light p-2 rounded">{{ number_format($apiReceiver->rate_limit) }} requests/hour</div>
                                </div>
                            @endif
                            <div class="mb-3">
                                <label class="font-weight-bold text-gray-700">Auto Create Table:</label>
                                <div>
                                    <span class="badge bg-{{ $apiReceiver->auto_create_table ? 'success' : 'secondary' }}">
                                        {{ $apiReceiver->auto_create_table ? 'Yes' : 'No' }}
                                    </span>
                                </div>
                            </div>
                        </div>
                    </div>

                    @if($apiReceiver->description)
                        <div class="mt-3">
                            <label class="font-weight-bold text-gray-700">Description:</label>
                            <div class="bg-light p-2 rounded">{{ $apiReceiver->description }}</div>
                        </div>
                    @endif

                    @if($apiReceiver->field_mapping)
                        <div class="mt-3">
                            <label class="font-weight-bold text-gray-700">Field Mapping:</label>
                            <div class="bg-light p-3 rounded">
                                <pre class="mb-0">{{ json_encode($apiReceiver->field_mapping, JSON_PRETTY_PRINT) }}</pre>
                            </div>
                        </div>
                    @endif

                    @if($apiReceiver->isGetOnly())
                        <!-- GET-Only Receiver Information -->
                        <div class="card border-info mb-4">
                            <div class="card-header bg-info text-white">
                                <h5 class="mb-0"><i class="fas fa-download me-2"></i>GET-Only Receiver</h5>
                            </div>
                            <div class="card-body">
                                <div class="alert alert-info">
                                    <i class="fas fa-info-circle me-2"></i>
                                    This receiver only accepts GET requests and returns data from the target table.
                                </div>

                                <!-- Usage Examples -->
                                <h6><i class="fas fa-code me-2"></i>Usage Examples</h6>
                                
                                <div class="mb-3">
                                    <label class="form-label">Basic GET Request:</label>
                                    <div class="input-group">
                                        <input type="text" class="form-control" readonly 
                                               value="{{ url('/api/in/' . $apiReceiver->endpoint) }}">
                                        <button class="btn btn-outline-secondary" type="button" 
                                                onclick="copyToClipboard('{{ url('/api/in/' . $apiReceiver->endpoint) }}')">
                                            <i class="fas fa-copy"></i>
                                        </button>
                                    </div>
                                </div>

                                <div class="mb-3">
                                    <label class="form-label">With Query Parameters:</label>
                                    <div class="input-group">
                                        <input type="text" class="form-control" readonly 
                                               value="{{ url('/api/in/' . $apiReceiver->endpoint) }}?id=123&status=active&limit=50">
                                        <button class="btn btn-outline-secondary" type="button" 
                                                onclick="copyToClipboard('{{ url('/api/in/' . $apiReceiver->endpoint) }}?id=123&status=active&limit=50')">
                                            <i class="fas fa-copy"></i>
                                        </button>
                                    </div>
                                </div>

                                @if($apiReceiver->auth_type !== 'none')
                                    <div class="mb-3">
                                        <label class="form-label">With Authentication:</label>
                                        @if($apiReceiver->auth_type === 'api_key')
                                            <div class="input-group">
                                                <input type="text" class="form-control" readonly 
                                                       value="{{ url('/api/in/' . $apiReceiver->endpoint) }}?api_key=YOUR_API_KEY">
                                                <button class="btn btn-outline-secondary" type="button" 
                                                        onclick="copyToClipboard('{{ url('/api/in/' . $apiReceiver->endpoint) }}?api_key=YOUR_API_KEY')">
                                                    <i class="fas fa-copy"></i>
                                                </button>
                                            </div>
                                            <small class="text-muted">Or use header: X-API-Key: YOUR_API_KEY</small>
                                        @elseif($apiReceiver->auth_type === 'bearer_token')
                                            <div class="input-group">
                                                <input type="text" class="form-control" readonly 
                                                       value="{{ url('/api/in/' . $apiReceiver->endpoint) }}?token=YOUR_TOKEN">
                                                <button class="btn btn-outline-secondary" type="button" 
                                                        onclick="copyToClipboard('{{ url('/api/in/' . $apiReceiver->endpoint) }}?token=YOUR_TOKEN')">
                                                    <i class="fas fa-copy"></i>
                                                </button>
                                            </div>
                                            <small class="text-muted">Or use header: Authorization: Bearer YOUR_TOKEN</small>
                                        @endif
                                    </div>
                                @endif

                                <!-- Validation Rules -->
                                @if($apiReceiver->validation_rules)
                                    <div class="mb-3">
                                        <label class="form-label">Parameter Validation Rules:</label>
                                        <pre class="bg-light p-3 rounded"><code>{{ json_encode($apiReceiver->getValidationRules(), JSON_PRETTY_PRINT) }}</code></pre>
                                    </div>
                                @endif

                                <!-- Response Format -->
                                <div class="mb-3">
                                    <label class="form-label">Response Format:</label>
                                    <span class="badge bg-primary">{{ strtoupper($apiReceiver->response_format ?? 'JSON') }}</span>
                                </div>

                                <!-- Max Records -->
                                <div class="mb-3">
                                    <label class="form-label">Max Records Per Request:</label>
                                    <span class="badge bg-secondary">{{ $apiReceiver->max_records_per_request ?? 10 }}</span>
                                </div>

                                <!-- Client Synchronization -->
                                @if($apiReceiver->requiresClientSync())
                                    <div class="alert alert-warning">
                                        <i class="fas fa-sync me-2"></i>
                                        <strong>Client Token Synchronization Enabled</strong><br>
                                        This receiver validates tokens against API Client: 
                                        <strong>{{ $apiReceiver->apiClient->name ?? 'Unknown' }}</strong>
                                    </div>
                                @endif

                                <!-- Test Button -->
                                <div class="mt-3">
                                    <button class="btn btn-success" onclick="testGetReceiver()">
                                        <i class="fas fa-play me-2"></i>Test GET Request
                                    </button>
                                </div>
                            </div>
                        </div>
                    @endif
                </div>
            </div>
        </div>

        <!-- Quick Actions & Recent Activity -->
        <div class="col-lg-4">
            <!-- Quick Actions -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-bolt me-2"></i>Quick Actions
                    </h6>
                </div>
                <div class="card-body">
                    <div class="d-grid gap-2">
                        <button type="button" class="btn btn-success btn-sm" onclick="testReceiver()">
                            <i class="fas fa-vial me-2"></i>Test Receiver
                        </button>
                        <a href="{{ route('api-receivers.logs', $apiReceiver) }}" class="btn btn-info btn-sm">
                            <i class="fas fa-list me-2"></i>View All Logs
                        </a>
                        <button type="button" class="btn btn-warning btn-sm" onclick="copyEndpointUrl()">
                            <i class="fas fa-copy me-2"></i>Copy Endpoint URL
                        </button>
                        @if($apiReceiver->is_active)
                            <button type="button" class="btn btn-secondary btn-sm" onclick="toggleStatus()">
                                <i class="fas fa-pause me-2"></i>Deactivate
                            </button>
                        @else
                            <button type="button" class="btn btn-success btn-sm" onclick="toggleStatus()">
                                <i class="fas fa-play me-2"></i>Activate
                            </button>
                        @endif
                    </div>
                </div>
            </div>

            <!-- Recent Activity -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-clock me-2"></i>Recent Activity
                    </h6>
                </div>
                <div class="card-body">
                    @php
                        $recentLogs = $apiReceiver->receiveLogs()->latest('created_at')->limit(5)->get();
                    @endphp
                    @if($recentLogs->count() > 0)
                        @foreach($recentLogs as $log)
                            <div class="d-flex align-items-center mb-3">
                                <div class="me-3">
                                    <div class="icon-circle bg-{{ $log->status_code >= 200 && $log->status_code < 300 ? 'success' : 'danger' }}">
                                        <i class="fas fa-{{ $log->status_code >= 200 && $log->status_code < 300 ? 'check' : 'times' }} text-white"></i>
                                    </div>
                                </div>
                                <div class="flex-grow-1">
                                    <div class="small">
                                        <span class="badge bg-{{ $log->status_code >= 200 && $log->status_code < 300 ? 'success' : 'danger' }}">
                                            {{ $log->status_code >= 200 && $log->status_code < 300 ? 'Success' : 'Error' }}
                                        </span>
                                        from {{ $log->ip_address }}
                                    </div>
                                    <div class="text-xs text-gray-500">
                                        {{ $log->created_at->diffForHumans() }}
                                    </div>
                                </div>
                            </div>
                        @endforeach
                        <div class="text-center">
                            <a href="{{ route('api-receivers.logs', $apiReceiver) }}" class="btn btn-sm btn-outline-primary">
                                View All Logs
                            </a>
                        </div>
                    @else
                        <div class="text-center text-gray-500">
                            <i class="fas fa-inbox fa-2x mb-2"></i>
                            <div>No activity yet</div>
                            <div class="small">Data received will appear here</div>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1" role="dialog" aria-labelledby="deleteModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteModalLabel">Confirm Delete</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                Are you sure you want to delete the API Receiver "<strong>{{ $apiReceiver->name }}</strong>"?
                <br><br>
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    This action cannot be undone and will also delete all associated logs.
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <form action="{{ route('api-receivers.destroy', $apiReceiver) }}" method="POST" style="display: inline;">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger">Delete</button>
                </form>
            </div>
        </div>
    </div>
</div>

@endsection

@section('scripts')
<script>
function testReceiver() {
    // Test the receiver endpoint
    const endpointUrl = '{{ addslashes(url("/api/in/" . $apiReceiver->endpoint)) }}';
    const testData = {
        test: true,
        timestamp: new Date().toISOString(),
        message: 'Test data from dashboard'
    };

    // Build headers dynamically
    const headers = {
        'Content-Type': 'application/json'
    };
    
    @if($apiReceiver->auth_type == 'api_key')
        headers['X-API-Key'] = '{{ addslashes($apiReceiver->auth_token) }}';
    @elseif($apiReceiver->auth_type == 'bearer_token')
        headers['Authorization'] = 'Bearer {{ addslashes($apiReceiver->auth_token) }}';
    @endif

    fetch(endpointUrl, {
        method: 'POST',
        headers: headers,
        body: JSON.stringify(testData)
    })
    .then(response => response.json())
    .then(function(response) {
        return response.json();
    })
    .then(function(data) {
        if (data.success) {
            Swal.fire('Success!', 'Test data sent successfully', 'success');
            setTimeout(function() {
                location.reload();
            }, 1500);
        } else {
            Swal.fire('Error!', data.error || 'Test failed', 'error');
        }
    })
    .catch(function(error) {
        Swal.fire('Error!', 'Network error: ' + error.message, 'error');
    });
}

function copyEndpointUrl() {
    const url = '{{ addslashes(url("/api/in/" . $apiReceiver->endpoint)) }}';
    navigator.clipboard.writeText(url).then(function() {
        Swal.fire('Copied!', 'Endpoint URL copied to clipboard', 'success');
    }).catch(function(error) {
        // Fallback for older browsers
        const textArea = document.createElement('textarea');
        textArea.value = url;
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
        Swal.fire('Copied!', 'Endpoint URL copied to clipboard', 'success');
    });
}

function toggleStatus() {
    const currentStatus = {{ $apiReceiver->is_active ? 'true' : 'false' }};
    const newStatus = !currentStatus;
    const action = newStatus ? 'activate' : 'deactivate';
    const actionCapitalized = action.charAt(0).toUpperCase() + action.slice(1);
    
    Swal.fire({
        title: actionCapitalized + ' Receiver?',
        text: 'Are you sure you want to ' + action + ' this receiver?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonText: 'Yes, ' + action + '!'
    }).then(function(result) {
        if (result.isConfirmed) {
            // Update status via AJAX (you might need to implement this endpoint)
            Swal.fire('Updated!', 'Receiver has been ' + action + 'd.', 'success');
        }
    });
}

function testGetReceiver() {
    // Test the receiver endpoint
    const endpointUrl = '{{ addslashes(url("/api/in/" . $apiReceiver->endpoint)) }}';
    const testData = {
        test: true,
        timestamp: new Date().toISOString(),
        message: 'Test data from dashboard'
    };

    // Build headers dynamically
    const headers = {
        'Content-Type': 'application/json'
    };
    
    @if($apiReceiver->auth_type == 'api_key')
        headers['X-API-Key'] = '{{ addslashes($apiReceiver->auth_token) }}';
    @elseif($apiReceiver->auth_type == 'bearer_token')
        headers['Authorization'] = 'Bearer {{ addslashes($apiReceiver->auth_token) }}';
    @endif

    fetch(endpointUrl, {
        method: 'POST',
        headers: headers,
        body: JSON.stringify(testData)
    })
    .then(response => response.json())
    .then(function(response) {
        return response.json();
    })
    .then(function(data) {
        if (data.success) {
            Swal.fire('Success!', 'Test data sent successfully', 'success');
            setTimeout(function() {
                location.reload();
            }, 1500);
        } else {
            Swal.fire('Error!', data.error || 'Test failed', 'error');
        }
    })
    .catch(function(error) {
        Swal.fire('Error!', 'Network error: ' + error.message, 'error');
    });
}

function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(function() {
        Swal.fire('Copied!', 'Text copied to clipboard', 'success');
    }).catch(function(error) {
        // Fallback for older browsers
        const textArea = document.createElement('textarea');
        textArea.value = text;
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
        Swal.fire('Copied!', 'Text copied to clipboard', 'success');
    });
}
</script>
@endsection 