@extends('layouts.app')

@section('title', 'API Source Data')

@section('breadcrumb')
    <li class="breadcrumb-item">
        <a href="{{ route('api-sources.index') }}">API Sources</a>
    </li>
    <li class="breadcrumb-item">
        <a href="{{ route('api-sources.show', $apiSource) }}">{{ $apiSource->name }}</a>
    </li>
    <li class="breadcrumb-item active">Data</li>
@endsection

@section('styles')
<style>
.card {
    transition: transform 0.2s ease-in-out, box-shadow 0.2s ease-in-out;
}

.card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
}

.table-sm td, .table-sm th {
    padding: 0.5rem;
    vertical-align: middle;
}

.text-truncate {
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.badge {
    font-size: 0.75rem;
}

.btn-sm {
    font-size: 0.75rem;
    padding: 0.25rem 0.5rem;
}

.table-responsive {
    font-size: 0.875rem;
}

.table th {
    font-weight: 600;
    font-size: 0.875rem;
    white-space: nowrap;
    text-align: center;
    background-color: #343a40;
    color: white;
    border: 1px solid #495057;
}

.table td {
    vertical-align: middle;
    font-size: 0.875rem;
    border: 1px solid #dee2e6;
    padding: 0.75rem 0.5rem;
}

.table th:first-child,
.table th:nth-child(2) {
    width: 80px;
    min-width: 80px;
}

.table th:last-child {
    width: 100px;
    min-width: 100px;
}

.table td:first-child,
.table td:nth-child(2) {
    width: 80px;
    min-width: 80px;
    text-align: center;
}

.table td:last-child {
    width: 100px;
    min-width: 100px;
    text-align: center;
}

/* Dynamic column sizing */
.table th:not(:first-child):not(:nth-child(2)):not(:last-child) {
    min-width: 120px;
    max-width: 300px;
}

.table td:not(:first-child):not(:nth-child(2)):not(:last-child) {
    min-width: 120px;
    max-width: 300px;
    word-wrap: break-word;
    overflow-wrap: break-word;
}

/* Column specific styling */
.table th:nth-child(3),
.table td:nth-child(3) {
    min-width: 150px;
}

/* Responsive column adjustments */
@media (max-width: 1200px) {
    .table th:not(:first-child):not(:nth-child(2)):not(:last-child),
    .table td:not(:first-child):not(:nth-child(2)):not(:last-child) {
        min-width: 100px;
        max-width: 200px;
    }
}

@media (max-width: 768px) {
    .table th:not(:first-child):not(:nth-child(2)):not(:last-child),
    .table td:not(:first-child):not(:nth-child(2)):not(:last-child) {
        min-width: 80px;
        max-width: 150px;
    }
}

.table-success {
    background-color: rgba(40, 167, 69, 0.1) !important;
}

.table-primary {
    background-color: rgba(0, 123, 255, 0.1) !important;
}

.table-info {
    background-color: rgba(23, 162, 184, 0.1) !important;
}

.table-warning {
    background-color: rgba(255, 193, 7, 0.1) !important;
}

@media (max-width: 768px) {
    .table-responsive {
        font-size: 0.75rem;
    }
    
    .table th, .table td {
        font-size: 0.75rem;
        padding: 0.5rem 0.25rem;
    }
    
    .btn-sm {
        font-size: 0.7rem;
        padding: 0.2rem 0.4rem;
    }
}

.card-header {
    border-bottom: none;
}

.table-borderless td, .table-borderless th {
    border: none;
}

.border-bottom {
    border-bottom: 1px solid #dee2e6 !important;
}

.table-striped > tbody > tr:nth-of-type(odd) {
    background-color: rgba(0, 0, 0, 0.02);
}
</style>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h1 class="h3 mb-0 text-gray-800">
                    <i class="fas fa-table me-2"></i>{{ $apiSource->name }} - Imported Data
                </h1>
                <div class="btn-group">
                    <button type="button" class="btn btn-success" onclick="importData({{ $apiSource->id }}, '{{ $apiSource->name }}')">
                        <i class="fas fa-download me-2"></i>Import New Data
                    </button>
                    <button type="button" class="btn btn-primary" onclick="autoImportData({{ $apiSource->id }}, '{{ $apiSource->name }}')">
                        <i class="fas fa-robot me-2"></i>Auto Import
                    </button>
                    <button type="button" class="btn btn-info" onclick="showImportStats({{ $apiSource->id }})">
                        <i class="fas fa-chart-bar me-2"></i>Import Stats
                    </button>
                    <button type="button" class="btn btn-warning" onclick="viewAllJsonData()">
                        <i class="fas fa-code me-2"></i>View JSON
                    </button>
                    <a href="{{ route('api-sources.show', $apiSource) }}" class="btn btn-info">
                        <i class="fas fa-eye me-2"></i>View Details
                    </a>
                    <a href="{{ route('api-sources.index') }}" class="btn btn-secondary">
                        <i class="fas fa-arrow-left me-2"></i>Back to List
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Information Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-primary shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">
                                Table Name</div>
                            <div class="h6 mb-0 font-weight-bold text-gray-800">
                                <code>{{ $apiSource->table_name }}</code>
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-table fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-success shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                API URL</div>
                            <div class="small text-gray-800">{{ Str::limit($apiSource->url, 25) }}</div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-link fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-info shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">
                                Method</div>
                            <div class="h6 mb-0 font-weight-bold text-gray-800">{{ $apiSource->method }}</div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-cog fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-warning shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">
                                Last Fetched</div>
                            <div class="small text-gray-800">
                                @if($apiSource->last_fetched_at)
                                    {{ $apiSource->last_fetched_at->diffForHumans() }}
                                @else
                                    Never
                                @endif
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-clock fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- JSON Access Links -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="card shadow">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-code me-2"></i>JSON Data Access
                    </h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <h6>Direct JSON Endpoints:</h6>
                            <div class="mb-2">
                                <a href="{{ route('api-sources.data', $apiSource) }}?format=json" target="_blank" class="btn btn-sm btn-outline-primary">
                                    <i class="fas fa-external-link-alt me-1"></i>JSON Format
                                </a>
                                <small class="text-muted ms-2">Add ?format=json to get JSON response</small>
                            </div>
                            <div class="mb-2">
                                <a href="{{ route('api-sources.json', $apiSource) }}" target="_blank" class="btn btn-sm btn-outline-success">
                                    <i class="fas fa-external-link-alt me-1"></i>Direct JSON
                                </a>
                                <small class="text-muted ms-2">Direct JSON endpoint</small>
                            </div>
                            <div class="mb-2">
                                <a href="/api/out/sources/{{ $apiSource->id }}/data" target="_blank" class="btn btn-sm btn-outline-info">
                                    <i class="fas fa-external-link-alt me-1"></i>API Endpoint
                                </a>
                                <small class="text-muted ms-2">REST API endpoint</small>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <h6>Quick Actions:</h6>
                            <div class="mb-2">
                                <button type="button" class="btn btn-sm btn-warning" onclick="viewAllJsonData()">
                                    <i class="fas fa-eye me-1"></i>View JSON in Modal
                                </button>
                                <small class="text-muted ms-2">Open JSON in popup modal</small>
                            </div>
                            <div class="mb-2">
                                <button type="button" class="btn btn-sm btn-info" onclick="viewJsonAsTable()">
                                    <i class="fas fa-table me-1"></i>View as Table
                                </button>
                                <small class="text-muted ms-2">Convert JSON to table format</small>
                            </div>
                            <div class="mb-2">
                                <button type="button" class="btn btn-sm btn-secondary" onclick="showRawData()">
                                    <i class="fas fa-code me-1"></i>View Raw Data
                                </button>
                                <small class="text-muted ms-2">View all data as JSON</small>
                            </div>
                            <div class="mb-2">
                                <button type="button" class="btn btn-sm btn-success" onclick="refreshApiData()">
                                    <i class="fas fa-sync-alt me-1"></i>Refresh API Data
                                </button>
                                <small class="text-muted ms-2">Fetch fresh data from external API</small>
                            </div>
                            <div class="mb-2">
                                <button type="button" class="btn btn-sm btn-primary" onclick="viewApiDataAsTable()">
                                    <i class="fas fa-table me-1"></i>API Data as Table
                                </button>
                                <small class="text-muted ms-2">Convert external API data to table</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Combined Data Table Section -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="card shadow">
                <div class="card-header py-3 d-flex justify-content-between align-items-center bg-dark text-white">
                    <h6 class="m-0 font-weight-bold">
                        <i class="fas fa-table me-2"></i>Combined Data View
                    </h6>
                    <div>
                        @if($apiData !== null)
                            <span class="badge bg-success me-2">API: {{ is_array($apiData) ? count($apiData) : 1 }} records</span>
                        @endif
                        <span class="badge bg-primary me-2">Local: {{ count($records) }} records</span>
                        <button type="button" class="btn btn-sm btn-light ms-2" onclick="refreshApiData()">
                            <i class="fas fa-sync-alt me-1"></i>Refresh API
                        </button>
                        <button type="button" class="btn btn-sm btn-light ms-2" onclick="exportCombinedData()">
                            <i class="fas fa-download me-1"></i>Export
                        </button>
                        <button type="button" class="btn btn-sm btn-light ms-2" onclick="adjustColumnWidths()">
                            <i class="fas fa-columns me-1"></i>Auto Fit
                        </button>
                    </div>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-bordered table-striped mb-0">
                            <thead class="table-dark">
                                <tr>
                                    <th style="width: 80px;">Source</th>
                                    <th style="width: 80px;">Record #</th>
                                    @php
                                        $allColumns = [];
                                        
                                        // Collect API columns
                                        if($apiData !== null && is_array($apiData)) {
                                            if(isset($apiData[0]) && is_array($apiData[0])) {
                                                $allColumns = array_merge($allColumns, array_keys($apiData[0]));
                                            } else {
                                                $allColumns = array_merge($allColumns, array_keys($apiData));
                                            }
                                        }
                                        
                                        // Collect local columns
                                        if(count($columns) > 0) {
                                            $allColumns = array_merge($allColumns, $columns);
                                        }
                                        
                                        // Remove duplicates and maintain order
                                        $allColumns = array_unique($allColumns);
                                    @endphp
                                    
                                    @if(count($allColumns) > 0)
                                        @foreach($allColumns as $column)
                                            <th>{{ ucwords(str_replace('_', ' ', $column)) }}</th>
                                        @endforeach
                                    @else
                                        <th>Field</th>
                                        <th>Value</th>
                                    @endif
                                    <th style="width: 100px;">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @if($apiData !== null)
                                    @if(is_array($apiData))
                                        @if(isset($apiData[0]) && is_array($apiData[0]))
                                            <!-- Array of objects from API -->
                                            @foreach($apiData as $index => $item)
                                                <tr class="table-success">
                                                    <td>
                                                        <span class="badge bg-success">API</span>
                                                    </td>
                                                    <td class="text-center">
                                                        <strong>{{ $index + 1 }}</strong>
                                                    </td>
                                                    @foreach($allColumns as $column)
                                                        <td>
                                                            @if(isset($item[$column]))
                                                                @php $value = $item[$column]; @endphp
                                                                @if(is_array($value))
                                                                    <button type="button" class="btn btn-sm btn-outline-success" onclick="viewJsonData({{ $index }}, '{{ $column }}')">
                                                                        <i class="fas fa-eye me-1"></i>JSON
                                                                    </button>
                                                                @elseif(is_bool($value))
                                                                    <span class="badge bg-{{ $value ? 'success' : 'secondary' }}">
                                                                        {{ $value ? 'Yes' : 'No' }}
                                                                    </span>
                                                                @elseif(is_numeric($value))
                                                                    <span class="text-primary fw-bold">{{ number_format($value) }}</span>
                                                                @elseif(is_string($value) && strlen($value) > 50)
                                                                    <span title="{{ $value }}" class="text-truncate d-inline-block" style="max-width: 200px;">
                                                                        {{ Str::limit($value, 50) }}
                                                                    </span>
                                                                @else
                                                                    <span class="text-dark">{{ $value }}</span>
                                                                @endif
                                                            @else
                                                                <span class="text-muted">-</span>
                                                            @endif
                                                        </td>
                                                    @endforeach
                                                    <td class="text-center">
                                                        <button type="button" class="btn btn-sm btn-outline-success" onclick="viewApiRecord({{ $index }})">
                                                            <i class="fas fa-eye me-1"></i>View
                                                        </button>
                                                    </td>
                                                </tr>
                                            @endforeach
                                        @else
                                            <!-- Simple array from API -->
                                            <tr class="table-info">
                                                <td>
                                                    <span class="badge bg-info">API</span>
                                                </td>
                                                <td class="text-center">
                                                    <strong>1</strong>
                                                </td>
                                                @foreach($allColumns as $column)
                                                    <td>
                                                        @if(isset($apiData[$column]))
                                                            @php $value = $apiData[$column]; @endphp
                                                            @if(is_array($value))
                                                                <button type="button" class="btn btn-sm btn-outline-info" onclick="viewJsonData(0, '{{ $column }}')">
                                                                    <i class="fas fa-eye me-1"></i>JSON
                                                                </button>
                                                            @elseif(is_bool($value))
                                                                <span class="badge bg-{{ $value ? 'success' : 'secondary' }}">
                                                                    {{ $value ? 'Yes' : 'No' }}
                                                                </span>
                                                            @elseif(is_numeric($value))
                                                                <span class="text-primary fw-bold">{{ number_format($value) }}</span>
                                                            @elseif(is_string($value) && strlen($value) > 100)
                                                                <span title="{{ $value }}" class="text-truncate d-inline-block" style="max-width: 300px;">
                                                                    {{ Str::limit($value, 100) }}
                                                                </span>
                                                            @else
                                                                <span class="text-dark">{{ $value }}</span>
                                                            @endif
                                                        @else
                                                            <span class="text-muted">-</span>
                                                        @endif
                                                    </td>
                                                @endforeach
                                                <td class="text-center">
                                                    <button type="button" class="btn btn-sm btn-outline-info" onclick="viewApiRecord(0)">
                                                        <i class="fas fa-eye me-1"></i>View
                                                    </button>
                                                </td>
                                            </tr>
                                        @endif
                                    @else
                                        <!-- Single value from API -->
                                        <tr class="table-warning">
                                            <td>
                                                <span class="badge bg-warning text-dark">API</span>
                                            </td>
                                            <td class="text-center">
                                                <strong>1</strong>
                                            </td>
                                            @if(count($allColumns) > 0)
                                                <td colspan="{{ count($allColumns) }}">
                                                    <span class="text-dark">{{ $apiData }}</span>
                                                </td>
                                            @else
                                                <td colspan="2">
                                                    <span class="text-dark">{{ $apiData }}</span>
                                                </td>
                                            @endif
                                            <td class="text-center">
                                                <button type="button" class="btn btn-sm btn-outline-warning" onclick="viewApiRecord(0)">
                                                    <i class="fas fa-eye me-1"></i>View
                                                </button>
                                            </td>
                                        </tr>
                                    @endif
                                @endif

                                @if(count($records) > 0 && count($columns) > 0)
                                    @foreach($records as $index => $record)
                                        <tr class="table-primary">
                                            <td>
                                                <span class="badge bg-primary">Local</span>
                                            </td>
                                            <td class="text-center">
                                                <strong>{{ $index + 1 }}</strong>
                                            </td>
                                            @foreach($allColumns as $column)
                                                <td>
                                                    @if(isset($record->$column))
                                                        @if($column === 'created_at' || $column === 'updated_at')
                                                            <span class="text-info">{{ $record->$column }}</span>
                                                        @elseif(is_bool($record->$column))
                                                            <span class="badge bg-{{ $record->$column ? 'success' : 'secondary' }}">
                                                                {{ $record->$column ? 'Yes' : 'No' }}
                                                            </span>
                                                        @elseif(is_numeric($record->$column))
                                                            <span class="text-primary fw-bold">{{ number_format($record->$column) }}</span>
                                                        @elseif(is_string($record->$column) && strlen($record->$column) > 50)
                                                            <span title="{{ $record->$column }}" class="text-truncate d-inline-block" style="max-width: 200px;">
                                                                {{ Str::limit($record->$column, 50) }}
                                                            </span>
                                                        @else
                                                            <span class="text-dark">{{ $record->$column }}</span>
                                                        @endif
                                                    @else
                                                        <span class="text-muted">-</span>
                                                    @endif
                                                </td>
                                            @endforeach
                                            <td class="text-center">
                                                <button type="button" class="btn btn-sm btn-outline-primary" onclick="viewRecord({{ $index }})">
                                                    <i class="fas fa-eye me-1"></i>View
                                                </button>
                                            </td>
                                        </tr>
                                    @endforeach
                                @endif

                                @if(($apiData === null || (is_array($apiData) && count($apiData) === 0)) && count($records) === 0)
                                    <tr>
                                        <td colspan="100" class="text-center py-5">
                                            <i class="fas fa-database fa-3x text-muted mb-3 d-block"></i>
                                            <h5 class="text-muted">No Data Available</h5>
                                            <p class="text-muted">
                                                @if($apiError)
                                                    API Error: {{ $apiError }}
                                                @else
                                                    No records found in both API and local database.
                                                @endif
                                            </p>
                                            <button type="button" class="btn btn-success" onclick="importData({{ $apiSource->id }}, '{{ $apiSource->name }}')">
                                                <i class="fas fa-download me-2"></i>Import Data Now
                                            </button>
                                        </td>
                                    </tr>
                                @endif
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    @if($apiError)
    <div class="row mb-4">
        <div class="col-12">
            <div class="card shadow border-danger">
                <div class="card-header py-3 bg-danger text-white">
                    <h6 class="m-0 font-weight-bold">
                        <i class="fas fa-exclamation-triangle me-2"></i>API Error
                    </h6>
                </div>
                <div class="card-body">
                    <div class="alert alert-danger">
                        <strong>Error fetching data from {{ $apiSource->url }}:</strong><br>
                        {{ $apiError }}
                    </div>
                </div>
            </div>
        </div>
    </div>
    @endif

</div>

<!-- Record Details Modal -->
<div class="modal fade" id="recordModal" tabindex="-1" role="dialog" aria-labelledby="recordModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="recordModalLabel">
                    <i class="fas fa-info-circle me-2"></i>Record Details
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div id="recordContent">
                    <!-- Record details will be loaded here -->
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<!-- Raw Data Modal -->
<div class="modal fade" id="rawDataModal" tabindex="-1" role="dialog" aria-labelledby="rawDataModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="rawDataModalLabel">
                    <i class="fas fa-code me-2"></i>Raw JSON Data
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <pre id="rawDataContent" class="bg-dark text-light p-3 rounded" style="max-height: 500px; overflow-y: auto;"></pre>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" onclick="copyRawData()">
                    <i class="fas fa-copy me-1"></i>Copy to Clipboard
                </button>
            </div>
        </div>
    </div>
</div>

<!-- JSON Data Modal -->
<div class="modal fade" id="jsonDataModal" tabindex="-1" role="dialog" aria-labelledby="jsonDataModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="jsonDataModalLabel">
                    <i class="fas fa-code me-2"></i>JSON Data
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <pre id="jsonDataContent" class="bg-dark text-light p-3 rounded" style="max-height: 400px; overflow-y: auto;"></pre>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" onclick="copyJsonData()">
                    <i class="fas fa-copy me-1"></i>Copy to Clipboard
                </button>
            </div>
        </div>
    </div>
</div>

<!-- JSON as Table Modal -->
<div class="modal fade" id="jsonTableModal" tabindex="-1" role="dialog" aria-labelledby="jsonTableModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="jsonTableModalLabel">
                    <i class="fas fa-table me-2"></i>JSON Data as Table
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div id="jsonTableContent" class="table-responsive">
                    <!-- Table content will be loaded here -->
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" onclick="exportTableToCSV()">
                    <i class="fas fa-download me-1"></i>Export CSV
                </button>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
// Real data from the server
const realData = @json($records);
const columns = @json($columns);
const apiData = @json($apiData);

function viewRecord(index) {
    const record = realData[index];
    if (record) {
        let content = '<div class="row">';
        
        columns.forEach((col, colIndex) => {
            const value = record[col];
            const displayValue = formatValue(value, col);
            
            content += `
                <div class="col-md-6 mb-3">
                    <div class="card">
                        <div class="card-header">
                            <strong>${formatColumnName(col)}</strong>
                        </div>
                        <div class="card-body">
                            ${displayValue}
                        </div>
                    </div>
                </div>
            `;
            
            // Add line break after every 2 columns
            if ((colIndex + 1) % 2 === 0) {
                content += '</div><div class="row">';
            }
        });
        
        content += '</div>';
        
        document.getElementById('recordContent').innerHTML = content;
        const modal = new bootstrap.Modal(document.getElementById('recordModal'));
        modal.show();
    }
}

function formatValue(value, column) {
    if (value === null || value === undefined) {
        return '<span class="text-muted">-</span>';
    }
    
    if (column === 'created_at' || column === 'updated_at') {
        return `<span class="text-info">${value}</span>`;
    }
    
    if (typeof value === 'object') {
        return `<button type="button" class="btn btn-sm btn-outline-info" onclick="viewJsonData(${realData.indexOf(value)}, '${column}')">
            <i class="fas fa-eye me-1"></i>View JSON
        </button>`;
    }
    
    if (typeof value === 'boolean') {
        return `<span class="badge badge-${value ? 'success' : 'secondary'}">${value ? 'Yes' : 'No'}</span>`;
    }
    
    if (typeof value === 'number') {
        return `<span class="text-primary">${value.toLocaleString()}</span>`;
    }
    
    return `<span>${value}</span>`;
}

function formatColumnName(column) {
    return column.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
}

function viewJsonData(index, column) {
    let data = null;
    
    // Check if it's API data or local data
    if (apiData && Array.isArray(apiData) && apiData[index]) {
        data = apiData[index][column];
    } else if (realData[index]) {
        data = realData[index][column];
    }
    
    if (data) {
        const jsonData = typeof data === 'string' ? data : JSON.stringify(data, null, 2);
        document.getElementById('jsonDataContent').textContent = jsonData;
        
        const modal = new bootstrap.Modal(document.getElementById('jsonDataModal'));
        modal.show();
    } else {
        if (window.Toast) {
            window.Toast.warning('No data available for this field');
        } else {
            alert('No data available for this field');
        }
    }
}

function viewAllJsonData() {
    const jsonResponse = {
        success: true,
        api_source: {
            id: {{ $apiSource->id }},
            name: "{{ $apiSource->name }}",
            url: "{{ $apiSource->url }}",
            method: "{{ $apiSource->method }}",
            table_name: "{{ $apiSource->table_name }}",
            status: "{{ $apiSource->status }}"
        },
        external_api: {
            success: apiData !== null,
            error: @json($apiError),
            data: apiData
        },
        local_data: {
            total_records: realData.length,
            columns: columns,
            records: realData
        },
        timestamp: new Date().toISOString()
    };
    
    const jsonData = JSON.stringify(jsonResponse, null, 2);
    document.getElementById('jsonDataContent').textContent = jsonData;
    
    const modal = new bootstrap.Modal(document.getElementById('jsonDataModal'));
    modal.show();
}

function showRawData() {
    if (realData.length === 0 && (!apiData || apiData.length === 0)) {
        if (window.Toast) {
            window.Toast.warning('No data available to display');
        } else {
            alert('No data available to display');
        }
        return;
    }
    
    const allData = {
        external_api: apiData,
        local_data: realData
    };
    
    const rawData = JSON.stringify(allData, null, 2);
    document.getElementById('rawDataContent').textContent = rawData;
    
    const modal = new bootstrap.Modal(document.getElementById('rawDataModal'));
    modal.show();
}

function refreshApiData() {
    if (window.LoadingOverlay) {
        window.LoadingOverlay.show('Refreshing API data...');
    }
    
    // Reload the page to fetch fresh API data
    location.reload();
}

function viewJsonAsTable() {
    const jsonResponse = {
        success: true,
        api_source: {
            id: {{ $apiSource->id }},
            name: "{{ $apiSource->name }}",
            url: "{{ $apiSource->url }}",
            method: "{{ $apiSource->method }}",
            table_name: "{{ $apiSource->table_name }}",
            status: "{{ $apiSource->status }}"
        },
        external_api: {
            success: apiData !== null,
            error: @json($apiError),
            data: apiData
        },
        local_data: {
            total_records: realData.length,
            columns: columns,
            records: realData
        },
        timestamp: new Date().toISOString()
    };
    
    const tableContent = convertJsonToTable(jsonResponse);
    document.getElementById('jsonTableContent').innerHTML = tableContent;
    
    const modal = new bootstrap.Modal(document.getElementById('jsonTableModal'));
    modal.show();
}

function convertJsonToTable(data, level = 0, parentKey = '') {
    if (level > 3) {
        return '<div class="text-muted">Data too deep to display</div>';
    }
    
    let html = '';
    
    if (Array.isArray(data)) {
        // Handle arrays
        if (data.length === 0) {
            return '<div class="text-muted">Empty array</div>';
        }
        
        if (typeof data[0] === 'object' && data[0] !== null) {
            // Array of objects - create table
            const keys = Object.keys(data[0]);
            html += '<table class="table table-bordered table-striped table-sm">';
            html += '<thead class="table-dark"><tr>';
            keys.forEach(key => {
                html += `<th>${formatColumnName(key)}</th>`;
            });
            html += '</tr></thead><tbody>';
            
            data.forEach((item, index) => {
                html += '<tr>';
                keys.forEach(key => {
                    const value = item[key];
                    html += '<td>' + formatTableCellValue(value, level + 1, `${parentKey}[${index}].${key}`) + '</td>';
                });
                html += '</tr>';
            });
            
            html += '</tbody></table>';
        } else {
            // Simple array
            html += '<table class="table table-bordered table-striped table-sm">';
            html += '<thead class="table-dark"><tr><th>Index</th><th>Value</th></tr></thead><tbody>';
            data.forEach((item, index) => {
                html += '<tr>';
                html += `<td>${index}</td>`;
                html += '<td>' + formatTableCellValue(item, level + 1, `${parentKey}[${index}]`) + '</td>';
                html += '</tr>';
            });
            html += '</tbody></table>';
        }
    } else if (typeof data === 'object' && data !== null) {
        // Handle objects
        const keys = Object.keys(data);
        if (keys.length === 0) {
            return '<div class="text-muted">Empty object</div>';
        }
        
        html += '<table class="table table-bordered table-striped table-sm">';
        html += '<thead class="table-dark"><tr><th>Field</th><th>Value</th></tr></thead><tbody>';
        
        keys.forEach(key => {
            const value = data[key];
            html += '<tr>';
            html += `<td><strong>${formatColumnName(key)}</strong></td>`;
            html += '<td>' + formatTableCellValue(value, level + 1, `${parentKey}.${key}`) + '</td>';
            html += '</tr>';
        });
        
        html += '</tbody></table>';
    } else {
        // Handle primitive values
        html += formatTableCellValue(data, level, parentKey);
    }
    
    return html;
}

function formatTableCellValue(value, level, key) {
    if (value === null || value === undefined) {
        return '<span class="text-muted">-</span>';
    }
    
    if (typeof value === 'boolean') {
        return `<span class="badge bg-${value ? 'success' : 'secondary'}">${value ? 'Yes' : 'No'}</span>`;
    }
    
    if (typeof value === 'number') {
        return `<span class="text-primary">${value.toLocaleString()}</span>`;
    }
    
    if (typeof value === 'string') {
        if (value.length > 100) {
            return `<span title="${value}">${value.substring(0, 100)}...</span>`;
        }
        return value;
    }
    
    if (typeof value === 'object') {
        if (level < 3) {
            return convertJsonToTable(value, level, key);
        } else {
            return `<button type="button" class="btn btn-sm btn-outline-info" onclick="viewJsonData(0, '${key}')">
                <i class="fas fa-eye me-1"></i>View JSON
            </button>`;
        }
    }
    
    return String(value);
}

function exportTableToCSV() {
    const table = document.querySelector('#jsonTableContent table');
    if (!table) {
        if (window.Toast) {
            window.Toast.warning('No table data to export');
        } else {
            alert('No table data to export');
        }
        return;
    }
    
    let csv = [];
    const rows = table.querySelectorAll('tr');
    
    rows.forEach(row => {
        const cols = row.querySelectorAll('td, th');
        const rowData = [];
        cols.forEach(col => {
            // Get text content and clean it
            let text = col.textContent || col.innerText || '';
            text = text.replace(/\s+/g, ' ').trim();
            // Escape quotes and wrap in quotes
            text = '"' + text.replace(/"/g, '""') + '"';
            rowData.push(text);
        });
        csv.push(rowData.join(','));
    });
    
    const csvContent = csv.join('\n');
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', `{{ $apiSource->name }}_data_${new Date().toISOString().split('T')[0]}.csv`);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    
    if (window.Toast) {
        window.Toast.success('CSV exported successfully');
    } else {
        alert('CSV exported successfully');
    }
}

function viewApiDataAsTable() {
    if (!apiData) {
        if (window.Toast) {
            window.Toast.warning('No external API data available');
        } else {
            alert('No external API data available');
        }
        return;
    }
    
    const tableContent = convertJsonToTable(apiData);
    document.getElementById('jsonTableContent').innerHTML = tableContent;
    
    const modal = new bootstrap.Modal(document.getElementById('jsonTableModal'));
    modal.show();
}

function toggleApiView() {
    const apiDataContainer = document.querySelector('.card-body.p-0[data-view="cards"]');
    const currentView = apiDataContainer.getAttribute('data-view') || 'cards';
    
    if (currentView === 'cards') {
        // Switch to table view
        apiDataContainer.setAttribute('data-view', 'table');
        const tableContent = convertJsonToTable(apiData);
        apiDataContainer.innerHTML = `<div class="p-3">${tableContent}</div>`;
    } else {
        // Switch back to cards view
        apiDataContainer.setAttribute('data-view', 'cards');
        location.reload(); // Reload to get back to card view
    }
}

function toggleLocalView() {
    const localDataContainer = document.querySelector('.card-body.p-0[data-view="cards"]:not(:first-child)');
    const currentView = localDataContainer.getAttribute('data-view') || 'cards';
    
    if (currentView === 'cards') {
        // Switch to table view
        localDataContainer.setAttribute('data-view', 'table');
        const tableContent = convertLocalDataToTable();
        localDataContainer.innerHTML = `<div class="p-3">${tableContent}</div>`;
    } else {
        // Switch back to cards view
        localDataContainer.setAttribute('data-view', 'cards');
        location.reload(); // Reload to get back to card view
    }
}

function convertLocalDataToTable() {
    if (realData.length === 0) {
        return '<div class="text-muted">No local data available</div>';
    }
    
    let html = '<table class="table table-bordered table-striped">';
    html += '<thead class="table-dark"><tr>';
    columns.forEach(col => {
        html += `<th>${formatColumnName(col)}</th>`;
    });
    html += '<th>Actions</th>';
    html += '</tr></thead><tbody>';
    
    realData.forEach((record, index) => {
        html += '<tr>';
        columns.forEach(col => {
            const value = record[col];
            html += '<td>' + formatTableCellValue(value, 0, col) + '</td>';
        });
        html += `<td><button type="button" class="btn btn-sm btn-outline-primary" onclick="viewRecord(${index})">
            <i class="fas fa-eye me-1"></i>View
        </button></td>`;
        html += '</tr>';
    });
    
    html += '</tbody></table>';
    return html;
}

function viewApiRecord(index) {
    if (!apiData) {
        if (window.Toast) {
            window.Toast.warning('No API data available');
        } else {
            alert('No API data available');
        }
        return;
    }
    
    let record = null;
    if (Array.isArray(apiData)) {
        if (apiData[index]) {
            record = apiData[index];
        } else if (apiData.length === 1) {
            record = apiData[0];
        }
    } else {
        record = apiData;
    }
    
    if (record) {
        let content = '<div class="row">';
        const keys = Object.keys(record);
        
        keys.forEach((key, keyIndex) => {
            const value = record[key];
            const displayValue = formatValue(value, key);
            
            content += `
                <div class="col-md-6 mb-3">
                    <div class="card">
                        <div class="card-header">
                            <strong>${formatColumnName(key)}</strong>
                        </div>
                        <div class="card-body">
                            ${displayValue}
                        </div>
                    </div>
                </div>
            `;
            
            // Add line break after every 2 columns
            if ((keyIndex + 1) % 2 === 0) {
                content += '</div><div class="row">';
            }
        });
        
        content += '</div>';
        
        document.getElementById('recordContent').innerHTML = content;
        const modal = new bootstrap.Modal(document.getElementById('recordModal'));
        modal.show();
    }
}

function exportCombinedData() {
    const table = document.querySelector('.table-responsive table');
    if (!table) {
        if (window.Toast) {
            window.Toast.warning('No data to export');
        } else {
            alert('No data to export');
        }
        return;
    }
    
    let csv = [];
    const rows = table.querySelectorAll('tr');
    
    rows.forEach(row => {
        const cols = row.querySelectorAll('td, th');
        const rowData = [];
        cols.forEach(col => {
            // Get text content and clean it
            let text = col.textContent || col.innerText || '';
            text = text.replace(/\s+/g, ' ').trim();
            // Escape quotes and wrap in quotes
            text = '"' + text.replace(/"/g, '""') + '"';
            rowData.push(text);
        });
        csv.push(rowData.join(','));
    });
    
    const csvContent = csv.join('\n');
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', `{{ $apiSource->name }}_combined_data_${new Date().toISOString().split('T')[0]}.csv`);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    
    if (window.Toast) {
        window.Toast.success('Combined data exported successfully');
    } else {
        alert('Combined data exported successfully');
    }
}

// Auto-adjust column widths based on content
function adjustColumnWidths() {
    const table = document.querySelector('.table-responsive table');
    if (!table) return;
    
    const headers = table.querySelectorAll('thead th');
    const dataRows = table.querySelectorAll('tbody tr');
    
    headers.forEach((header, index) => {
        if (index <= 1 || index === headers.length - 1) return; // Skip source, record #, and actions columns
        
        let maxWidth = header.offsetWidth;
        
        // Check data rows for this column
        dataRows.forEach(row => {
            const cell = row.cells[index];
            if (cell) {
                const contentWidth = cell.scrollWidth;
                if (contentWidth > maxWidth) {
                    maxWidth = contentWidth;
                }
            }
        });
        
        // Set minimum and maximum constraints
        maxWidth = Math.max(120, Math.min(300, maxWidth + 20));
        
        // Apply width to all cells in this column
        header.style.width = maxWidth + 'px';
        dataRows.forEach(row => {
            const cell = row.cells[index];
            if (cell) {
                cell.style.width = maxWidth + 'px';
            }
        });
    });
}

// Call adjustColumnWidths when page loads
document.addEventListener('DOMContentLoaded', function() {
    setTimeout(adjustColumnWidths, 100);
});

// Adjust column widths when window resizes
window.addEventListener('resize', function() {
    setTimeout(adjustColumnWidths, 100);
});



function copyRawData() {
    const rawData = document.getElementById('rawDataContent').textContent;
    navigator.clipboard.writeText(rawData).then(function() {
        if (window.Toast) {
            window.Toast.success('Raw data copied to clipboard');
        } else {
            alert('Raw data copied to clipboard');
        }
    }).catch(function() {
        if (window.Toast) {
            window.Toast.error('Failed to copy data to clipboard');
        } else {
            alert('Failed to copy data to clipboard');
        }
    });
}

function copyJsonData() {
    const jsonData = document.getElementById('jsonDataContent').textContent;
    navigator.clipboard.writeText(jsonData).then(function() {
        if (window.Toast) {
            window.Toast.success('JSON data copied to clipboard');
        } else {
            alert('JSON data copied to clipboard');
        }
    }).catch(function() {
        if (window.Toast) {
            window.Toast.error('Failed to copy JSON data to clipboard');
        } else {
            alert('Failed to copy JSON data to clipboard');
        }
    });
}

function autoImportData(id, name) {
    if (confirm(`Auto import data from "${name}" with intelligent column mapping?`)) {
        if (window.LoadingOverlay) {
            window.LoadingOverlay.show('Auto importing data with column mapping...');
        }
        
        fetch(`/api-sources/${id}/auto-import`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (window.LoadingOverlay) {
                window.LoadingOverlay.hide();
            }
            if (data.success) {
                let message = data.message;
                if (data.mapped_fields && data.mapped_fields.length > 0) {
                    message += `\nMapped fields: ${data.mapped_fields.join(', ')}`;
                }
                
                if (window.Toast) {
                    window.Toast.success(message);
                } else {
                    alert(message);
                }
                setTimeout(() => location.reload(), 2000);
            } else {
                if (window.Toast) {
                    window.Toast.error(data.message || 'Auto import failed');
                } else {
                    alert(data.message || 'Auto import failed');
                }
            }
        })
        .catch(error => {
            if (window.LoadingOverlay) {
                window.LoadingOverlay.hide();
            }
            if (window.Toast) {
                window.Toast.error('Auto import failed: ' + error.message);
            } else {
                alert('Auto import failed: ' + error.message);
            }
        });
    }
}

function showImportStats(id) {
    if (window.LoadingOverlay) {
        window.LoadingOverlay.show('Loading import statistics...');
    }
    
    fetch(`/api-sources/${id}/import-stats`, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        }
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        return response.json();
    })
    .then(data => {
        if (window.LoadingOverlay) {
            window.LoadingOverlay.hide();
        }
        
        if (data.success && data.stats) {
            const stats = data.stats;
            let message = `Import Statistics:\n\n`;
            
            // Add table status
            if (stats.table_exists === false) {
                message += `⚠️  ${stats.message}\n\n`;
            }
            
            message += `Total Records: ${stats.total_records}\n`;
            message += `Today's Records: ${stats.today_records}\n`;
            message += `Schedule Type: ${stats.schedule_type || 'Manual'}\n`;
            message += `Schedule Frequency: ${stats.schedule_frequency || 'N/A'}\n`;
            
            if (stats.last_import) {
                const lastImport = new Date(stats.last_import);
                message += `Last Import: ${lastImport.toLocaleString()}\n`;
            } else {
                message += `Last Import: Never\n`;
            }
            
            if (stats.next_import) {
                const nextImport = new Date(stats.next_import);
                message += `Next Import: ${nextImport.toLocaleString()}\n`;
            } else {
                message += `Next Import: Not scheduled\n`;
            }
            
            if (window.Toast) {
                window.Toast.info(message);
            } else {
                alert(message);
            }
        } else {
            const errorMessage = data.message || 'Failed to load import statistics';
            if (window.Toast) {
                window.Toast.error(errorMessage);
            } else {
                alert(errorMessage);
            }
        }
    })
    .catch(error => {
        if (window.LoadingOverlay) {
            window.LoadingOverlay.hide();
        }
        console.error('Import stats error:', error);
        
        const errorMessage = 'Failed to load import statistics: ' + error.message;
        if (window.Toast) {
            window.Toast.error(errorMessage);
        } else {
            alert(errorMessage);
        }
    });
}

function importData(id, name) {
    if (confirm(`Import new data from "${name}"?`)) {
        if (window.LoadingOverlay) {
            window.LoadingOverlay.show('Importing data...');
        }
        
        fetch(`/api-sources/${id}/import`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (window.LoadingOverlay) {
                window.LoadingOverlay.hide();
            }
            if (data.success) {
                if (window.Toast) {
                    window.Toast.success(data.message || 'Data imported successfully');
                } else {
                    alert(data.message || 'Data imported successfully');
                }
                setTimeout(() => location.reload(), 2000);
            } else {
                if (window.Toast) {
                    window.Toast.error(data.message || 'Import failed');
                } else {
                    alert(data.message || 'Import failed');
                }
            }
        })
        .catch(error => {
            if (window.LoadingOverlay) {
                window.LoadingOverlay.hide();
            }
            if (window.Toast) {
                window.Toast.error('Import failed: ' + error.message);
            } else {
                alert('Import failed: ' + error.message);
            }
        });
    }
}

function refreshData() {
    if (window.Toast) {
        window.Toast.info('Refreshing data...');
    }
    setTimeout(() => {
        location.reload();
    }, 1000);
}

function exportData() {
    if (realData.length === 0) {
        if (window.Toast) {
            window.Toast.warning('No data to export');
        } else {
            alert('No data to export');
        }
        return;
    }
    
    // Create CSV content
    const headers = columns.map(col => formatColumnName(col));
    const csvContent = [
        headers.join(','),
        ...realData.map(record => 
            columns.map(col => {
                const value = record[col];
                if (value === null || value === undefined) return '';
                if (typeof value === 'object') return JSON.stringify(value);
                return `"${String(value).replace(/"/g, '""')}"`;
            }).join(',')
        )
    ].join('\n');
    
    // Create and download file
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', `{{ $apiSource->table_name }}_data_${new Date().toISOString().split('T')[0]}.csv`);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    
    if (window.Toast) {
        window.Toast.success('Data exported successfully');
    } else {
        alert('Data exported successfully');
    }
}

function clearData() {
    if (confirm('Are you sure you want to clear all imported data? This action cannot be undone.')) {
        if (window.LoadingOverlay) {
            window.LoadingOverlay.show('Clearing data...');
        }
        
        fetch(`/api-sources/{{ $apiSource->id }}/clear-data`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (window.LoadingOverlay) {
                window.LoadingOverlay.hide();
            }
            if (data.success) {
                if (window.Toast) {
                    window.Toast.success(data.message || 'Data cleared successfully');
                } else {
                    alert(data.message || 'Data cleared successfully');
                }
                setTimeout(() => location.reload(), 2000);
            } else {
                if (window.Toast) {
                    window.Toast.error(data.message || 'Failed to clear data');
                } else {
                    alert(data.message || 'Failed to clear data');
                }
            }
        })
        .catch(error => {
            if (window.LoadingOverlay) {
                window.LoadingOverlay.hide();
            }
            if (window.Toast) {
                window.Toast.error('Failed to clear data: ' + error.message);
            } else {
                alert('Failed to clear data: ' + error.message);
            }
        });
    }
}

// Initialize DataTable if available
document.addEventListener('DOMContentLoaded', function() {
    if (typeof $.fn.DataTable !== 'undefined' && $('#dataTable').length) {
        $('#dataTable').DataTable({
            pageLength: 25,
            order: [[0, 'desc']],
            responsive: true,
            language: {
                search: "Search:",
                lengthMenu: "Show _MENU_ records per page",
                info: "Showing _START_ to _END_ of _TOTAL_ records",
                infoEmpty: "Showing 0 to 0 of 0 records",
                infoFiltered: "(filtered from _MAX_ total records)"
            }
        });
    }
});
</script>
@endpush
@endsection 