@extends('layouts.app')

@section('title', 'API Sources')

@section('breadcrumb')
    <li class="breadcrumb-item active">API Sources</li>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h1 class="h3 mb-0 text-gray-800">
                    <i class="fas fa-database me-2"></i>API Sources
                </h1>
                <div class="btn-group">
                    <button type="button" class="btn btn-success" onclick="bulkAutoImport()">
                        <i class="fas fa-robot me-2"></i>Bulk Auto Import
                    </button>
                    <a href="{{ route('api-sources.create') }}" class="btn btn-primary">
                        <i class="fas fa-plus me-2"></i>Add New API Source
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-primary shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">
                                Total Sources</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">{{ $api_sources->total() }}</div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-database fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-success shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                Active Sources</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">{{ \App\Models\ApiSource::where('status', 'active')->count() }}</div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-check-circle fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-info shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">
                                Total Requests</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">{{ \App\Models\ApiLog::count() }}</div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-chart-line fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-warning shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">
                                Inactive Sources</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">{{ \App\Models\ApiSource::where('status', 'inactive')->count() }}</div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-pause-circle fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- API Sources Table -->
    <div class="row">
        <div class="col-12">
            <div class="card shadow mb-4">
                <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between">
                    <h6 class="m-0 font-weight-bold text-primary">API Sources List</h6>
                    <div class="dropdown no-arrow">
                        <a class="dropdown-toggle" href="#" role="button" id="dropdownMenuLink"
                            data-bs-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                            <i class="fas fa-ellipsis-v fa-sm fa-fw text-gray-400"></i>
                        </a>
                        <div class="dropdown-menu dropdown-menu-right shadow animated--fade-in"
                            aria-labelledby="dropdownMenuLink">
                            <div class="dropdown-header">Actions:</div>
                            <a class="dropdown-item" href="{{ route('api-sources.create') }}">
                                <i class="fas fa-plus fa-sm fa-fw mr-2 text-gray-400"></i>
                                Add New Source
                            </a>
                            <div class="dropdown-divider"></div>
                            <a class="dropdown-item" href="#" onclick="refreshPage()">
                                <i class="fas fa-sync fa-sm fa-fw mr-2 text-gray-400"></i>
                                Refresh
                            </a>
                            <div class="dropdown-divider"></div>
                            <a class="dropdown-item" href="#" onclick="testDelete(11, 'ambil data')">
                                <i class="fas fa-trash fa-sm fa-fw mr-2 text-gray-400"></i>
                                Test Delete
                            </a>
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    @if($api_sources->count() > 0)
                        <div class="table-responsive">
                            <table class="table table-bordered" id="dataTable" width="100%" cellspacing="0">
                                <thead>
                                    <tr>
                                        <th>Name</th>
                                        <th>URL</th>
                                        <th>Method</th>
                                        <th>Table</th>
                                        <th>Schedule</th>
                                        <th>Status</th>
                                        <th>Last Fetched</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($api_sources as $source)
                                        <tr>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="icon-circle bg-primary me-3">
                                                        <i class="fas fa-database text-white"></i>
                                                    </div>
                                                    <div>
                                                        <div class="font-weight-bold">{{ $source->name }}</div>
                                                        @if($source->description)
                                                            <div class="small text-gray-500">{{ Str::limit($source->description, 50) }}</div>
                                                        @endif
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <a href="{{ $source->url }}" target="_blank" class="text-decoration-none">
                                                    {{ Str::limit($source->url, 40) }}
                                                    <i class="fas fa-external-link-alt ms-1 text-gray-400"></i>
                                                </a>
                                            </td>
                                            <td>
                                                <span class="badge bg-{{ $source->method == 'GET' ? 'success' : ($source->method == 'POST' ? 'primary' : 'warning') }}">
                                                    {{ $source->method }}
                                                </span>
                                            </td>
                                            <td>
                                                <code class="bg-light p-1 rounded">{{ $source->table_name }}</code>
                                            </td>
                                            <td>
                                                <span class="badge bg-info">{{ ucfirst($source->schedule_type) }}</span>
                                                @if($source->schedule_frequency)
                                                    <br><small class="text-muted">Every {{ $source->schedule_frequency }}{{ $source->schedule_type == 'hourly' ? 'h' : 'd' }}</small>
                                                @endif
                                            </td>
                                            <td>
                                                <span class="badge bg-{{ $source->status == 'active' ? 'success' : 'secondary' }}">
                                                    {{ ucfirst($source->status) }}
                                                </span>
                                            </td>
                                            <td>
                                                @if($source->last_fetched_at)
                                                    <span title="{{ $source->last_fetched_at }}">
                                                        {{ $source->last_fetched_at->diffForHumans() }}
                                                    </span>
                                                @else
                                                    <span class="text-muted">Never</span>
                                                @endif
                                            </td>
                                            <td>
                                                <div class="d-flex flex-wrap gap-1">
                                                    <a href="{{ route('api-sources.show', $source) }}" 
                                                       class="btn btn-sm btn-outline-info" title="View Details">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                    <a href="{{ route('api-sources.edit', $source) }}" 
                                                       class="btn btn-sm btn-outline-primary" title="Edit">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <button type="button" 
                                                            class="btn btn-sm btn-outline-success" 
                                                            title="Import Data"
                                                            onclick="importData({{ $source->id }}, '{{ $source->name }}')">
                                                        <i class="fas fa-download"></i>
                                                    </button>
                                                    <a href="{{ route('api-sources.logs', $source) }}" 
                                                       class="btn btn-sm btn-outline-warning" title="View Logs">
                                                        <i class="fas fa-list"></i>
                                                    </a>
                                                    <a href="{{ route('api-sources.preview-pdf', $source) }}" 
                                                       class="btn btn-sm btn-outline-success" title="Preview PDF" target="_blank">
                                                        <i class="fas fa-eye"></i> Preview
                                                    </a>
                                                    <a href="{{ route('api-sources.export-pdf', $source) }}" 
                                                       class="btn btn-sm btn-outline-danger" title="Export PDF" target="_blank">
                                                        <i class="fas fa-file-pdf"></i> PDF
                                                    </a>
                                                    <button type="button" 
                                                            class="btn btn-sm btn-outline-danger" 
                                                            title="Delete"
                                                            onclick="deleteSource({{ $source->id }}, '{{ $source->name }}')">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                        
                        <!-- Pagination -->
                        <div class="d-flex justify-content-between align-items-center mt-3">
                            <div class="small text-muted">
                                Showing {{ $api_sources->firstItem() }} to {{ $api_sources->lastItem() }} of {{ $api_sources->total() }} results
                            </div>
                            {{ $api_sources->links() }}
                        </div>
                    @else
                        <div class="text-center py-5">
                            <div class="mb-3">
                                <i class="fas fa-database fa-3x text-gray-300"></i>
                            </div>
                            <h5 class="text-gray-500">No API Sources Found</h5>
                            <p class="text-gray-400">Get started by creating your first API source.</p>
                            <a href="{{ route('api-sources.create') }}" class="btn btn-primary">
                                <i class="fas fa-plus me-2"></i>Create API Source
                            </a>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>



@push('scripts')
<script>
function deleteSource(id, name) {
    console.log('deleteSource called with:', id, name);
    
    if (confirm(`Are you sure you want to delete "${name}"? This action cannot be undone.`)) {
        directDelete(id, name);
    }
}

function directDelete(id, name) {
    console.log('Direct delete called for:', id, name);
    
    if (confirm(`Are you sure you want to delete "${name}"? This action cannot be undone.`)) {
        const formData = new FormData();
        formData.append('_token', document.querySelector('meta[name="csrf-token"]').getAttribute('content'));
        formData.append('_method', 'DELETE');
        
        fetch(`/api-sources/${id}`, {
            method: 'POST',
            body: formData,
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => {
            console.log('Response status:', response.status);
            if (response.redirected) {
                window.location.href = response.url;
            } else if (response.ok) {
                alert('API Source deleted successfully');
                location.reload();
            } else {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
        })
        .catch(error => {
            console.error('Delete error:', error);
            alert('Error deleting API Source: ' + error.message);
        });
    }
}

function bulkAutoImport() {
    if (confirm('Auto import data from all active API sources with intelligent column mapping?')) {
        window.LoadingOverlay.show('Bulk auto importing data...');
        
        fetch('/api-sources/bulk-auto-import', {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            window.LoadingOverlay.hide();
            if (data.success) {
                let message = data.message;
                if (data.results && data.results.length > 0) {
                    message += '\n\nResults:';
                    data.results.forEach(result => {
                        const status = result.success ? '✓' : '✗';
                        message += `\n${status} ${result.source_name}: ${result.message}`;
                    });
                }
                
                window.Toast.success(message);
                setTimeout(() => location.reload(), 3000);
            } else {
                window.Toast.error(data.message || 'Bulk auto import failed');
            }
        })
        .catch(error => {
            window.LoadingOverlay.hide();
            window.Toast.error('Bulk auto import failed: ' + error.message);
        });
    }
}

function importData(id, name) {
    if (confirm(`Import data from "${name}"?`)) {
        window.LoadingOverlay.show('Importing data...');
        
        fetch(`/api-sources/${id}/import`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            window.LoadingOverlay.hide();
            if (data.success) {
                window.Toast.success(data.message || 'Data imported successfully');
                setTimeout(() => location.reload(), 1500);
            } else {
                window.Toast.error(data.message || 'Import failed');
            }
        })
        .catch(error => {
            window.LoadingOverlay.hide();
            window.Toast.error('Import failed: ' + error.message);
        });
    }
}

function refreshPage() {
    location.reload();
}

// Test function to verify delete functionality
function testDelete(id, name) {
    console.log('Testing delete functionality for:', id, name);
    directDelete(id, name);
}


</script>
@endpush
@endsection 