@extends('layouts.app')

@section('title', 'API Source Details')

@section('breadcrumb')
    <li class="breadcrumb-item">
        <a href="{{ route('api-sources.index') }}">API Sources</a>
    </li>
    <li class="breadcrumb-item active">{{ $apiSource->name }}</li>
@endsection

@section('styles')
<style>
/* Badge Colors */
.badge.bg-success {
    color: #2b4a34 !important;
    background-color: #9de3bf !important;
}

.badge.bg-primary {
    color: #1a365d !important;
    background-color: #9ec5fe !important;
}

.badge.bg-info {
    color: #1e4e8c !important;
    background-color: #a5d7f7 !important;
}

.badge.bg-warning {
    color: #603b00 !important;
    background-color: #ffd699 !important;
}

.badge.bg-secondary {
    color: #383d41 !important;
    background-color: #e2e6ea !important;
}

.badge.bg-danger {
    color: #fff !important;
    background-color: #dc3545 !important;
}
</style>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h1 class="h3 mb-0 text-gray-800">
                    <i class="fas fa-database me-2"></i>{{ $apiSource->name }}
                </h1>
                <div class="btn-group">
                    <a href="{{ route('api-sources.edit', $apiSource) }}" class="btn btn-primary">
                        <i class="fas fa-edit me-2"></i>Edit
                    </a>
                    <button type="button" class="btn btn-success" onclick="importData({{ $apiSource->id }}, '{{ $apiSource->name }}')">
                        <i class="fas fa-download me-2"></i>Import Data
                    </button>
                    <a href="{{ route('api-sources.index') }}" class="btn btn-secondary">
                        <i class="fas fa-arrow-left me-2"></i>Back to List
                    </a>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <!-- Main Information -->
        <div class="col-lg-8">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-info-circle me-2"></i>API Source Information
                    </h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Name:</label>
                                <p class="text-gray-800">{{ $apiSource->name }}</p>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Status:</label>
                                <p>
                                    <span class="badge bg-{{ $apiSource->status == 'active' ? 'success' : 'secondary' }}">
                                        {{ ucfirst($apiSource->status) }}
                                    </span>
                                </p>
                            </div>
                        </div>
                    </div>

                    @if($apiSource->description)
                        <div class="mb-3">
                            <label class="form-label font-weight-bold">Description:</label>
                            <p class="text-gray-700">{{ $apiSource->description }}</p>
                        </div>
                    @endif

                    <hr class="my-4">
                    <h5 class="mb-3">
                        <i class="fas fa-cog me-2"></i>API Configuration
                    </h5>

                    <div class="row">
                        <div class="col-md-12">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">API URL:</label>
                                <p>
                                    <a href="{{ $apiSource->url }}" target="_blank" class="text-decoration-none">
                                        {{ $apiSource->url }}
                                        <i class="fas fa-external-link-alt ms-2 text-gray-400"></i>
                                    </a>
                                </p>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">HTTP Method:</label>
                                <p>
                                    <span class="badge bg-{{ $apiSource->method == 'GET' ? 'success' : ($apiSource->method == 'POST' ? 'primary' : 'warning') }}">
                                        {{ $apiSource->method }}
                                    </span>
                                </p>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Table Name:</label>
                                <p><code class="bg-light p-2 rounded">{{ $apiSource->table_name }}</code></p>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Authentication:</label>
                                <p>
                                    <span class="badge bg-info">
                                        {{ $apiSource->auth_type == 'none' ? 'None' : ucfirst(str_replace('_', ' ', $apiSource->auth_type)) }}
                                    </span>
                                </p>
                            </div>
                        </div>
                    </div>

                    @if($apiSource->auth_type === 'basic_auth')
                        <tr>
                            <th>Basic Auth Username</th>
                            <td>{{ $apiSource->auth_username }}</td>
                        </tr>
                        <tr>
                            <th>Basic Auth Password</th>
                            <td>********</td>
                        </tr>
                    @endif

                    <hr class="my-4">
                    <h5 class="mb-3">
                        <i class="fas fa-clock me-2"></i>Scheduling
                    </h5>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Schedule Type:</label>
                                <p>
                                    <span class="badge bg-info">{{ ucfirst($apiSource->schedule_type) }}</span>
                                    @if($apiSource->schedule_frequency && $apiSource->schedule_type !== 'manual')
                                        <small class="text-muted ms-2">Every {{ $apiSource->schedule_frequency }} {{ $apiSource->schedule_type == 'hourly' ? 'hour(s)' : ($apiSource->schedule_type == 'daily' ? 'day(s)' : 'week(s)') }}</small>
                                    @endif
                                </p>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Last Fetched:</label>
                                <p class="text-gray-700">
                                    @if($apiSource->last_fetched_at)
                                        {{ $apiSource->last_fetched_at->format('d M Y, H:i:s') }}
                                        <small class="text-muted">({{ $apiSource->last_fetched_at->diffForHumans() }})</small>
                                    @else
                                        <span class="text-muted">Never</span>
                                    @endif
                                </p>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Created:</label>
                                <p class="text-gray-700">
                                    {{ $apiSource->created_at->format('d M Y, H:i:s') }}
                                    <small class="text-muted">({{ $apiSource->created_at->diffForHumans() }})</small>
                                </p>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Last Updated:</label>
                                <p class="text-gray-700">
                                    {{ $apiSource->updated_at->format('d M Y, H:i:s') }}
                                    <small class="text-muted">({{ $apiSource->updated_at->diffForHumans() }})</small>
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Recent Logs -->
            <div class="card shadow mb-4">
                <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-list me-2"></i>Recent Activity
                    </h6>
                    <a href="{{ route('api-sources.logs', $apiSource) }}" class="btn btn-sm btn-outline-primary">
                        View All Logs
                    </a>
                </div>
                <div class="card-body">
                    @php
                        $recentLogs = $apiSource->apiLogs()->latest()->limit(5)->get();
                    @endphp
                    
                    @if($recentLogs->count() > 0)
                        @foreach($recentLogs as $log)
                            <div class="d-flex align-items-center mb-3 pb-3 border-bottom">
                                <div class="me-3">
                                    <div class="icon-circle bg-{{ $log->status_code >= 200 && $log->status_code < 300 ? 'success' : 'danger' }}">
                                        <i class="fas fa-{{ $log->status_code >= 200 && $log->status_code < 300 ? 'check' : 'times' }} text-white"></i>
                                    </div>
                                </div>
                                <div class="flex-grow-1">
                                    <div class="d-flex justify-content-between align-items-start">
                                        <div>
                                            <div class="font-weight-bold">
                                                {{ $log->status_code >= 200 && $log->status_code < 300 ? 'Data Import Successful' : 'Import Failed' }}
                                            </div>
                                            <div class="small text-gray-500">
                                                {{ $log->created_at->format('d M Y, H:i:s') }} 
                                                ({{ $log->created_at->diffForHumans() }})
                                            </div>
                                            @if($log->status_code >= 200 && $log->status_code < 300 && $log->response_data)
                                                <div class="small text-success">
                                                    <i class="fas fa-download me-1"></i>Data imported successfully
                                                </div>
                                            @endif
                                            @if($log->error_message)
                                                <div class="small text-danger">
                                                    <i class="fas fa-exclamation-triangle me-1"></i>{{ Str::limit($log->error_message, 100) }}
                                                </div>
                                            @endif
                                        </div>
                                        <div class="text-end">
                                            @if($log->status_code)
                                                <span class="badge badge-{{ $log->status_code < 300 ? 'success' : ($log->status_code < 400 ? 'warning' : 'danger') }}">
                                                    {{ $log->status_code }}
                                                </span>
                                            @endif
                                            @if($log->response_time)
                                                <div class="small text-muted">{{ number_format($log->response_time, 2) }}s</div>
                                            @endif
                                        </div>
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    @else
                        <div class="text-center py-4">
                            <i class="fas fa-history fa-2x text-gray-300 mb-2"></i>
                            <p class="text-muted">No activity logs found</p>
                            <button type="button" class="btn btn-sm btn-primary" onclick="importData({{ $apiSource->id }}, '{{ $apiSource->name }}')">
                                <i class="fas fa-download me-1"></i>Import Data Now
                            </button>
                        </div>
                    @endif
                </div>
            </div>
        </div>

        <!-- Sidebar -->
        <div class="col-lg-4">
            <!-- Quick Actions -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-success">
                        <i class="fas fa-bolt me-2"></i>Quick Actions
                    </h6>
                </div>
                <div class="card-body">
                    <div class="d-grid gap-2">
                        <button type="button" class="btn btn-success" onclick="importData({{ $apiSource->id }}, '{{ $apiSource->name }}')">
                            <i class="fas fa-download me-2"></i>Import Data Now
                        </button>
                        <a href="{{ route('api-sources.edit', $apiSource) }}" class="btn btn-primary">
                            <i class="fas fa-edit me-2"></i>Edit Source
                        </a>
                        <a href="{{ route('api-sources.logs', $apiSource) }}" class="btn btn-info">
                            <i class="fas fa-list me-2"></i>View All Logs
                        </a>
                        <a href="{{ route('api-sources.data', $apiSource) }}" class="btn btn-warning">
                            <i class="fas fa-table me-2"></i>View Data
                        </a>
                        <hr>
                        <button type="button" class="btn btn-outline-danger" onclick="deleteSource({{ $apiSource->id }}, '{{ $apiSource->name }}')">
                            <i class="fas fa-trash me-2"></i>Delete Source
                        </button>
                    </div>
                </div>
            </div>

            <!-- Statistics -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-info">
                        <i class="fas fa-chart-bar me-2"></i>Statistics
                    </h6>
                </div>
                <div class="card-body">
                    @php
                        $totalLogs = $apiSource->apiLogs()->count();
                        $successLogs = $apiSource->apiLogs()->where('status_code', '>=', 200)->where('status_code', '<', 300)->count();
                        $errorLogs = $apiSource->apiLogs()->where('status_code', '>=', 400)->count();
                        $totalRecords = $successLogs; // Use success count as proxy for records
                        $avgResponseTime = $apiSource->apiLogs()->where('status_code', '>=', 200)->where('status_code', '<', 300)->avg('response_time');
                    @endphp
                    
                    <div class="row text-center">
                        <div class="col-6 mb-3">
                            <div class="h4 text-primary mb-0">{{ $totalLogs }}</div>
                            <small class="text-muted">Total Requests</small>
                        </div>
                        <div class="col-6 mb-3">
                            <div class="h4 text-success mb-0">{{ $successLogs }}</div>
                            <small class="text-muted">Successful</small>
                        </div>
                        <div class="col-6 mb-3">
                            <div class="h4 text-danger mb-0">{{ $errorLogs }}</div>
                            <small class="text-muted">Failed</small>
                        </div>
                        <div class="col-6 mb-3">
                            <div class="h4 text-info mb-0">{{ number_format($totalRecords) }}</div>
                            <small class="text-muted">Records Imported</small>
                        </div>
                    </div>
                    
                    @if($avgResponseTime)
                        <div class="text-center mt-3">
                            <div class="small text-muted">Average Response Time</div>
                            <div class="h5 text-warning">{{ number_format($avgResponseTime, 2) }}s</div>
                        </div>
                    @endif
                    
                    @if($totalLogs > 0)
                        <div class="mt-3">
                            <div class="d-flex justify-content-between small text-muted mb-1">
                                <span>Success Rate</span>
                                <span>{{ number_format(($successLogs / $totalLogs) * 100, 1) }}%</span>
                            </div>
                            <div class="progress">
                                <div class="progress-bar bg-success" role="progressbar" 
                                     style="width: {{ ($successLogs / $totalLogs) * 100 }}%"></div>
                            </div>
                        </div>
                    @endif
                </div>
            </div>

            <!-- API Test -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-warning">
                        <i class="fas fa-flask me-2"></i>API Test
                    </h6>
                </div>
                <div class="card-body">
                    <p class="small text-muted mb-3">Test the API endpoint to check connectivity and response.</p>
                    <button type="button" class="btn btn-warning btn-sm w-100" onclick="testApi({{ $apiSource->id }})">
                        <i class="fas fa-play me-2"></i>Test API Connection
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1" role="dialog" aria-labelledby="deleteModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteModalLabel">Confirm Delete</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete this API source?</p>
                <p><strong id="deleteSourceName"></strong></p>
                <p class="text-danger">This action cannot be undone and will also delete all associated logs.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <form id="deleteForm" method="POST" style="display: inline;" onsubmit="handleDeleteSubmit(event)">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger" data-original-text="Delete">Delete</button>
                </form>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
function deleteSource(id, name) {
    document.getElementById('deleteSourceName').textContent = name;
    document.getElementById('deleteForm').action = `/api-sources/${id}`;
    
    const modal = new bootstrap.Modal(document.getElementById('deleteModal'));
    modal.show();
}

function importData(id, name) {
    if (confirm(`Import data from "${name}"?`)) {
        window.LoadingOverlay.show('Importing data...');
        
        fetch(`/api-sources/${id}/import`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            window.LoadingOverlay.hide();
            if (data.success) {
                window.Toast.success(data.message || 'Data imported successfully');
                setTimeout(() => location.reload(), 2000);
            } else {
                window.Toast.error(data.message || 'Import failed');
            }
        })
        .catch(error => {
            window.LoadingOverlay.hide();
            window.Toast.error('Import failed: ' + error.message);
        });
    }
}

function testApi(id) {
    window.LoadingOverlay.show('Testing API connection...');
    
    fetch(`/api-sources/${id}/test`, {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'Accept': 'application/json',
            'Content-Type': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        window.LoadingOverlay.hide();
        if (data.success) {
            window.Toast.success(`API test successful! Response: ${data.status_code} (${data.response_time}ms)`);
        } else {
            window.Toast.error(`API test failed: ${data.message}`);
        }
    })
    .catch(error => {
        window.LoadingOverlay.hide();
        window.Toast.error('API test failed: ' + error.message);
    });
}

function handleDeleteSubmit(event) {
    event.preventDefault();
    console.log('Delete form submitted');
    
    const form = event.target;
    const submitButton = form.querySelector('button[type="submit"]');
    const originalText = submitButton.getAttribute('data-original-text') || 'Delete';
    
    // Disable button and show loading state
    submitButton.disabled = true;
    submitButton.innerHTML = '<span class="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></span>Loading...';
    
    // Add timeout to prevent infinite loading
    const timeout = setTimeout(() => {
        submitButton.disabled = false;
        submitButton.innerHTML = originalText;
        alert('Request timed out. Please try again.');
    }, 30000); // 30 seconds timeout
    
    const formData = new FormData(form);
    
    console.log('Submitting to:', form.action);
    console.log('Form data:', Object.fromEntries(formData));
    
    fetch(form.action, {
        method: 'POST',
        body: formData,
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        }
    })
    .then(response => {
        console.log('Response status:', response.status);
        
        if (response.redirected) {
            window.location.href = response.url;
            return null;
        } else if (response.ok) {
            return response.text();
        } else {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
    })
    .then(data => {
        console.log('Response data:', data);
        
        // Clear timeout
        clearTimeout(timeout);
        
        // Close modal
        const modal = bootstrap.Modal.getInstance(document.getElementById('deleteModal'));
        if (modal) {
            modal.hide();
        }
        
        // Show success message and reload
        if (typeof window.Toast !== 'undefined') {
            window.Toast.success('API Source deleted successfully');
        } else {
            alert('API Source deleted successfully');
        }
        
        setTimeout(() => {
            location.reload();
        }, 1000);
    })
    .catch(error => {
        console.error('Delete error:', error);
        
        // Clear timeout
        clearTimeout(timeout);
        
        // Restore button state
        submitButton.disabled = false;
        submitButton.innerHTML = originalText;
        
        // Show error message
        if (typeof window.Toast !== 'undefined') {
            window.Toast.error('Error deleting API Source: ' + error.message);
        } else {
            alert('Error deleting API Source: ' + error.message);
        }
    });
}
</script>
@endpush
@endsection 