@extends('layouts.app')

@section('title', 'Create API Target')

@section('breadcrumb')
    <li class="breadcrumb-item">
        <a href="{{ route('api-targets.index') }}">API Targets</a>
    </li>
    <li class="breadcrumb-item active">Create</li>
@endsection

@section('content')
<div class="container-fluid">
    <!-- Alert Messages -->
    <div id="alert-container">
        @if(session('success'))
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="fas fa-check-circle me-2"></i>{{ session('success') }}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        @endif
        
        @if(session('csrf_fixed'))
            <div class="alert alert-info alert-dismissible fade show" role="alert">
                <i class="fas fa-shield-alt me-2"></i>
                <strong>CSRF Token Fixed!</strong> Form is now ready to submit.
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        @endif
        
        @if(session('error'))
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="fas fa-exclamation-circle me-2"></i>{{ session('error') }}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        @endif
        
        @if($errors->any())
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <strong>Please fix the following errors:</strong>
                <ul class="mb-0 mt-2">
                    @foreach($errors->all() as $error)
                        <li>{{ $error }}</li>
                    @endforeach
                </ul>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        @endif
        
        <!-- CSRF Error Quick Fix -->
        <div id="csrf-error-alert" class="alert alert-warning alert-dismissible fade" style="display: none;" role="alert">
            <i class="fas fa-exclamation-triangle me-2"></i>
            <strong>CSRF Token Error Detected!</strong> 
            <p class="mb-2">If you're getting cookie/CSRF errors, click the button below to fix:</p>
            <a href="/fix-csrf" class="btn btn-sm btn-warning me-2">
                <i class="fas fa-wrench me-1"></i>Auto-Fix CSRF
            </a>
            <a href="/csrf-fix.html" class="btn btn-sm btn-info">
                <i class="fas fa-tools me-1"></i>Manual Fix
            </a>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    </div>

    <div class="row">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h1 class="h3 mb-0 text-gray-800">
                    <i class="fas fa-plus me-2"></i>Create API Target
                </h1>
                <a href="{{ route('api-targets.index') }}" class="btn btn-secondary">
                    <i class="fas fa-arrow-left me-2"></i>Back to List
                </a>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-lg-8">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-bullseye me-2"></i>API Target Information
                    </h6>
                </div>
                <div class="card-body">
                    <form action="{{ route('api-targets.store') }}" method="POST" id="createForm" novalidate>
                        @csrf
                        <input type="hidden" name="_token" value="{{ csrf_token() }}">
                        
                        <!-- Basic Information -->
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="name" class="form-label">Name <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control @error('name') is-invalid @enderror" 
                                           id="name" name="name" value="{{ old('name') }}" required>
                                    @error('name')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="is_active" class="form-label">Status</label>
                                    <div class="form-check">
                                        <input type="hidden" name="is_active" value="0">
                                        <input type="checkbox" class="form-check-input" id="is_active" name="is_active" 
                                               value="1" {{ old('is_active') ? 'checked' : '' }}>
                                        <label class="form-check-label" for="is_active">
                                            Active
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="form-group mb-3">
                            <label for="description" class="form-label">Description</label>
                            <textarea class="form-control @error('description') is-invalid @enderror" 
                                      id="description" name="description" rows="3" 
                                      placeholder="Brief description of this API target">{{ old('description') }}</textarea>
                            @error('description')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- API Configuration -->
                        <hr class="my-4">
                        <h5 class="mb-3">
                            <i class="fas fa-cog me-2"></i>API Configuration
                        </h5>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="base_url" class="form-label">Base URL <span class="text-danger">*</span></label>
                                    <input type="url" class="form-control @error('base_url') is-invalid @enderror" 
                                           id="base_url" name="base_url" value="{{ old('base_url') }}" 
                                           placeholder="https://api.example.com" required>
                                    @error('base_url')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="endpoint" class="form-label">Endpoint <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control @error('endpoint') is-invalid @enderror" 
                                           id="endpoint" name="endpoint" value="{{ old('endpoint') }}" 
                                           placeholder="/data" required>
                                    @error('endpoint')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="method" class="form-label">HTTP Method <span class="text-danger">*</span></label>
                                    <select class="form-control @error('method') is-invalid @enderror" id="method" name="method" required>
                                        <option value="">Select Method</option>
                                        <option value="GET" {{ old('method') == 'GET' ? 'selected' : '' }}>GET</option>
                                        <option value="POST" {{ old('method') == 'POST' ? 'selected' : '' }}>POST</option>
                                        <option value="PUT" {{ old('method') == 'PUT' ? 'selected' : '' }}>PUT</option>
                                        <option value="DELETE" {{ old('method') == 'DELETE' ? 'selected' : '' }}>DELETE</option>
                                    </select>
                                    @error('method')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="source_table" class="form-label">Source Table <span class="text-danger">*</span></label>
                                    <select class="form-control @error('source_table') is-invalid @enderror" 
                                            id="source_table" name="source_table" required>
                                        <option value="">Select Source Table</option>
                                        @foreach($tables as $table)
                                            <option value="{{ $table }}" {{ old('source_table') == $table ? 'selected' : '' }}>
                                                {{ $table }}
                                            </option>
                                        @endforeach
                                    </select>
                                    @error('source_table')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <!-- Authentication -->
                        <hr class="my-4">
                        <h5 class="mb-3">
                            <i class="fas fa-shield-alt me-2"></i>Authentication
                        </h5>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="auth_type" class="form-label">Authentication Type <span class="text-danger">*</span></label>
                                    <select class="form-control @error('auth_type') is-invalid @enderror" 
                                            id="auth_type" name="auth_type" required>
                                        <option value="">Select Auth Type</option>
                                        <option value="none" {{ old('auth_type') == 'none' ? 'selected' : '' }}>None</option>
                                        <option value="api_key" {{ old('auth_type') == 'api_key' ? 'selected' : '' }}>API Key</option>
                                        <option value="bearer_token" {{ old('auth_type') == 'bearer_token' ? 'selected' : '' }}>Bearer Token</option>
                                        <option value="basic_auth" {{ old('auth_type') == 'basic_auth' ? 'selected' : '' }}>Basic Auth</option>
                                    </select>
                                    @error('auth_type')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3" id="auth_token_group" style="display: none;">
                                    <label for="auth_token" class="form-label">Authentication Token</label>
                                    <input type="text" class="form-control @error('auth_token') is-invalid @enderror" 
                                           id="auth_token" name="auth_token" value="{{ old('auth_token') }}" 
                                           placeholder="Enter your API key or token">
                                    @error('auth_token')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row" id="basic_auth_group" style="display: none;">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="auth_username" class="form-label">Username</label>
                                    <input type="text" class="form-control @error('auth_username') is-invalid @enderror" 
                                           id="auth_username" name="auth_username" value="{{ old('auth_username') }}" 
                                           placeholder="Enter username">
                                    @error('auth_username')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="auth_password" class="form-label">Password</label>
                                    <div class="input-group">
                                        <input type="password" class="form-control @error('auth_password') is-invalid @enderror" id="auth_password" name="auth_password" value="{{ old('auth_password') }}" placeholder="Enter password">
                                        <button class="btn btn-outline-secondary" type="button" id="togglePassword"><i class="fas fa-eye"></i></button>
                                    </div>
                                    @error('auth_password')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <!-- Scheduling -->
                        <hr class="my-4">
                        <h5 class="mb-3">
                            <i class="fas fa-clock me-2"></i>Scheduling
                        </h5>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="schedule_type" class="form-label">Schedule Type <span class="text-danger">*</span></label>
                                    <select class="form-control @error('schedule_type') is-invalid @enderror" 
                                            id="schedule_type" name="schedule_type" required>
                                        <option value="">Select Schedule</option>
                                        <option value="manual" {{ old('schedule_type') == 'manual' ? 'selected' : '' }}>Manual</option>
                                        <option value="hourly" {{ old('schedule_type') == 'hourly' ? 'selected' : '' }}>Hourly</option>
                                        <option value="daily" {{ old('schedule_type') == 'daily' ? 'selected' : '' }}>Daily</option>
                                        <option value="weekly" {{ old('schedule_type') == 'weekly' ? 'selected' : '' }}>Weekly</option>
                                    </select>
                                    @error('schedule_type')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3" id="schedule_frequency_group" style="display: none;">
                                    <label for="schedule_frequency" class="form-label">Frequency</label>
                                    <input type="number" class="form-control @error('schedule_frequency') is-invalid @enderror" 
                                           id="schedule_frequency" name="schedule_frequency" value="{{ old('schedule_frequency') }}" 
                                           min="1" placeholder="e.g., 2 for every 2 hours/days">
                                    <small class="form-text text-muted" id="frequency_help">Leave empty for default (every 1 unit)</small>
                                    @error('schedule_frequency')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="form-group mt-4">
                            <div class="row">
                                <div class="col-12 mb-3">
                                    <button type="submit" class="btn btn-primary btn-lg me-2" id="altSubmitBtn">
                                        <i class="fas fa-rocket me-2"></i>SUBMIT NOW
                                    </button>
                                </div>
                                <div class="col-12 mb-3">
                                    <button type="button" class="btn btn-info me-2" id="simpleSubmitBtn">
                                        <i class="fas fa-paper-plane me-2"></i>SIMPLE SUBMIT
                                    </button>
                                    <a href="{{ route('api-targets.index') }}" class="btn btn-secondary me-2">
                                        <i class="fas fa-times me-2"></i>Cancel
                                    </a>
                                    <button type="button" class="btn btn-warning me-2" id="testConnectionBtn">
                                        <i class="fas fa-plug me-2"></i>Test Connection
                                    </button>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-lg-4">
            <!-- Help Card -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-info">
                        <i class="fas fa-info-circle me-2"></i>Help & Tips
                    </h6>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <h6 class="text-primary">Base URL</h6>
                        <p class="small text-muted">Enter the base URL of the API you want to send data to (e.g., https://api.example.com).</p>
                    </div>
                    
                    <div class="mb-3">
                        <h6 class="text-primary">Endpoint</h6>
                        <p class="small text-muted">The specific endpoint path where data will be sent (e.g., /webhook or /data).</p>
                    </div>
                    
                    <div class="mb-3">
                        <h6 class="text-primary">Source Table</h6>
                        <p class="small text-muted">Select the database table from which data will be sent to this API target. Only available tables are shown in the dropdown.</p>
                    </div>
                    
                    <div class="mb-3">
                        <h6 class="text-primary">Authentication</h6>
                        <p class="small text-muted">If the target API requires authentication, select the appropriate type and provide the token.</p>
                    </div>
                    
                    <div class="mb-3">
                        <h6 class="text-primary">Scheduling</h6>
                        <p class="small text-muted">Set how often data should be sent automatically. Choose "Manual" for on-demand sending.</p>
                    </div>
                </div>
            </div>

            <!-- Preview Card -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-success">
                        <i class="fas fa-eye me-2"></i>Preview
                    </h6>
                </div>
                <div class="card-body">
                    <div class="mb-2">
                        <strong>Full URL:</strong> <span id="preview_full_url" class="text-muted">Not set</span>
                    </div>
                    <div class="mb-2">
                        <strong>Method:</strong> <span id="preview_method" class="text-muted">Not set</span>
                    </div>
                    <div class="mb-2">
                        <strong>Source:</strong> <span id="preview_source" class="text-muted">Not set</span>
                    </div>
                    <div class="mb-2">
                        <strong>Auth:</strong> <span id="preview_auth" class="text-muted">Not set</span>
                    </div>
                    <div class="mb-2">
                        <strong>Schedule:</strong> <span id="preview_schedule" class="text-muted">Not set</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
// ============================================================================
// CLEAN JAVASCRIPT - NO ICONS, EVENT LISTENERS ONLY  
// ============================================================================

function showAlert(message, type = 'info') {
    const alertContainer = document.getElementById('alert-container');
    if (!alertContainer) return;
    
    const alert = document.createElement('div');
    alert.className = `alert alert-${type} alert-dismissible fade show`;
    alert.innerHTML = `
        <strong>${type === 'success' ? 'Success!' : type === 'danger' ? 'Error!' : 'Info:'}</strong> ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    alertContainer.appendChild(alert);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (alert.parentNode) {
            alert.remove();
        }
    }, 5000);
}

// ============================================================================
// MAIN FUNCTIONS
// ============================================================================

function initializeEventListeners() {
    // Source table change listener
    const sourceTable = document.getElementById('source_table');
    if (sourceTable) {
        sourceTable.addEventListener('change', updatePreview);
    }
}

function toggleAuthToken() {
    const authType = document.getElementById('auth_type');
    const authTokenGroup = document.getElementById('auth_token_group');
    const authTokenField = document.getElementById('auth_token');
    const basicAuthGroup = document.getElementById('basic_auth_group');
    const authUsernameField = document.getElementById('auth_username');
    const authPasswordField = document.getElementById('auth_password');
    
    if (!authType) return;
    
    // Hide all auth groups first
    if (authTokenGroup) authTokenGroup.style.display = 'none';
    if (basicAuthGroup) basicAuthGroup.style.display = 'none';
    
    // Reset required fields
    if (authTokenField) {
        authTokenField.required = false;
        authTokenField.value = '';
    }
    if (authUsernameField) {
        authUsernameField.required = false;
        authUsernameField.value = '';
    }
    if (authPasswordField) {
        authPasswordField.required = false;
        authPasswordField.value = '';
    }
    
    // Show appropriate group based on auth type
    if (authType.value === 'api_key' || authType.value === 'bearer_token') {
        if (authTokenGroup) {
            authTokenGroup.style.display = 'block';
            if (authTokenField) authTokenField.required = true;
        }
    } else if (authType.value === 'basic_auth') {
        if (basicAuthGroup) {
            basicAuthGroup.style.display = 'block';
            if (authUsernameField) authUsernameField.required = true;
            if (authPasswordField) authPasswordField.required = true;
        }
    }
    
    updatePreview();
}

function toggleScheduleFrequency() {
    const scheduleType = document.getElementById('schedule_type');
    const frequencyGroup = document.getElementById('schedule_frequency_group');
    const frequencyHelp = document.getElementById('frequency_help');
    const frequencyField = document.getElementById('schedule_frequency');
    
    if (!scheduleType || !frequencyGroup) return;
    
    if (scheduleType.value === 'hourly' || scheduleType.value === 'daily' || scheduleType.value === 'weekly') {
        frequencyGroup.style.display = 'block';
        if (frequencyHelp) {
            if (scheduleType.value === 'hourly') {
                frequencyHelp.textContent = 'e.g., 2 for every 2 hours';
            } else if (scheduleType.value === 'daily') {
                frequencyHelp.textContent = 'e.g., 3 for every 3 days';
            } else {
                frequencyHelp.textContent = 'e.g., 2 for every 2 weeks';
            }
        }
    } else {
        frequencyGroup.style.display = 'none';
        if (frequencyField) {
            frequencyField.value = '';
        }
    }
    
    updatePreview();
}

function updatePreview() {
    const baseUrl = document.getElementById('base_url')?.value || '';
    const endpoint = document.getElementById('endpoint')?.value || '';
    const method = document.getElementById('method')?.value || 'Not set';
    const sourceTable = document.getElementById('source_table')?.value || 'Not set';
    const authType = document.getElementById('auth_type')?.value || 'Not set';
    const scheduleType = document.getElementById('schedule_type')?.value || 'Not set';
    const scheduleFreq = document.getElementById('schedule_frequency')?.value;
    
    // Create full URL
    let fullUrl = 'Not set';
    if (baseUrl) {
        fullUrl = baseUrl;
        if (endpoint) {
            fullUrl += (endpoint.startsWith('/') ? '' : '/') + endpoint;
        }
    }
    
    // Update preview elements safely
    const previewFullUrl = document.getElementById('preview_full_url');
    const previewMethod = document.getElementById('preview_method');
    const previewSource = document.getElementById('preview_source');
    const previewAuth = document.getElementById('preview_auth');
    const previewSchedule = document.getElementById('preview_schedule');
    
    if (previewFullUrl) previewFullUrl.textContent = fullUrl.length > 40 ? fullUrl.substring(0, 40) + '...' : fullUrl;
    if (previewMethod) previewMethod.textContent = method;
    if (previewSource) previewSource.textContent = sourceTable;
    if (previewAuth) previewAuth.textContent = authType === 'none' ? 'None' : authType.replace('_', ' ').toUpperCase();
    
    if (previewSchedule) {
        let scheduleText = scheduleType;
        if (scheduleFreq && scheduleType !== 'manual') {
            scheduleText += ` (every ${scheduleFreq})`;
        }
        previewSchedule.textContent = scheduleText;
    }
}

function testConnection() {
    const baseUrl = document.getElementById('base_url')?.value;
    const testBtn = document.getElementById('testConnectionBtn');
    
    if (!baseUrl) {
        showAlert('Please enter a base URL first', 'warning');
        return;
    }
    
    if (testBtn) {
        testBtn.disabled = true;
        testBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Testing...';
    }
    
    fetch('{{ route("api-targets.test-connection") }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
            'Accept': 'application/json'
        },
        body: JSON.stringify({ base_url: baseUrl })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('Connection successful!', 'success');
        } else {
            showAlert('Connection failed: ' + data.message, 'danger');
        }
    })
    .catch(error => {
        showAlert('Connection test failed: ' + error.message, 'danger');
    })
    .finally(() => {
        if (testBtn) {
            testBtn.disabled = false;
            testBtn.innerHTML = '<i class="fas fa-plug me-2"></i>Test Connection';
        }
    });
}

function submitFormDirectly() {
    console.log('Alternative submit triggered');
    
    const form = document.getElementById('createForm');
    const saveBtn = document.getElementById('saveBtn');
    const altSubmitBtn = document.getElementById('altSubmitBtn');
    
    if (!form) return;
    
    // Validate required fields manually
    const requiredFields = form.querySelectorAll('[required]');
    let isValid = true;
    let firstInvalidField = null;
    
    requiredFields.forEach(field => {
        if (!field.value.trim()) {
            isValid = false;
            field.classList.add('is-invalid');
            if (!firstInvalidField) {
                firstInvalidField = field;
            }
        } else {
            field.classList.remove('is-invalid');
        }
    });
    
    if (!isValid) {
        showAlert('Please fill in all required fields!', 'danger');
        if (firstInvalidField) {
            firstInvalidField.focus();
        }
        return;
    }
    
    // Show loading state
    if (altSubmitBtn) {
        altSubmitBtn.disabled = true;
        altSubmitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>SUBMITTING...';
    }
    if (saveBtn) {
        saveBtn.disabled = true;
    }
    
    showAlert('Submitting form data...', 'info');
    
    // Submit using fetch API with direct POST
    const formData = new FormData(form);
    
    fetch(form.action, {
        method: 'POST',
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json, text/html',
        }
    })
    .then(response => {
        console.log('Response received:', response.status);
        
        if (response.ok) {
            showAlert('Form submitted successfully! Redirecting...', 'success');
            setTimeout(() => {
                window.location.href = '/api-targets';
            }, 1500);
        } else {
            return response.text().then(text => {
                showAlert('Submission failed: ' + response.statusText, 'danger');
                console.error('Error response:', text);
            });
        }
    })
    .catch(error => {
        console.error('Submit error:', error);
        
        // Check for CSRF-related errors
        if (error.message && (
            error.message.includes('CSRF') || 
            error.message.includes('Cookie') || 
            error.message.includes('array given') ||
            error.message.includes('token mismatch')
        )) {
            const csrfAlert = document.getElementById('csrf-error-alert');
            if (csrfAlert) {
                csrfAlert.style.display = 'block';
                csrfAlert.classList.add('show');
            }
            showAlert('CSRF Token Error detected. Please use the Auto-Fix button above.', 'warning');
        } else {
            showAlert('Network error: ' + error.message, 'danger');
        }
    })
    .finally(() => {
        // Re-enable buttons
        if (altSubmitBtn) {
            altSubmitBtn.disabled = false;
            altSubmitBtn.innerHTML = '<i class="fas fa-rocket me-2"></i>SUBMIT NOW';
        }
        if (saveBtn) {
            saveBtn.disabled = false;
        }
    });
}

function submitSimple() {
    console.log('Simple submit triggered');
    
    const form = document.getElementById('createForm');
    if (!form) return;
    
    // Simple validation check
    const name = document.getElementById('name')?.value;
    const baseUrl = document.getElementById('base_url')?.value;
    const endpoint = document.getElementById('endpoint')?.value;
    const method = document.getElementById('method')?.value;
    const sourceTable = document.getElementById('source_table')?.value;
    const authType = document.getElementById('auth_type')?.value;
    const scheduleType = document.getElementById('schedule_type')?.value;
    
    if (!name || !baseUrl || !endpoint || !method || !sourceTable || !authType || !scheduleType) {
        alert('Please fill in all required fields!');
        return;
    }
    
    // Show confirmation
    const confirmed = confirm(`Ready to save this API Target?\n\nName: ${name}\nURL: ${baseUrl}${endpoint}\nMethod: ${method}\nSource Table: ${sourceTable}\n\nClick OK to proceed.`);
    
    if (confirmed) {
        // Create a simple loading message
        const simpleBtn = document.getElementById('simpleSubmitBtn');
        if (simpleBtn) {
            simpleBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>SUBMITTING...';
            simpleBtn.disabled = true;
        }
        
        // Add a hidden field to identify this submission method
        const hiddenField = document.createElement('input');
        hiddenField.type = 'hidden';
        hiddenField.name = 'submit_method';
        hiddenField.value = 'simple';
        form.appendChild(hiddenField);
        
        // Submit the form normally
        form.submit();
    }
}

// ============================================================================
// EVENT LISTENERS - NO INLINE HTML EVENTS
// ============================================================================

document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM loaded, setting up event listeners...');
    
    // Auth type change listener
    const authType = document.getElementById('auth_type');
    if (authType) {
        authType.addEventListener('change', toggleAuthToken);
    }
    
    // Schedule type change listener  
    const scheduleType = document.getElementById('schedule_type');
    if (scheduleType) {
        scheduleType.addEventListener('change', toggleScheduleFrequency);
    }
    
    // Alternative submit button
    const altSubmitBtn = document.getElementById('altSubmitBtn');
    if (altSubmitBtn) {
        altSubmitBtn.addEventListener('click', submitFormDirectly);
    }
    
    // Simple submit button
    const simpleSubmitBtn = document.getElementById('simpleSubmitBtn');
    if (simpleSubmitBtn) {
        simpleSubmitBtn.addEventListener('click', submitSimple);
    }
    
    // Test connection button
    const testConnectionBtn = document.getElementById('testConnectionBtn');
    if (testConnectionBtn) {
        testConnectionBtn.addEventListener('click', testConnection);
    }
    
    // Preview update listeners
    const fieldsToWatch = ['base_url', 'endpoint', 'method', 'source_table', 'auth_type', 'schedule_type', 'schedule_frequency'];
    fieldsToWatch.forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (field) {
            field.addEventListener('input', updatePreview);
            field.addEventListener('change', updatePreview);
        }
    });
    
    // Primary form submit handler
    const form = document.getElementById('createForm');
    const saveBtn = document.getElementById('saveBtn');
    
    if (form && saveBtn) {
        form.addEventListener('submit', function(e) {
            console.log('Form submitted via primary button');
            
            // Simple validation
            const requiredFields = form.querySelectorAll('[required]');
            let hasErrors = false;
            
            requiredFields.forEach(field => {
                if (!field.value.trim()) {
                    field.classList.add('is-invalid');
                    hasErrors = true;
                } else {
                    field.classList.remove('is-invalid');
                }
            });
            
            if (hasErrors) {
                e.preventDefault();
                showAlert('Please fill in all required fields!', 'danger');
                return false;
            }
            
            // Show loading state
            saveBtn.disabled = true;
            saveBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>SAVING...';
            
            // Re-enable button after delay if submission fails
            setTimeout(() => {
                if (saveBtn.disabled) {
                    saveBtn.disabled = false;
                    saveBtn.innerHTML = '<i class="fas fa-save me-2"></i>SAVE API TARGET';
                }
            }, 15000);
        });
    }
    
    // Initialize state
    initializeEventListeners();
    toggleAuthToken();
    toggleScheduleFrequency();
    updatePreview();
    
    // Show success message if redirected from successful creation
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.get('success')) {
        showAlert('API Target created successfully!', 'success');
    }
    
    // Global error handler for CSRF-related errors
    window.addEventListener('error', function(e) {
        if (e.message && (
            e.message.includes('CSRF') || 
            e.message.includes('Cookie') || 
            e.message.includes('array given') ||
            e.message.includes('token mismatch')
        )) {
            const csrfAlert = document.getElementById('csrf-error-alert');
            if (csrfAlert) {
                csrfAlert.style.display = 'block';
                csrfAlert.classList.add('show');
            }
        }
    });
    
    console.log('All event listeners set up successfully!');
});

document.getElementById('togglePassword')?.addEventListener('click', function() {
    const pwd = document.getElementById('auth_password');
    const icon = this.querySelector('i');
    if (pwd.type === 'password') {
        pwd.type = 'text';
        icon.classList.remove('fa-eye');
        icon.classList.add('fa-eye-slash');
    } else {
        pwd.type = 'password';
        icon.classList.remove('fa-eye-slash');
        icon.classList.add('fa-eye');
    }
});
</script>
@endpush
@endsection 