@extends('layouts.app')

@section('title', 'API Target Details')

@section('breadcrumb')
    <li class="breadcrumb-item">
        <a href="{{ route('api-targets.index') }}">API Targets</a>
    </li>
    <li class="breadcrumb-item active">{{ $apiTarget->name }}</li>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h1 class="h3 mb-0 text-gray-800">
                    <i class="fas fa-bullseye me-2"></i>{{ $apiTarget->name }}
                </h1>
                <div class="btn-group">
                    <a href="{{ route('api-targets.documentation', $apiTarget) }}" class="btn btn-info">
                        <i class="fas fa-book me-2"></i>Documentation
                    </a>
                    <a href="{{ route('api-targets.edit', $apiTarget) }}" class="btn btn-primary">
                        <i class="fas fa-edit me-2"></i>Edit
                    </a>
                    <button type="button" class="btn btn-success" onclick="sendData({{ $apiTarget->id }}, '{{ $apiTarget->name }}')">
                        <i class="fas fa-upload me-2"></i>Send Data
                    </button>
                    <a href="{{ route('api-targets.index') }}" class="btn btn-secondary">
                        <i class="fas fa-arrow-left me-2"></i>Back to List
                    </a>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <!-- Main Information -->
        <div class="col-lg-8">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-info-circle me-2"></i>API Target Information
                    </h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Name:</label>
                                <p class="text-gray-800">{{ $apiTarget->name }}</p>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Status:</label>
                                <p>
                                    <span class="badge bg-{{ $apiTarget->is_active ? 'success' : 'secondary' }}">
                                        {{ $apiTarget->is_active ? 'Active' : 'Inactive' }}
                                    </span>
                                </p>
                            </div>
                        </div>
                    </div>

                    @if($apiTarget->description)
                        <div class="mb-3">
                            <label class="form-label font-weight-bold">Description:</label>
                            <p class="text-gray-700">{{ $apiTarget->description }}</p>
                        </div>
                    @endif

                    <hr class="my-4">
                    <h5 class="mb-3">
                        <i class="fas fa-cog me-2"></i>API Configuration
                    </h5>

                    <div class="row">
                        <div class="col-md-12">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Full URL:</label>
                                <p>
                                    <a href="{{ $apiTarget->base_url . ($apiTarget->endpoint ?? '') }}" target="_blank" class="text-decoration-none">
                                        {{ $apiTarget->base_url . ($apiTarget->endpoint ?? '') }}
                                        <i class="fas fa-external-link-alt ms-2 text-gray-400"></i>
                                    </a>
                                </p>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">HTTP Method:</label>
                                <p>
                                    <span class="badge bg-{{ $apiTarget->method == 'GET' ? 'success' : ($apiTarget->method == 'POST' ? 'primary' : 'warning') }}">
                                        {{ $apiTarget->method }}
                                    </span>
                                </p>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Source Table:</label>
                                <p><code class="bg-light p-2 rounded">{{ $apiTarget->source_table }}</code></p>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Authentication:</label>
                                <p>
                                    <span class="badge bg-{{ $apiTarget->auth_type == 'none' ? 'warning' : 'success' }}">
                                        {{ $apiTarget->auth_type == 'none' ? 'None' : ucfirst(str_replace('_', ' ', $apiTarget->auth_type)) }}
                                    </span>
                                </p>
                            </div>
                        </div>
                    </div>

                    <hr class="my-4">
                    <h5 class="mb-3">
                        <i class="fas fa-clock me-2"></i>Scheduling
                    </h5>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Schedule Type:</label>
                                <p>
                                    <span class="badge bg-info">{{ ucfirst($apiTarget->schedule_type) }}</span>
                                    @if($apiTarget->schedule_frequency && $apiTarget->schedule_type !== 'manual')
                                        <small class="text-muted ms-2">Every {{ $apiTarget->schedule_frequency }} {{ $apiTarget->schedule_type == 'hourly' ? 'hour(s)' : ($apiTarget->schedule_type == 'daily' ? 'day(s)' : 'week(s)') }}</small>
                                    @endif
                                </p>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Last Sent:</label>
                                <p class="text-gray-700">
                                    @if($apiTarget->last_sent_at)
                                        {{ $apiTarget->last_sent_at->format('d M Y, H:i:s') }}
                                        <small class="text-muted">({{ $apiTarget->last_sent_at->diffForHumans() }})</small>
                                    @else
                                        <span class="text-muted">Never</span>
                                    @endif
                                </p>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Created:</label>
                                <p class="text-gray-700">
                                    {{ $apiTarget->created_at->format('d M Y, H:i:s') }}
                                    <small class="text-muted">({{ $apiTarget->created_at->diffForHumans() }})</small>
                                </p>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label font-weight-bold">Last Updated:</label>
                                <p class="text-gray-700">
                                    {{ $apiTarget->updated_at->format('d M Y, H:i:s') }}
                                    <small class="text-muted">({{ $apiTarget->updated_at->diffForHumans() }})</small>
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Recent Logs -->
            <div class="card shadow mb-4">
                <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-list me-2"></i>Recent Activity
                    </h6>
                    <a href="{{ route('api-targets.logs', $apiTarget) }}" class="btn btn-sm btn-outline-primary">
                        View All Logs
                    </a>
                </div>
                <div class="card-body">
                    @php
                        $recentLogs = $apiTarget->sendLogs()->latest()->limit(5)->get();
                    @endphp
                    
                    @if($recentLogs->count() > 0)
                        @foreach($recentLogs as $log)
                            <div class="d-flex align-items-center mb-3 pb-3 border-bottom">
                                <div class="me-3">
                                    <div class="icon-circle bg-{{ $log->success ? 'success' : 'danger' }}">
                                        <i class="fas fa-{{ $log->success ? 'check' : 'times' }} text-white"></i>
                                    </div>
                                </div>
                                <div class="flex-grow-1">
                                    <div class="d-flex justify-content-between align-items-start">
                                        <div>
                                            <div class="font-weight-bold">
                                                Status: {{ $log->status_code }} {{ $log->error_message ?: 'Success' }}
                                            </div>
                                            <div class="small text-gray-500">
                                                {{ $log->sent_at ? $log->sent_at->format('d M Y, H:i:s') : $log->created_at->format('d M Y, H:i:s') }}
                                                ({{ $log->sent_at ? $log->sent_at->diffForHumans() : $log->created_at->diffForHumans() }})
                                                @if($log->records_sent)
                                                    • {{ $log->records_sent }} records sent
                                                @endif
                                                @if($log->response_time)
                                                    • {{ $log->response_time }}ms
                                                @endif
                                            </div>
                                        </div>
                                        <div>
                                            <button type="button" class="btn btn-sm btn-outline-info" 
                                                    onclick="viewLogDetails({{ $log->id }})">
                                                <i class="fas fa-eye"></i>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    @else
                        <div class="text-center text-gray-500 py-3">
                            <i class="fas fa-info-circle me-2"></i>No recent activity
                        </div>
                    @endif
                </div>
            </div>
        </div>

        <div class="col-lg-4">
            <!-- Statistics Card -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-chart-bar me-2"></i>Statistics
                    </h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-6 mb-3">
                            <div class="text-center">
                                <div class="h4 mb-0 text-gray-800">{{ $apiTarget->sendLogs()->count() }}</div>
                                <div class="small text-gray-500">Total Requests</div>
                            </div>
                        </div>
                        <div class="col-6 mb-3">
                            <div class="text-center">
                                <div class="h4 mb-0 text-success">
                                    {{ $apiTarget->sendLogs()->where('success', true)->count() }}
                                </div>
                                <div class="small text-gray-500">Successful</div>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="text-center">
                                <div class="h4 mb-0 text-danger">
                                    {{ $apiTarget->sendLogs()->where('success', false)->count() }}
                                </div>
                                <div class="small text-gray-500">Failed</div>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="text-center">
                                <div class="h4 mb-0 text-warning">
                                    {{ $apiTarget->sendLogs()->where('retry_count', '>', 0)->count() }}
                                </div>
                                <div class="small text-gray-500">Retried</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-bolt me-2"></i>Quick Actions
                    </h6>
                </div>
                <div class="card-body">
                    <div class="list-group">
                        <button type="button" class="list-group-item list-group-item-action" onclick="testConnection()">
                            <i class="fas fa-plug me-2"></i>Test Connection
                        </button>
                        <button type="button" class="list-group-item list-group-item-action" onclick="sendData({{ $apiTarget->id }}, '{{ $apiTarget->name }}')">
                            <i class="fas fa-upload me-2"></i>Send Data Now
                        </button>
                        <a href="{{ route('api-targets.logs', $apiTarget) }}" class="list-group-item list-group-item-action">
                            <i class="fas fa-history me-2"></i>View All Logs
                        </a>
                        <a href="{{ route('api-targets.edit', $apiTarget) }}" class="list-group-item list-group-item-action">
                            <i class="fas fa-edit me-2"></i>Edit Configuration
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Log Details Modal -->
<div class="modal fade" id="logDetailsModal" tabindex="-1" aria-labelledby="logDetailsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="logDetailsModalLabel">Log Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div id="logDetailsContent">
                    Loading...
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
function testConnection() {
    const baseUrl = '{{ $apiTarget->base_url }}';
    const testBtn = document.querySelector('.list-group-item:first-child');
    
    if (!baseUrl) {
        showAlert('No base URL configured', 'warning');
        return;
    }
    
    if (testBtn) {
        testBtn.disabled = true;
        testBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Testing...';
    }
    
    fetch('{{ route("api-targets.test-connection") }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
            'Accept': 'application/json'
        },
        body: JSON.stringify({ base_url: baseUrl })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('Connection successful!', 'success');
        } else {
            showAlert('Connection failed: ' + data.message, 'danger');
        }
    })
    .catch(error => {
        showAlert('Connection test failed: ' + error.message, 'danger');
    })
    .finally(() => {
        if (testBtn) {
            testBtn.disabled = false;
            testBtn.innerHTML = '<i class="fas fa-plug me-2"></i>Test Connection';
        }
    });
}

function sendData(id, name) {
    if (confirm(`Are you sure you want to send data to ${name} now?`)) {
        const sendBtn = document.querySelector('.btn-success');
        if (sendBtn) {
            sendBtn.disabled = true;
            sendBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Sending...';
        }
        
        fetch(`/api-targets/${id}/send`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
                'Accept': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('Data sent successfully!', 'success');
                setTimeout(() => location.reload(), 1500);
            } else {
                showAlert('Failed to send data: ' + data.message, 'danger');
            }
        })
        .catch(error => {
            showAlert('Error sending data: ' + error.message, 'danger');
        })
        .finally(() => {
            if (sendBtn) {
                sendBtn.disabled = false;
                sendBtn.innerHTML = '<i class="fas fa-upload me-2"></i>Send Data';
            }
        });
    }
}

function viewLogDetails(logId) {
    const modal = new bootstrap.Modal(document.getElementById('logDetailsModal'));
    const content = document.getElementById('logDetailsContent');
    
    content.innerHTML = '<div class="text-center"><i class="fas fa-spinner fa-spin me-2"></i>Loading log details...</div>';
    modal.show();
    
    fetch(`/api-targets/logs/${logId}`)
        .then(response => response.json())
        .then(data => {
            content.innerHTML = `
                <div class="mb-3">
                    <h6 class="font-weight-bold">Request Details</h6>
                    <p class="mb-1"><strong>URL:</strong> ${data.url}</p>
                    <p class="mb-1"><strong>Method:</strong> ${data.method}</p>
                    <p class="mb-1"><strong>Status:</strong> ${data.status_code} ${data.error_message || 'Success'}</p>
                    <p class="mb-1"><strong>Time:</strong> ${data.sent_at}</p>
                    <p class="mb-1"><strong>Records Sent:</strong> ${data.records_sent}</p>
                    <p class="mb-1"><strong>Response Time:</strong> ${data.response_time}ms</p>
                    ${data.retry_count > 0 ? `<p class="mb-1"><strong>Retries:</strong> ${data.retry_count}</p>` : ''}
                </div>
                <div class="mb-3">
                    <h6 class="font-weight-bold">Request Payload</h6>
                    <pre class="bg-light p-2 rounded"><code>${JSON.stringify(data.payload, null, 2)}</code></pre>
                </div>
                <div>
                    <h6 class="font-weight-bold">Response Body</h6>
                    <pre class="bg-light p-2 rounded"><code>${JSON.stringify(data.response_body, null, 2)}</code></pre>
                </div>
            `;
        })
        .catch(error => {
            content.innerHTML = `<div class="alert alert-danger">Error loading log details: ${error.message}</div>`;
        });
}

function showAlert(message, type = 'info') {
    const alertContainer = document.createElement('div');
    alertContainer.className = `alert alert-${type} alert-dismissible fade show`;
    alertContainer.innerHTML = `
        <strong>${type === 'success' ? 'Success!' : type === 'danger' ? 'Error!' : 'Info:'}</strong> ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.querySelector('.container-fluid').insertBefore(alertContainer, document.querySelector('.row'));
    
    setTimeout(() => {
        alertContainer.remove();
    }, 5000);
}
</script>
@endpush

@section('styles')
<style>
/* Badge Colors */
.badge.bg-success {
    color: #2b4a34 !important;
    background-color: #9de3bf !important;
}

.badge.bg-primary {
    color: #1a365d !important;
    background-color: #9ec5fe !important;
}

.badge.bg-info {
    color: #1e4e8c !important;
    background-color: #a5d7f7 !important;
}

.badge.bg-warning {
    color: #603b00 !important;
    background-color: #ffd699 !important;
}

.badge.bg-secondary {
    color: #383d41 !important;
    background-color: #e2e6ea !important;
}

/* Text Colors */
.text-success {
    color: #2b4a34 !important;
}

.text-primary {
    color: #1a365d !important;
}

.text-info {
    color: #1e4e8c !important;
}

.text-warning {
    color: #603b00 !important;
}

.text-secondary {
    color: #383d41 !important;
}

.text-gray-300 {
    color: #dddfeb !important;
}

.text-gray-400 {
    color: #858796 !important;
}

.text-gray-500 {
    color: #6c757d !important;
}

.text-gray-700 {
    color: #5a5c69 !important;
}

.text-gray-800 {
    color: #5a5c69 !important;
}

/* Card Styles */
.card {
    border: none;
    box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
    transition: transform 0.2s ease-in-out;
}

.card:hover {
    transform: translateY(-2px);
}

.card-header {
    background-color: #f8f9fc;
    border-bottom: 1px solid #e3e6f0;
}

/* Form Label Styles */
.form-label {
    font-weight: 600;
    color: #5a5c69;
}

/* Code Block Styles */
code {
    padding: 0.2rem 0.4rem;
    font-size: 87.5%;
    background-color: #f8f9fc;
    border-radius: 0.2rem;
}

.badge {
    font-size: 0.85em;
    padding: 0.5em 0.85em;
}
</style>
@endsection
@endsection 