@extends('layouts.app')

@section('title', 'Create GET-Only Receiver')

@section('content')
<div class="container-fluid">
    <!-- Page Heading -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h3 mb-0 text-gray-800">
            <i class="fas fa-plus me-2"></i>Create GET-Only Receiver
        </h1>
        <a href="{{ route('get-only-receivers.index') }}" class="btn btn-secondary">
            <i class="fas fa-arrow-left me-2"></i>Back to List
        </a>
    </div>

    @if($errors->any())
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-circle me-2"></i>
            <strong>Please fix the following errors:</strong>
            <ul class="mb-0 mt-2">
                @foreach($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    <form action="{{ route('get-only-receivers.store') }}" method="POST" id="createForm">
        @csrf
        
        <div class="row">
            <!-- Basic Information -->
            <div class="col-lg-8">
                <div class="card shadow mb-4">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-primary">Basic Information</h6>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="name" class="form-label">Name <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="name" name="name" 
                                           value="{{ old('name') }}" required>
                                    <div class="form-text">Descriptive name for this receiver</div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="endpoint" class="form-label">Endpoint <span class="text-danger">*</span></label>
                                    <div class="input-group">
                                        <span class="input-group-text">/api/get/</span>
                                        <input type="text" class="form-control" id="endpoint" name="endpoint" 
                                               value="{{ old('endpoint') }}" required pattern="[a-zA-Z0-9\-_]+">
                                    </div>
                                    <div class="form-text">URL endpoint (alphanumeric, dash, underscore only)</div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="form-group mb-3">
                            <label for="description" class="form-label">Description</label>
                            <textarea class="form-control" id="description" name="description" rows="3">{{ old('description') }}</textarea>
                            <div class="form-text">Optional description of what this receiver provides</div>
                        </div>
                    </div>
                </div>

                <!-- Data Source Configuration -->
                <div class="card shadow mb-4">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-primary">Data Source Configuration</h6>
                    </div>
                    <div class="card-body">
                        <div class="form-group mb-3">
                            <label for="data_source" class="form-label">Data Source <span class="text-danger">*</span></label>
                            <select class="form-control" id="data_source" name="data_source" required>
                                <option value="">Select Data Source</option>
                                <option value="table" {{ old('data_source') === 'table' ? 'selected' : '' }}>Database Table</option>
                                <option value="api" {{ old('data_source') === 'api' ? 'selected' : '' }}>External API</option>
                                <option value="custom" {{ old('data_source') === 'custom' ? 'selected' : '' }}>Custom Query</option>
                            </select>
                        </div>

                        <!-- Table Source -->
                        <div id="table_source" class="data-source-config" style="display: none;">
                            <div class="form-group mb-3">
                                <label for="source_table" class="form-label">Source Table <span class="text-danger">*</span></label>
                                <select class="form-control" id="source_table" name="source_table">
                                    <option value="">Select Table</option>
                                    @foreach($tables as $table)
                                        <option value="{{ $table }}" {{ old('source_table') === $table ? 'selected' : '' }}>
                                            {{ $table }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                        </div>

                        <!-- API Source -->
                        <div id="api_source" class="data-source-config" style="display: none;">
                            <div class="form-group mb-3">
                                <label for="source_api_url" class="form-label">API URL <span class="text-danger">*</span></label>
                                <input type="url" class="form-control" id="source_api_url" name="source_api_url" 
                                       value="{{ old('source_api_url') }}" placeholder="https://api.example.com/data">
                            </div>
                            <div class="form-group mb-3">
                                <label for="source_api_headers" class="form-label">API Headers (JSON)</label>
                                <textarea class="form-control" id="source_api_headers" name="source_api_headers" rows="3" 
                                          placeholder='{"Authorization": "Bearer token", "Content-Type": "application/json"}'>{{ old('source_api_headers') }}</textarea>
                            </div>
                        </div>

                        <!-- Custom Query -->
                        <div id="custom_source" class="data-source-config" style="display: none;">
                            <div class="form-group mb-3">
                                <label for="custom_query" class="form-label">Custom SQL Query <span class="text-danger">*</span></label>
                                <textarea class="form-control" id="custom_query" name="custom_query" rows="5" 
                                          placeholder="SELECT * FROM users WHERE status = 'active'">{{ old('custom_query') }}</textarea>
                                <div class="form-text">Use :parameter_name for dynamic parameters</div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Response Configuration -->
                <div class="card shadow mb-4">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-primary">Response Configuration</h6>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="response_format" class="form-label">Response Format <span class="text-danger">*</span></label>
                                    <select class="form-control" id="response_format" name="response_format" required>
                                        <option value="json" {{ old('response_format') === 'json' ? 'selected' : '' }}>JSON</option>
                                        <option value="xml" {{ old('response_format') === 'xml' ? 'selected' : '' }}>XML</option>
                                        <option value="csv" {{ old('response_format') === 'csv' ? 'selected' : '' }}>CSV</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label class="form-label">Enable Pagination</label>
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="enable_pagination" 
                                               name="enable_pagination" value="1" {{ old('enable_pagination') ? 'checked' : '' }}>
                                        <label class="form-check-label" for="enable_pagination">
                                            Enable pagination support
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="default_limit" class="form-label">Default Limit <span class="text-danger">*</span></label>
                                    <input type="number" class="form-control" id="default_limit" name="default_limit" 
                                           value="{{ old('default_limit', 50) }}" min="1" max="1000" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="max_limit" class="form-label">Maximum Limit <span class="text-danger">*</span></label>
                                    <input type="number" class="form-control" id="max_limit" name="max_limit" 
                                           value="{{ old('max_limit', 1000) }}" min="1" max="10000" required>
                                </div>
                            </div>
                        </div>

                        <div class="form-group mb-3">
                            <label for="response_fields" class="form-label">Response Fields (Optional)</label>
                            <input type="text" class="form-control" id="response_fields" name="response_fields" 
                                   value="{{ old('response_fields') }}" placeholder="field1,field2,field3">
                            <div class="form-text">Comma-separated list of fields to include in response (leave empty for all)</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Security & Settings -->
            <div class="col-lg-4">
                <!-- Token Management -->
                <div class="card shadow mb-4">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-primary">Token Management</h6>
                    </div>
                    <div class="card-body">
                        <div class="form-group mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="require_token" 
                                       name="require_token" value="1" {{ old('require_token', 1) ? 'checked' : '' }}>
                                <label class="form-check-label" for="require_token">
                                    Require API Token
                                </label>
                            </div>
                        </div>

                        <div class="form-group mb-3">
                            <label for="api_token_id" class="form-label">Primary Token</label>
                            <select class="form-control" id="api_token_id" name="api_token_id">
                                <option value="">Select Token</option>
                                @foreach($apiTokens as $token)
                                    <option value="{{ $token->id }}" {{ old('api_token_id') == $token->id ? 'selected' : '' }}>
                                        {{ $token->name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>

                        <div class="form-group mb-3">
                            <label for="allowed_tokens" class="form-label">Additional Allowed Tokens</label>
                            <select class="form-control" id="allowed_tokens" name="allowed_tokens[]" multiple>
                                @foreach($apiTokens as $token)
                                    <option value="{{ $token->id }}" {{ in_array($token->id, old('allowed_tokens', [])) ? 'selected' : '' }}>
                                        {{ $token->name }}
                                    </option>
                                @endforeach
                            </select>
                            <div class="form-text">Hold Ctrl/Cmd to select multiple</div>
                        </div>
                    </div>
                </div>

                <!-- Rate Limiting -->
                <div class="card shadow mb-4">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-primary">Rate Limiting</h6>
                    </div>
                    <div class="card-body">
                        <div class="form-group mb-3">
                            <label for="rate_limit_per_minute" class="form-label">Per Minute <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" id="rate_limit_per_minute" name="rate_limit_per_minute" 
                                   value="{{ old('rate_limit_per_minute', 60) }}" min="1" max="1000" required>
                        </div>
                        <div class="form-group mb-3">
                            <label for="rate_limit_per_hour" class="form-label">Per Hour <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" id="rate_limit_per_hour" name="rate_limit_per_hour" 
                                   value="{{ old('rate_limit_per_hour', 1000) }}" min="1" max="10000" required>
                        </div>
                    </div>
                </div>

                <!-- Caching -->
                <div class="card shadow mb-4">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-primary">Caching</h6>
                    </div>
                    <div class="card-body">
                        <div class="form-group mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="enable_cache" 
                                       name="enable_cache" value="1" {{ old('enable_cache', 1) ? 'checked' : '' }}>
                                <label class="form-check-label" for="enable_cache">
                                    Enable Response Caching
                                </label>
                            </div>
                        </div>
                        <div class="form-group mb-3">
                            <label for="cache_duration" class="form-label">Cache Duration (seconds) <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" id="cache_duration" name="cache_duration" 
                                   value="{{ old('cache_duration', 300) }}" min="60" max="86400" required>
                        </div>
                    </div>
                </div>

                <!-- CORS -->
                <div class="card shadow mb-4">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-primary">CORS Settings</h6>
                    </div>
                    <div class="card-body">
                        <div class="form-group mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="enable_cors" 
                                       name="enable_cors" value="1" {{ old('enable_cors', 1) ? 'checked' : '' }}>
                                <label class="form-check-label" for="enable_cors">
                                    Enable CORS
                                </label>
                            </div>
                        </div>
                        <div class="form-group mb-3">
                            <label for="cors_origins" class="form-label">Allowed Origins</label>
                            <input type="text" class="form-control" id="cors_origins" name="cors_origins" 
                                   value="{{ old('cors_origins') }}" placeholder="https://example.com,https://app.example.com">
                            <div class="form-text">Comma-separated list (leave empty for all origins)</div>
                        </div>
                    </div>
                </div>

                <!-- Status -->
                <div class="card shadow mb-4">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-primary">Status</h6>
                    </div>
                    <div class="card-body">
                        <div class="form-group mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="is_active" 
                                       name="is_active" value="1" {{ old('is_active', 1) ? 'checked' : '' }}>
                                <label class="form-check-label" for="is_active">
                                    Active
                                </label>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Submit Buttons -->
        <div class="form-group mt-4">
            <div class="row">
                <div class="col-12 mb-3">
                    <button type="submit" class="btn btn-success me-2" id="saveBtn">
                        <i class="fas fa-save me-2"></i>SAVE GET-ONLY RECEIVER
                    </button>
                    <button type="button" class="btn btn-primary me-2" id="altSubmitBtn">
                        <i class="fas fa-rocket me-2"></i>SUBMIT NOW
                    </button>
                </div>
                <div class="col-12 mb-3">
                    <button type="button" class="btn btn-info me-2" id="simpleSubmitBtn">
                        <i class="fas fa-paper-plane me-2"></i>SIMPLE SUBMIT
                    </button>
                    <button type="button" class="btn btn-secondary me-2" onclick="window.location.href='{{ route('get-only-receivers.index') }}'">
                        <i class="fas fa-times me-2"></i>Cancel
                    </button>
                    <button type="button" class="btn btn-warning me-2" id="testEndpointBtn">
                        <i class="fas fa-vial me-2"></i>Test Endpoint
                    </button>
                </div>
            </div>
        </div>
    </form>
</div>
@endsection

@section('scripts')
<script>
// Ensure DOM is ready with both jQuery and vanilla JS
document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM loaded, initializing form...');
    
    // Get form element
    const form = document.getElementById('createForm');
    if (!form) {
        console.error('Form with ID createForm not found!');
        return;
    }
    
    // Handle data source change
    const dataSourceSelect = document.getElementById('data_source');
    if (dataSourceSelect) {
        dataSourceSelect.addEventListener('change', function() {
            console.log('Data source changed to:', this.value);
            
            // Hide all data source configs
            const configs = document.querySelectorAll('.data-source-config');
            configs.forEach(config => config.style.display = 'none');
            
            // Show selected config
            const selectedConfig = document.getElementById(this.value + '_source');
            if (selectedConfig) {
                selectedConfig.style.display = 'block';
            }
        });
        
        // Trigger change on load
        dataSourceSelect.dispatchEvent(new Event('change'));
    }
    
    // Auto-generate endpoint from name
    const nameInput = document.getElementById('name');
    const endpointInput = document.getElementById('endpoint');
    
    if (nameInput && endpointInput) {
        nameInput.addEventListener('input', function() {
            const name = this.value;
            const endpoint = name.toLowerCase()
                .replace(/[^a-z0-9\s\-_]/g, '')
                .replace(/\s+/g, '-')
                .replace(/\-+/g, '-')
                .replace(/^-+|-+$/g, '');
            
            if (endpointInput.value === '' || endpointInput.dataset.autoGenerated === 'true') {
                endpointInput.value = endpoint;
                endpointInput.dataset.autoGenerated = 'true';
            }
        });
        
        endpointInput.addEventListener('input', function() {
            this.dataset.autoGenerated = 'false';
        });
    }
    
    // Form validation function
    function validateForm() {
        console.log('Validating form...');
        
        const requiredFields = ['name', 'endpoint', 'data_source', 'response_format'];
        const errors = [];
        
        requiredFields.forEach(fieldName => {
            const field = document.getElementById(fieldName);
            if (!field || !field.value.trim()) {
                errors.push(fieldName + ' is required');
            }
        });
        
        // Check conditional requirements
        const dataSource = document.getElementById('data_source').value;
        if (dataSource === 'table') {
            const sourceTable = document.getElementById('source_table');
            if (!sourceTable || !sourceTable.value) {
                errors.push('Source table is required when data source is table');
            }
        } else if (dataSource === 'api') {
            const sourceApiUrl = document.getElementById('source_api_url');
            if (!sourceApiUrl || !sourceApiUrl.value) {
                errors.push('API URL is required when data source is API');
            }
        } else if (dataSource === 'custom') {
            const customQuery = document.getElementById('custom_query');
            if (!customQuery || !customQuery.value) {
                errors.push('Custom query is required when data source is custom');
            }
        }
        
        if (errors.length > 0) {
            console.error('Validation errors:', errors);
            alert('Please fix the following errors:\n' + errors.join('\n'));
            return false;
        }
        
        console.log('Form validation passed');
        return true;
    }
    
    // Get all buttons
    const saveBtn = document.getElementById('saveBtn');
    const altSubmitBtn = document.getElementById('altSubmitBtn');
    const simpleSubmitBtn = document.getElementById('simpleSubmitBtn');
    const testEndpointBtn = document.getElementById('testEndpointBtn');
    
    // Button event handlers
    if (saveBtn) {
        saveBtn.addEventListener('click', function(e) {
            console.log('Save button clicked');
            e.preventDefault();
            
            if (validateForm()) {
                console.log('Submitting form...');
                // Add loading state
                this.disabled = true;
                this.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Saving...';
                form.submit();
            }
        });
    }
    
    if (altSubmitBtn) {
        altSubmitBtn.addEventListener('click', function(e) {
            console.log('Alt submit button clicked');
            e.preventDefault();
            
            if (validateForm()) {
                console.log('Submitting form...');
                // Add loading state
                this.disabled = true;
                this.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Submitting...';
                form.submit();
            }
        });
    }
    
    if (simpleSubmitBtn) {
        simpleSubmitBtn.addEventListener('click', function(e) {
            console.log('Simple submit button clicked');
            e.preventDefault();
            
            // Simple submit with minimal validation
            const name = document.getElementById('name').value;
            const endpoint = document.getElementById('endpoint').value;
            const dataSource = document.getElementById('data_source').value;
            
            if (!name || !endpoint || !dataSource) {
                alert('Please fill in Name, Endpoint, and Data Source');
                return;
            }
            
            console.log('Simple submitting form...');
            // Add loading state
            this.disabled = true;
            this.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Submitting...';
            form.submit();
        });
    }
    
    if (testEndpointBtn) {
        testEndpointBtn.addEventListener('click', function(e) {
            console.log('Test endpoint button clicked');
            e.preventDefault();
            
            const endpoint = document.getElementById('endpoint').value;
            if (!endpoint) {
                alert('Please enter an endpoint first');
                return;
            }
            
            const testUrl = '/api/get/' + endpoint + '?test=1';
            console.log('Opening test URL:', testUrl);
            window.open(testUrl, '_blank');
        });
    }
    
    console.log('Form initialization complete');
});

// jQuery fallback for additional compatibility
if (typeof jQuery !== 'undefined') {
    $(document).ready(function() {
        console.log('jQuery fallback initialized');
        
        // Handle data source change with jQuery
        $('#data_source').change(function() {
            $('.data-source-config').hide();
            const selected = $(this).val();
            if (selected) {
                $(`#${selected}_source`).show();
            }
        });

        // Trigger change on page load
        $('#data_source').trigger('change');

        // Convert response fields to array
        $('#response_fields').on('blur', function() {
            const value = $(this).val();
            if (value) {
                const fields = value.split(',').map(f => f.trim()).filter(f => f);
                $(this).val(fields.join(','));
            }
        });

        // Convert CORS origins to array
        $('#cors_origins').on('blur', function() {
            const value = $(this).val();
            if (value) {
                const origins = value.split(',').map(o => o.trim()).filter(o => o);
                $(this).val(origins.join(','));
            }
        });
    });
}
</script>
@endsection 