<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\SystemController;
use App\Http\Controllers\ApiReceiverController;
use App\Http\Controllers\PostOnlyReceiverController;
use App\Http\Controllers\Api\TestController;
use App\Http\Controllers\ApiSourceController;
use App\Http\Controllers\ApiTargetController;
use App\Http\Controllers\TransceiverController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

// Public test routes (no token required)
Route::get('/ping', function() {
    return response()->json([
        'status' => 'success',
        'message' => 'API is accessible',
        'timestamp' => now()->toIso8601String()
    ]);
});

Route::get('/test-token', function(Request $request) {
    $token = $request->bearerToken();
    return response()->json([
        'status' => 'success',
        'message' => 'Token test endpoint',
        'data' => [
            'received_token' => $token ? 'Yes' : 'No',
            'token_value' => $token ? substr($token, 0, 10) . '...' : null,
            'headers' => $request->headers->all()
        ]
    ]);
});

// Protected API routes (require valid token)
Route::middleware('api.token')->group(function () {
    Route::get('/test', [TestController::class, 'test']);
    Route::get('/user', [TestController::class, 'user']);
});

// System API Routes (public access for monitoring)
Route::get('/system/stats', [SystemController::class, 'stats']);
Route::get('/system/status', [SystemController::class, 'stats']); // Alias for status
Route::get('/system/health', [SystemController::class, 'health']);

// API Receiver routes - Enhanced with GET-only support
Route::prefix('in')->group(function () {
    // POST-Only Receiver endpoint (specific for POST-Only receivers) - MUST BE FIRST
    Route::post('{endpoint}', [App\Http\Controllers\PostOnlyReceiverController::class, 'receive'])
        ->name('api.post-only-receiver.receive');
    
    // Specific route for test12 endpoint (no database required)
    Route::match(['GET', 'POST', 'PUT', 'PATCH', 'DELETE'], 'test12', function(Request $request) {
        return response()->json([
            'success' => true,
            'message' => 'Test12 endpoint is working!',
            'data' => [
                'method' => $request->method(),
                'endpoint' => 'test12',
                'timestamp' => now()->toISOString(),
                'request_data' => $request->all(),
                'headers' => $request->headers->all()
            ]
        ]);
    })->name('api.receiver.test12');
    
    // Enhanced receiver endpoint with GET-only support and client token sync
    Route::match(['GET', 'POST', 'PUT', 'PATCH', 'DELETE'], '{endpoint}', 
        [ApiReceiverController::class, 'receive'])
        ->name('api.receiver.receive');
    
    // GET-only receiver endpoint with additional query parameter support
    Route::get('{endpoint}/data', [ApiReceiverController::class, 'receive'])
        ->name('api.receiver.get_data');
});

// Tambahan agar /api/post/{endpoint} juga diterima
Route::prefix('post')->group(function () {
    Route::match(['GET', 'POST', 'PUT', 'PATCH', 'DELETE'], '{endpoint}', [ApiReceiverController::class, 'receive']);
});

// API Source routes
Route::prefix('out')->group(function () {
    Route::get('sources', [ApiSourceController::class, 'index']);
    Route::post('sources', [ApiSourceController::class, 'store']);
    Route::get('sources/{source}', [ApiSourceController::class, 'show']);
    Route::put('sources/{source}', [ApiSourceController::class, 'update']);
    Route::delete('sources/{source}', [ApiSourceController::class, 'destroy']);
    Route::post('sources/{source}/execute', [ApiSourceController::class, 'execute']);
});

// API Target routes
Route::prefix('send')->group(function () {
    Route::get('targets', [ApiTargetController::class, 'index']);
    Route::post('targets', [ApiTargetController::class, 'store']);
    Route::get('targets/{target}', [ApiTargetController::class, 'show']);
    Route::put('targets/{target}', [ApiTargetController::class, 'update']);
    Route::delete('targets/{target}', [ApiTargetController::class, 'destroy']);
    Route::post('targets/{target}/send', [ApiTargetController::class, 'send']);
});

// System API routes
Route::prefix('system')->group(function () {
    Route::get('logs', [SystemController::class, 'logs']);
});

Route::match(['POST', 'PUT'], '/transceivers/{transceiver}/send', [TransceiverController::class, 'send']);
