<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\ApiTokenController;
use App\Http\Controllers\ApiClientController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\MonitoringController;
use App\Http\Controllers\AnalyticsController;
use App\Http\Controllers\Auth\LoginController;
use App\Http\Controllers\Auth\RegisterController;
use App\Http\Controllers\ApiSourceController;
use App\Http\Controllers\ApiTargetController;
use App\Http\Controllers\ApiReceiverController;
use App\Http\Controllers\TransceiverController;
use App\Http\Controllers\ApiTransceiverController;
use App\Http\Controllers\ProfileController;
use App\Http\Controllers\UserController;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "web" middleware group. Make something great!
|
*/

// Authentication routes
Route::get('login', [LoginController::class, 'showLoginForm'])->name('login');
Route::post('login', [LoginController::class, 'login']);
Route::post('logout', [LoginController::class, 'logout'])->name('logout');
// Registration routes
Route::get('register', [RegisterController::class, 'showRegistrationForm'])->name('register');
Route::post('register', [RegisterController::class, 'register']);

// All routes are public by default
Route::get('/', function () {
    // Check if this is an API request (based on Accept header or query parameter)
    if (request()->wantsJson() || request()->has('api') || request()->header('Accept') === 'application/json') {
        // Return API endpoint information
        return response()->json([
            'success' => true,
            'message' => 'API Management System is running',
            'endpoints' => [
                'api_receivers' => url('/api/in/{endpoint}'),
                'api_sources' => url('/api/out/sources'),
                'api_targets' => url('/api/send/targets'),
                'system_status' => url('/api/system/status'),
                'ping' => url('/api/ping')
            ],
            'timestamp' => now()->toISOString(),
            'version' => '1.0.0'
        ]);
    }
    
    // For web requests, try to redirect to dashboard, but with fallback
    try {
        return redirect()->route('dashboard.index');
    } catch (\Exception $e) {
        // If dashboard route fails, show welcome page directly
        return view('welcome');
    }
})->name('home');

// Admin interface route
Route::get('/admin', function () {
    return view('welcome');
})->name('admin');

Route::get('/dashboard', function() {
    try {
        return app(\App\Http\Controllers\DashboardController::class)->index();
    } catch (\Exception $e) {
        // If dashboard fails, show a simple status page
        return response()->json([
            'success' => true,
            'message' => 'API Management System Dashboard',
            'status' => 'online',
            'timestamp' => now()->toISOString(),
            'note' => 'Dashboard view failed, showing API status instead'
        ]);
    }
})->middleware('auth')->name('dashboard.index');

// Profile routes (admin/user)
Route::middleware('auth')->group(function () {
    Route::get('/profile/edit', [ProfileController::class, 'edit'])->name('profile.edit');
    Route::put('/profile', [ProfileController::class, 'update'])->name('profile.update');
    // User management (create only for now)
    Route::get('/users', [UserController::class, 'index'])->name('users.index');
    Route::get('/users/create', [UserController::class, 'create'])->name('users.create');
    Route::post('/users', [UserController::class, 'store'])->name('users.store');
    Route::get('/users/{user}/edit', [UserController::class, 'edit'])->name('users.edit');
    Route::put('/users/{user}', [UserController::class, 'update'])->name('users.update');
    Route::delete('/users/{user}', [UserController::class, 'destroy'])->name('users.destroy');
    // Quick delete route (GET) mirroring api-clients delete, for hosting environments where DELETE is blocked
    Route::get('/users/{user}/delete', [UserController::class, 'delete'])->name('users.delete');
    // Alternative delete via POST (no method spoofing)
    Route::post('/users/{user}/delete', [UserController::class, 'delete'])->name('users.delete.post');
});

// API Management Routes (Protected)
Route::middleware('auth')->group(function () {
    Route::resource('api-tokens', ApiTokenController::class);
    Route::get('api-tokens/{apiToken}/revoke', [ApiTokenController::class, 'revoke'])->name('api-tokens.revoke.get');
    Route::post('api-tokens/{apiToken}/revoke', [ApiTokenController::class, 'revoke'])->name('api-tokens.revoke');
    Route::post('api-tokens/{apiToken}/regenerate', [ApiTokenController::class, 'regenerate'])->name('api-tokens.regenerate');
    Route::get('api-tokens/{apiToken}/regenerate', [ApiTokenController::class, 'regenerate'])->name('api-tokens.regenerate.get');
    // Quick delete route for API Tokens (GET), mirroring clients delete behavior
    Route::get('api-tokens/{apiToken}/delete', function($apiToken) {
        try {
            $token = \App\Models\ApiToken::findOrFail($apiToken);
            $name = $token->name ?: 'Token#' . $token->id;
            $token->delete();
            return redirect()->route('api-tokens.index')->with('success', "API Token '{$name}' deleted successfully!");
        } catch (\Exception $e) {
            return redirect()->route('api-tokens.index')->with('error', 'Error deleting API Token: ' . $e->getMessage());
        }
    })->name('api-tokens.delete');

    Route::resource('api-clients', ApiClientController::class);
});
Route::middleware('auth')->group(function () {
    Route::get('api-clients/{apiClient}/credentials', [ApiClientController::class, 'getCredentials'])->name('api-clients.credentials');
    Route::get('api-clients/{id}/credentials-simple', function($id) {
        return response()->json([
            'success' => true,
            'message' => 'Simple credentials route working',
            'id' => $id,
            'url' => request()->url()
        ]);
    })->name('api-clients.credentials-simple');
    Route::get('debug-credentials/{apiClient}', function($apiClient) {
        \Illuminate\Support\Facades\Log::info('Debug credentials route called', [
            'apiClient' => $apiClient,
            'type' => gettype($apiClient),
            'url' => request()->url()
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'Debug route working',
            'apiClient' => $apiClient,
            'type' => gettype($apiClient),
            'url' => request()->url()
        ]);
    })->name('debug.credentials');

    // Simple test route without model binding
    Route::get('simple-credentials/{id}', function($id) {
        return response()->json([
            'success' => true,
            'message' => 'Simple route working',
            'id' => $id,
            'url' => request()->url()
        ]);
    })->name('simple.credentials');
    Route::get('test-credentials/{id}', function($id) {
        return response()->json([
            'success' => true,
            'message' => 'Test route working',
            'id' => $id,
            'url' => request()->url()
        ]);
    })->name('test.credentials');
    Route::get('api-clients/{apiClient}/delete', function($apiClient) {
        try {
            $client = \App\Models\ApiClient::findOrFail($apiClient);
            $name = $client->name;
            $client->delete();
            return redirect()->route('api-clients.index')->with('success', "API Client '{$name}' deleted successfully!");
        } catch (\Exception $e) {
            return redirect()->route('api-clients.index')->with('error', 'Error deleting API Client: ' . $e->getMessage());
        }
    })->name('api-clients.delete');
});

// Monitoring & Analytics (Protected)
Route::middleware('auth')->group(function () {
    Route::get('monitoring', [MonitoringController::class, 'index'])->name('monitoring.index');
    Route::get('analytics', [AnalyticsController::class, 'index'])->name('analytics.index');
});

// Public Routes (no authentication required)
Route::get('/home', function () {
    return view('home');
})->name('home');

// API Sources Routes (Protected)
Route::middleware('auth')->group(function () {
    Route::resource('api-sources', ApiSourceController::class);
    Route::post('api-sources/{apiSource}/import', [ApiSourceController::class, 'import'])->name('api-sources.import');
    Route::post('api-sources/{apiSource}/auto-import', [ApiSourceController::class, 'autoImport'])->name('api-sources.auto-import');
    Route::get('api-sources/{apiSource}/import-stats', [ApiSourceController::class, 'importStats'])->name('api-sources.import-stats');
    Route::post('api-sources/bulk-auto-import', [ApiSourceController::class, 'bulkAutoImport'])->name('api-sources.bulk-auto-import');
    Route::post('api-sources/{apiSource}/test', [ApiSourceController::class, 'test'])->name('api-sources.test');
    Route::post('api-sources/{apiSource}/clear-logs', [ApiSourceController::class, 'clearLogs'])->name('api-sources.clear-logs');
    Route::post('api-sources/{apiSource}/clear-data', [ApiSourceController::class, 'clearData'])->name('api-sources.clear-data');
    Route::get('api-sources/{apiSource}/data', [ApiSourceController::class, 'data'])->name('api-sources.data');
    Route::get('api-sources/{apiSource}/json', [ApiSourceController::class, 'data'])->name('api-sources.json');
    Route::get('api-sources/{apiSource}/logs', [ApiSourceController::class, 'logs'])->name('api-sources.logs');
    Route::match(['GET', 'POST'], 'import-data/{apiSource}', [ApiSourceController::class, 'import'])->name('import-data');
    Route::get('api-sources/{apiSource}/preview-pdf', [ApiSourceController::class, 'previewPdf'])->name('api-sources.preview-pdf');
    Route::get('api-sources/{apiSource}/export-pdf', [App\Http\Controllers\ApiSourceController::class, 'exportPdf'])->name('api-sources.export-pdf');
});

// API Targets Routes (Protected)
Route::middleware('auth')->group(function () {
    Route::resource('api-targets', ApiTargetController::class)->parameters(['api-targets' => 'apiTarget']);
    Route::post('api-targets/{apiTarget}/send', [ApiTargetController::class, 'send'])->name('api-targets.send');
    Route::get('api-targets/{apiTarget}/logs', [ApiTargetController::class, 'logs'])->name('api-targets.logs');
    Route::get('api-targets/logs/{logId}', [ApiTargetController::class, 'logDetails'])->name('api-targets.log-details');
    Route::get('api-targets/{apiTarget}/documentation', [ApiTargetController::class, 'documentation'])->name('api-targets.documentation');
    Route::post('api-targets/test-connection', [ApiTargetController::class, 'testConnection'])->name('api-targets.test-connection');
    Route::post('api-targets/table-fields', [ApiTargetController::class, 'tableFields'])->name('api-targets.table-fields');
});

// API Receivers Routes (Protected)
Route::middleware('auth')->group(function () {
    Route::resource('api-receivers', ApiReceiverController::class)->parameters(['api-receivers' => 'apiReceiver']);
    Route::get('api-receivers/{apiReceiver}/logs', [ApiReceiverController::class, 'logs'])->name('api-receivers.logs');
    Route::get('api-receivers/{apiReceiver}/preview-pdf', [App\Http\Controllers\ApiReceiverController::class, 'previewPdf'])->name('api-receivers.preview-pdf');
    Route::get('api-receivers/{apiReceiver}/export-pdf', [App\Http\Controllers\ApiReceiverController::class, 'exportPdf'])->name('api-receivers.export-pdf');
    Route::get('api-receivers/{apiReceiver}/delete', function($apiReceiver) {
        try {
            $receiver = \App\Models\ApiReceiver::findOrFail($apiReceiver);
            $name = $receiver->name;
            $receiver->delete();
            return redirect()->route('api-receivers.index')->with('success', "API Receiver '{$name}' deleted successfully!");
        } catch (\Exception $e) {
            return redirect()->route('api-receivers.index')->with('error', 'Error deleting API Receiver: ' . $e->getMessage());
        }
    })->name('api-receivers.delete');
});

// Transceivers Routes (Protected)
Route::middleware('auth')->group(function () {
    Route::resource('transceivers', TransceiverController::class);
});

// API Transceivers Routes (Protected) (relasi receiver-target via client)
Route::middleware('auth')->group(function () {
    Route::resource('api-transceivers', ApiTransceiverController::class);
    Route::post('api-transceivers/{apiTransceiver}/sync', [ApiTransceiverController::class, 'sync'])->name('api-transceivers.sync');
    Route::post('api-transceivers/{apiTransceiver}/toggle', [ApiTransceiverController::class, 'toggleStatus'])->name('api-transceivers.toggle');
    Route::get('api-transceivers/{apiTransceiver}/logs', [ApiTransceiverController::class, 'logs'])->name('api-transceivers.logs');
    Route::post('api-transceivers/{apiTransceiver}/test', [ApiTransceiverController::class, 'test'])->name('api-transceivers.test');
});

// Test route
Route::get('/test-route', function() {
    return 'Route test successful!';
});

// Basic test route
Route::get('/basic-test', function() {
    return 'Basic route test successful!';
});

// API test route that returns JSON
Route::get('/api-test', function() {
    return response()->json([
        'status' => 'success',
        'message' => 'API test route successful',
        'timestamp' => now()->toIso8601String()
    ]);
});

// API Endpoints at root level for copilotproduction.site
Route::prefix('api')->group(function () {
    // Ping endpoint
    Route::get('/ping', function() {
        return response()->json([
            'status' => 'success',
            'message' => 'API is accessible',
            'timestamp' => now()->toIso8601String()
        ]);
    });
    
    // System status endpoint
    Route::get('/system/status', function() {
        return response()->json([
            'status' => 'success',
            'message' => 'System is running',
            'timestamp' => now()->toIso8601String(),
            'uptime' => 'online'
        ]);
    });
    
    // API receiver endpoints
    Route::prefix('in')->group(function () {
        Route::match(['GET', 'POST', 'PUT', 'PATCH', 'DELETE'], '{endpoint}', 
            [ApiReceiverController::class, 'receive'])
            ->name('api.receiver.receive');
    });
    
    // API source endpoints
    Route::prefix('out')->group(function () {
        Route::get('sources', [ApiSourceController::class, 'index']);
        Route::post('sources', [ApiSourceController::class, 'store']);
        Route::get('sources/{source}', [ApiSourceController::class, 'show']);
        Route::put('sources/{source}', [ApiSourceController::class, 'update']);
        Route::delete('sources/{source}', [ApiSourceController::class, 'destroy']);
        Route::post('sources/{source}/execute', [ApiSourceController::class, 'execute']);
    });
    
    // API target endpoints
    Route::prefix('send')->group(function () {
        Route::get('targets', [ApiTargetController::class, 'index']);
        Route::post('targets', [ApiTargetController::class, 'store']);
        Route::get('targets/{target}', [ApiTargetController::class, 'show']);
        Route::put('targets/{target}', [ApiTargetController::class, 'update']);
        Route::delete('targets/{target}', [ApiTargetController::class, 'destroy']);
        Route::post('targets/{target}/send', [ApiTargetController::class, 'send']);
    });
});

// Direct API endpoints at root level (for copilotproduction.site)
Route::match(['GET', 'POST', 'PUT', 'PATCH', 'DELETE'], '/{endpoint}', function($endpoint) {
    // Check if this is an API request
    if (request()->wantsJson() || request()->has('api') || request()->header('Accept') === 'application/json') {
        // Route to API receiver
        return app(\App\Http\Controllers\ApiReceiverController::class)->receive(request(), $endpoint);
    }
    
    // For non-API requests, return 404
    return response()->json([
        'success' => false,
        'error' => 'Endpoint not found',
        'message' => "No API endpoint found for: {$endpoint}",
        'available_endpoints' => [
            'ping' => url('/api/ping'),
            'system_status' => url('/api/system/status'),
            'api_receivers' => url('/api/in/{endpoint}'),
            'admin_interface' => url('/admin')
        ],
        'timestamp' => now()->toISOString()
    ], 404);
})->where('endpoint', '.*');

// Debug route for API source delete
Route::delete('/debug-api-sources/{id}', function($id) {
    return response()->json([
        'success' => true,
        'message' => 'Delete route hit successfully',
        'id' => $id,
        'timestamp' => now()->toIso8601String()
    ]);
})->name('debug.api-sources.delete');

// Debug route for API receiver delete
Route::delete('/debug-api-receivers/{id}', function($id) {
    return response()->json([
        'success' => true,
        'message' => 'API Receiver delete route hit successfully',
        'id' => $id,
        'timestamp' => now()->toIso8601String()
    ]);
})->name('debug.api-receivers.delete');

// Test route for API source delete
Route::delete('/test-api-sources/{id}', function($id) {
    try {
        $apiSource = \App\Models\ApiSource::find($id);
        if (!$apiSource) {
            return response()->json([
                'success' => false,
                'message' => 'API Source not found',
                'id' => $id
            ], 404);
        }
        
        $apiSource->delete();
        
        return response()->json([
            'success' => true,
            'message' => 'API Source deleted successfully',
            'id' => $id,
            'name' => $apiSource->name
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error deleting API Source: ' . $e->getMessage(),
            'id' => $id
        ], 500);
    }
})->name('test.api-sources.delete');

// Test route for API receiver delete
Route::delete('/test-api-receivers/{id}', function($id) {
    try {
        $apiReceiver = \App\Models\ApiReceiver::find($id);
        if (!$apiReceiver) {
            return response()->json([
                'success' => false,
                'message' => 'API Receiver not found',
                'id' => $id
            ], 404);
        }
        
        $apiReceiver->delete();
        
        return response()->json([
            'success' => true,
            'message' => 'API Receiver deleted successfully',
            'id' => $id,
            'name' => $apiReceiver->name
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error deleting API Receiver: ' . $e->getMessage(),
            'id' => $id
        ], 500);
    }
})->name('test.api-receivers.delete');

// Simple test route for API receiver delete (no CSRF required)
Route::get('/simple-delete-receiver/{id}', function($id) {
    try {
        $apiReceiver = \App\Models\ApiReceiver::find($id);
        if (!$apiReceiver) {
            return "API Receiver ID {$id} not found";
        }
        
        $name = $apiReceiver->name;
        $apiReceiver->delete();
        
        return "API Receiver '{$name}' (ID: {$id}) deleted successfully! <a href='/api-receivers'>Back to list</a>";
    } catch (\Exception $e) {
        return "Error deleting API Receiver: " . $e->getMessage();
    }
})->name('simple.delete.receiver');

// Test route untuk memastikan routing berfungsi
Route::get('/test-routing', function() {
    return response()->json([
        'status' => 'success',
        'message' => 'Routing test successful',
        'timestamp' => now()->toISOString(),
        'routes' => [
            'api-receivers.index' => route('api-receivers.index'),
            'api-receivers.create' => route('api-receivers.create'),
            'api-receivers.store' => route('api-receivers.store'),
        ]
    ]);
})->name('test.routing');

// API Receivers Routes - Perbaikan dengan explicit routes (Protected)
Route::middleware('auth')->group(function () {
    Route::prefix('api-receivers')->name('api-receivers.')->group(function () {
        Route::get('/', [ApiReceiverController::class, 'index'])->name('index');
        Route::get('/create', [ApiReceiverController::class, 'create'])->name('create');
        Route::post('/', [ApiReceiverController::class, 'store'])->name('store');
        Route::get('/{apiReceiver}', [ApiReceiverController::class, 'show'])->name('show');
        Route::get('/{apiReceiver}/edit', [ApiReceiverController::class, 'edit'])->name('edit');
        Route::put('/{apiReceiver}', [ApiReceiverController::class, 'update'])->name('update');
        Route::delete('/{apiReceiver}', [ApiReceiverController::class, 'destroy'])->name('destroy');
        Route::get('/{apiReceiver}/logs', [ApiReceiverController::class, 'logs'])->name('logs');
        Route::get('/{apiReceiver}/preview-pdf', [ApiReceiverController::class, 'previewPdf'])->name('preview-pdf');
        Route::get('/{apiReceiver}/export-pdf', [ApiReceiverController::class, 'exportPdf'])->name('export-pdf');
    });
});