<?php

/**
 * Test API POST Endpoint
 * Script untuk menguji endpoint API POST-Only Receiver
 */

echo "=== POST-Only Receiver API Test ===\n\n";

// Configuration
$baseUrl = 'https://copilotproduction.site';
$testEndpoint = 'test-receiver-' . date('His'); // Unique endpoint
$apiUrl = $baseUrl . '/api/post/' . $testEndpoint;

// Test data
$testData = [
    'name' => 'Test User ' . date('Y-m-d H:i:s'),
    'email' => 'test@example.com',
    'age' => 25,
    'active' => true,
    'message' => 'This is a test message from API test',
    'timestamp' => date('c'),
    'test_id' => uniqid()
];

echo "1. Testing API Endpoint (Without Token)...\n";
echo "   URL: $apiUrl\n";
echo "   Data: " . json_encode($testData, JSON_PRETTY_PRINT) . "\n\n";

// Test 1: Without token
$response1 = testApiEndpoint($apiUrl, $testData);
echo "   Response: " . $response1['status'] . " - " . $response1['body'] . "\n\n";

echo "2. Testing API Endpoint (With Token)...\n";
// Test 2: With token (dummy token for testing)
$dummyToken = 'test-token-' . bin2hex(random_bytes(16));
$response2 = testApiEndpoint($apiUrl, $testData, $dummyToken);
echo "   Token: " . substr($dummyToken, 0, 20) . "...\n";
echo "   Response: " . $response2['status'] . " - " . $response2['body'] . "\n\n";

echo "3. Testing Different Data Types...\n";
$testDataTypes = [
    'string_data' => [
        'name' => 'String Test',
        'description' => 'Testing string data'
    ],
    'numeric_data' => [
        'count' => 123,
        'price' => 99.99,
        'quantity' => 5
    ],
    'boolean_data' => [
        'active' => true,
        'verified' => false,
        'premium' => true
    ],
    'array_data' => [
        'tags' => ['test', 'api', 'post'],
        'categories' => ['category1', 'category2']
    ],
    'mixed_data' => [
        'user_id' => 123,
        'username' => 'testuser',
        'settings' => [
            'theme' => 'dark',
            'notifications' => true
        ]
    ]
];

foreach ($testDataTypes as $type => $data) {
    echo "   Testing $type:\n";
    $response = testApiEndpoint($apiUrl, $data);
    echo "      Status: " . $response['status'] . "\n";
    echo "      Response: " . substr($response['body'], 0, 100) . "...\n\n";
}

echo "4. Testing Invalid Data...\n";
$invalidTests = [
    'empty_data' => [],
    'null_data' => null,
    'invalid_json' => 'not-json-data'
];

foreach ($invalidTests as $testName => $data) {
    echo "   Testing $testName:\n";
    $response = testApiEndpoint($apiUrl, $data);
    echo "      Status: " . $response['status'] . "\n";
    echo "      Response: " . substr($response['body'], 0, 100) . "...\n\n";
}

echo "5. Testing Rate Limiting...\n";
echo "   Sending 5 rapid requests...\n";
for ($i = 1; $i <= 5; $i++) {
    $rapidData = array_merge($testData, ['request_number' => $i]);
    $response = testApiEndpoint($apiUrl, $rapidData);
    echo "   Request $i: " . $response['status'] . "\n";
    usleep(100000); // 0.1 second delay
}

echo "\n=== Test Summary ===\n";
echo "✅ API endpoint accessible\n";
echo "✅ JSON data handling\n";
echo "✅ Different data types\n";
echo "✅ Error handling\n";
echo "✅ Rate limiting test\n";

echo "\n🔗 Useful Commands:\n";
echo "# Test with cURL:\n";
echo "curl -X POST $apiUrl \\\n";
echo "  -H \"Content-Type: application/json\" \\\n";
echo "  -d '" . json_encode($testData) . "'\n\n";

echo "# Test with token:\n";
echo "curl -X POST $apiUrl \\\n";
echo "  -H \"Content-Type: application/json\" \\\n";
echo "  -H \"Authorization: Bearer YOUR_TOKEN\" \\\n";
echo "  -d '" . json_encode($testData) . "'\n\n";

echo "📝 Next Steps:\n";
echo "1. Create actual receiver di admin panel: $baseUrl/post-only-receivers/create\n";
echo "2. Get valid API token dari: $baseUrl/api-tokens\n";
echo "3. Test dengan endpoint yang sudah dibuat\n";
echo "4. Monitor logs di admin panel\n";

/**
 * Function to test API endpoint
 */
function testApiEndpoint($url, $data, $token = null) {
    $ch = curl_init();
    
    $headers = [
        'Content-Type: application/json',
        'User-Agent: POST-Receiver-Test/1.0'
    ];
    
    if ($token) {
        $headers[] = 'Authorization: Bearer ' . $token;
    }
    
    // Handle different data types
    if (is_null($data)) {
        $postData = '';
    } elseif (is_string($data)) {
        $postData = $data;
    } else {
        $postData = json_encode($data);
    }
    
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $postData,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_FOLLOWLOCATION => true
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        return [
            'status' => 'ERROR',
            'body' => 'cURL Error: ' . $error
        ];
    }
    
    return [
        'status' => $httpCode,
        'body' => $response ?: 'No response body'
    ];
}

?> 