<?php

/**
 * Manual Test Script untuk API Targets
 * 
 * Script ini untuk menguji fungsi-fungsi API Targets secara manual
 * Jalankan dengan: php test-api-targets.php
 */

require_once 'vendor/autoload.php';

use App\Models\User;
use App\Models\ApiTarget;
use App\Models\ApiSendLog;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;

// Bootstrap Laravel
$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

echo "=== MANUAL TEST SCRIPT UNTUK API TARGETS ===\n\n";

try {
    // Test 1: Membuat user test
    echo "1. Membuat user test...\n";
    $user = User::factory()->create([
        'name' => 'Test User',
        'email' => 'test@example.com',
        'password' => bcrypt('password')
    ]);
    echo "✓ User berhasil dibuat: {$user->name} ({$user->email})\n\n";

    // Test 2: Membuat API Target dengan berbagai konfigurasi
    echo "2. Membuat API Target test...\n";
    $apiTarget = ApiTarget::factory()->create([
        'name' => 'Test API Target',
        'description' => 'API Target untuk testing',
        'base_url' => 'https://httpbin.org',
        'endpoint' => '/post',
        'method' => 'POST',
        'source_table' => 'users',
        'auth_type' => 'none',
        'schedule_type' => 'manual',
        'is_active' => true,
        'created_by' => $user->id
    ]);
    echo "✓ API Target berhasil dibuat: {$apiTarget->name}\n";
    echo "  - URL: {$apiTarget->url}\n";
    echo "  - Method: {$apiTarget->method}\n";
    echo "  - Source Table: {$apiTarget->source_table}\n\n";

    // Test 3: Test URL attribute
    echo "3. Testing URL attribute...\n";
    $testUrl = $apiTarget->url;
    echo "✓ URL lengkap: {$testUrl}\n";
    
    // Test dengan trailing slash
    $apiTarget->base_url = 'https://httpbin.org/';
    $apiTarget->endpoint = '/post';
    echo "✓ URL dengan trailing slash: {$apiTarget->url}\n\n";

    // Test 4: Membuat API Target dengan Basic Auth
    echo "4. Membuat API Target dengan Basic Auth...\n";
    $basicAuthTarget = ApiTarget::factory()->withBasicAuth()->create([
        'name' => 'Basic Auth API Target',
        'base_url' => 'https://httpbin.org',
        'endpoint' => '/basic-auth/testuser/testpass',
        'method' => 'GET',
        'source_table' => 'users',
        'created_by' => $user->id
    ]);
    echo "✓ Basic Auth API Target berhasil dibuat\n";
    echo "  - Username: {$basicAuthTarget->auth_username}\n";
    echo "  - Password: {$basicAuthTarget->auth_password}\n\n";

    // Test 5: Membuat API Target dengan Bearer Token
    echo "5. Membuat API Target dengan Bearer Token...\n";
    $bearerTarget = ApiTarget::factory()->withBearerToken()->create([
        'name' => 'Bearer Token API Target',
        'base_url' => 'https://httpbin.org',
        'endpoint' => '/headers',
        'method' => 'GET',
        'source_table' => 'users',
        'created_by' => $user->id
    ]);
    echo "✓ Bearer Token API Target berhasil dibuat\n";
    echo "  - Token: {$bearerTarget->auth_token}\n\n";

    // Test 6: Membuat API Target dengan API Key
    echo "6. Membuat API Target dengan API Key...\n";
    $apiKeyTarget = ApiTarget::factory()->withApiKey()->create([
        'name' => 'API Key Target',
        'base_url' => 'https://httpbin.org',
        'endpoint' => '/headers',
        'method' => 'GET',
        'source_table' => 'users',
        'created_by' => $user->id
    ]);
    echo "✓ API Key Target berhasil dibuat\n";
    echo "  - API Key: {$apiKeyTarget->auth_token}\n\n";

    // Test 7: Testing schedule types
    echo "7. Testing berbagai schedule types...\n";
    $hourlyTarget = ApiTarget::factory()->hourly()->create([
        'name' => 'Hourly Schedule Target',
        'created_by' => $user->id
    ]);
    echo "✓ Hourly target: {$hourlyTarget->schedule_type} (setiap {$hourlyTarget->schedule_frequency} jam)\n";

    $dailyTarget = ApiTarget::factory()->daily()->create([
        'name' => 'Daily Schedule Target',
        'created_by' => $user->id
    ]);
    echo "✓ Daily target: {$dailyTarget->schedule_type} (setiap {$dailyTarget->schedule_frequency} hari)\n";

    $weeklyTarget = ApiTarget::factory()->weekly()->create([
        'name' => 'Weekly Schedule Target',
        'created_by' => $user->id
    ]);
    echo "✓ Weekly target: {$weeklyTarget->schedule_type} (setiap {$weeklyTarget->schedule_frequency} minggu)\n\n";

    // Test 8: Testing HTTP requests (menggunakan httpbin.org)
    echo "8. Testing HTTP requests...\n";
    
    // Test POST request
    echo "  Testing POST request ke httpbin.org...\n";
    $response = Http::timeout(10)->post('https://httpbin.org/post', [
        'test' => 'data',
        'timestamp' => now()->toISOString()
    ]);
    
    if ($response->successful()) {
        echo "✓ POST request berhasil (Status: {$response->status()})\n";
        $responseData = $response->json();
        echo "  - Response data: " . json_encode($responseData['json']) . "\n";
    } else {
        echo "✗ POST request gagal (Status: {$response->status()})\n";
    }

    // Test GET request
    echo "  Testing GET request ke httpbin.org...\n";
    $response = Http::timeout(10)->get('https://httpbin.org/get');
    
    if ($response->successful()) {
        echo "✓ GET request berhasil (Status: {$response->status()})\n";
    } else {
        echo "✗ GET request gagal (Status: {$response->status()})\n";
    }
    echo "\n";

    // Test 9: Testing database operations
    echo "9. Testing database operations...\n";
    
    // Count API Targets
    $targetCount = ApiTarget::count();
    echo "✓ Total API Targets: {$targetCount}\n";
    
    // Count active targets
    $activeCount = ApiTarget::where('is_active', true)->count();
    echo "✓ Active API Targets: {$activeCount}\n";
    
    // Count by auth type
    $authTypes = ApiTarget::select('auth_type', DB::raw('count(*) as count'))
        ->groupBy('auth_type')
        ->get();
    
    echo "✓ API Targets by auth type:\n";
    foreach ($authTypes as $authType) {
        echo "  - {$authType->auth_type}: {$authType->count}\n";
    }
    echo "\n";

    // Test 10: Testing ApiSendLog creation
    echo "10. Testing ApiSendLog creation...\n";
    
    $sendLog = ApiSendLog::factory()->successful()->create([
        'api_target_id' => $apiTarget->id,
        'payload' => [
            'test' => 'data',
            'timestamp' => now()->toISOString()
        ],
        'response_body' => [
            'status' => 'success',
            'message' => 'Data sent successfully'
        ]
    ]);
    
    echo "✓ ApiSendLog berhasil dibuat\n";
    echo "  - Success: " . ($sendLog->success ? 'Yes' : 'No') . "\n";
    echo "  - Status Code: {$sendLog->status_code}\n";
    echo "  - Response Time: {$sendLog->response_time}s\n";
    echo "  - Records Sent: {$sendLog->records_sent}\n\n";

    // Test 11: Testing error handling
    echo "11. Testing error handling...\n";
    
    $errorLog = ApiSendLog::factory()->failed()->create([
        'api_target_id' => $apiTarget->id,
        'error_message' => 'Connection timeout'
    ]);
    
    echo "✓ Error log berhasil dibuat\n";
    echo "  - Success: " . ($errorLog->success ? 'Yes' : 'No') . "\n";
    echo "  - Error: {$errorLog->error_message}\n\n";

    // Test 12: Testing table structure
    echo "12. Testing table structure...\n";
    
    $tables = ['users', 'api_targets', 'api_send_logs'];
    foreach ($tables as $table) {
        if (DB::getSchemaBuilder()->hasTable($table)) {
            $columns = DB::select("SHOW COLUMNS FROM {$table}");
            echo "✓ Table {$table} exists with " . count($columns) . " columns\n";
        } else {
            echo "✗ Table {$table} does not exist\n";
        }
    }
    echo "\n";

    // Test 13: Testing data retrieval from source table
    echo "13. Testing data retrieval from source table...\n";
    
    $userData = DB::table('users')->get();
    echo "✓ Retrieved " . $userData->count() . " records from users table\n";
    
    if ($userData->count() > 0) {
        $sampleUser = $userData->first();
        echo "  - Sample user: " . json_encode([
            'id' => $sampleUser->id,
            'name' => $sampleUser->name,
            'email' => $sampleUser->email
        ]) . "\n";
    }
    echo "\n";

    // Test 14: Testing API Target relationships
    echo "14. Testing API Target relationships...\n";
    
    $targetWithLogs = ApiTarget::with('sendLogs')->find($apiTarget->id);
    echo "✓ API Target '{$targetWithLogs->name}' has {$targetWithLogs->sendLogs->count()} logs\n";
    
    if ($targetWithLogs->sendLogs->count() > 0) {
        $latestLog = $targetWithLogs->sendLogs->latest()->first();
        echo "  - Latest log: " . ($latestLog->success ? 'Success' : 'Failed') . " at {$latestLog->sent_at}\n";
    }
    echo "\n";

    // Test 15: Performance test
    echo "15. Performance test...\n";
    
    $startTime = microtime(true);
    
    // Create multiple API Targets
    $targets = ApiTarget::factory()->count(10)->create([
        'created_by' => $user->id
    ]);
    
    $endTime = microtime(true);
    $executionTime = ($endTime - $startTime) * 1000; // Convert to milliseconds
    
    echo "✓ Created 10 API Targets in {$executionTime}ms\n";
    
    // Clean up test data
    echo "\n16. Cleaning up test data...\n";
    ApiSendLog::where('api_target_id', $apiTarget->id)->delete();
    ApiTarget::where('created_by', $user->id)->delete();
    $user->delete();
    
    echo "✓ Test data cleaned up successfully\n\n";

    echo "=== SEMUA TEST BERHASIL! ===\n";
    echo "Sistem API Targets berfungsi dengan baik.\n";

} catch (Exception $e) {
    echo "✗ ERROR: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
} 