<?php

/**
 * Test Script untuk POST-Only Receiver
 * 
 * Script ini akan menguji:
 * 1. Koneksi database
 * 2. Tabel dan struktur database
 * 3. Membuat receiver test
 * 4. Menguji endpoint API
 * 5. Validasi data
 */

echo "=== POST-Only Receiver Test Suite ===\n\n";

// Database configuration
$dbConfig = [
    'host' => '127.0.0.1',
    'database' => 'copg1493_api',
    'username' => 'copg1493_api',
    'password' => 'P7H_s;Aco25Vv??J'
];

// Test 1: Database Connection
echo "1. Testing Database Connection...\n";
try {
    $pdo = new PDO(
        "mysql:host={$dbConfig['host']};dbname={$dbConfig['database']}", 
        $dbConfig['username'], 
        $dbConfig['password']
    );
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    echo "   ✅ Database connection successful\n";
} catch (PDOException $e) {
    echo "   ❌ Database connection failed: " . $e->getMessage() . "\n";
    exit(1);
}

// Test 2: Check Required Tables
echo "\n2. Checking Required Tables...\n";
$requiredTables = [
    'post_only_receivers',
    'post_only_receive_logs',
    'api_clients',
    'api_tokens'
];

foreach ($requiredTables as $table) {
    try {
        $stmt = $pdo->query("SHOW TABLES LIKE '$table'");
        if ($stmt->rowCount() > 0) {
            echo "   ✅ Table '$table' exists\n";
        } else {
            echo "   ❌ Table '$table' missing\n";
        }
    } catch (PDOException $e) {
        echo "   ❌ Error checking table '$table': " . $e->getMessage() . "\n";
    }
}

// Test 3: Check API Client
echo "\n3. Checking API Client...\n";
try {
    $stmt = $pdo->query("SELECT id, name, is_active FROM api_clients WHERE is_active = 1 LIMIT 1");
    $apiClient = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($apiClient) {
        echo "   ✅ Found active API client: {$apiClient['name']} (ID: {$apiClient['id']})\n";
        $apiClientId = $apiClient['id'];
    } else {
        echo "   ⚠️  No active API client found, creating one...\n";
        
        // Create a test API client
        $stmt = $pdo->prepare("INSERT INTO api_clients (name, description, client_id, client_secret, usertoken, passtoken, is_active, rate_limit) VALUES (?, ?, ?, ?, ?, ?, 1, 1000)");
        $stmt->execute([
            'Test Client for POST Receiver',
            'Auto-created client for testing POST-Only Receiver',
            'test_client_' . uniqid(),
            bin2hex(random_bytes(32)),
            'test_token_' . uniqid(),
            bin2hex(random_bytes(16))
        ]);
        $apiClientId = $pdo->lastInsertId();
        echo "   ✅ Created test API client (ID: $apiClientId)\n";
    }
} catch (PDOException $e) {
    echo "   ❌ Error with API client: " . $e->getMessage() . "\n";
    $apiClientId = null;
}

// Test 4: Create Test POST-Only Receiver
echo "\n4. Creating Test POST-Only Receiver...\n";
$testEndpoint = 'test-receiver-' . uniqid();
$testTableName = 'test_data_' . uniqid();

try {
    // Check if endpoint already exists
    $stmt = $pdo->prepare("SELECT id FROM post_only_receivers WHERE endpoint = ?");
    $stmt->execute([$testEndpoint]);
    
    if ($stmt->rowCount() > 0) {
        echo "   ⚠️  Endpoint '$testEndpoint' already exists\n";
    } else {
        // Create test receiver
        $sampleData = json_encode([
            'name' => 'Test User',
            'email' => 'test@example.com',
            'age' => 25,
            'active' => true,
            'created_at' => date('Y-m-d H:i:s')
        ]);
        
        $validationRules = json_encode([
            'name' => ['required' => true, 'type' => 'string', 'max_length' => 100],
            'email' => ['required' => true, 'type' => 'email', 'max_length' => 150],
            'age' => ['required' => false, 'type' => 'integer'],
            'active' => ['required' => false, 'type' => 'boolean']
        ]);
        
        $stmt = $pdo->prepare("
            INSERT INTO post_only_receivers 
            (name, description, endpoint, target_table, api_client_id, require_token, 
             validation_rules, auto_create_table, is_active, rate_limit, sample_data, created_at, updated_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
        ");
        
        $stmt->execute([
            'Test Receiver - ' . date('Y-m-d H:i:s'),
            'Auto-created receiver for testing',
            $testEndpoint,
            $testTableName,
            $apiClientId,
            1, // require_token
            $validationRules,
            1, // auto_create_table
            1, // is_active
            100, // rate_limit
            $sampleData
        ]);
        
        $receiverId = $pdo->lastInsertId();
        echo "   ✅ Created test receiver (ID: $receiverId)\n";
        echo "   📍 Endpoint: https://copilotproduction.site/api/post/$testEndpoint\n";
        echo "   📊 Target table: $testTableName\n";
    }
} catch (PDOException $e) {
    echo "   ❌ Error creating test receiver: " . $e->getMessage() . "\n";
}

// Test 5: Get API Token
echo "\n5. Getting API Token...\n";
try {
    $stmt = $pdo->prepare("SELECT token FROM api_tokens WHERE api_client_id = ? AND is_active = 1 LIMIT 1");
    $stmt->execute([$apiClientId]);
    $tokenResult = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($tokenResult) {
        $apiToken = $tokenResult['token'];
        echo "   ✅ Found API token: " . substr($apiToken, 0, 20) . "...\n";
    } else {
        echo "   ⚠️  No API token found, creating one...\n";
        
        // Create test token
        $newToken = bin2hex(random_bytes(32));
        $stmt = $pdo->prepare("
            INSERT INTO api_tokens (api_client_id, token, name, description, is_active, expires_at, created_at, updated_at) 
            VALUES (?, ?, ?, ?, 1, DATE_ADD(NOW(), INTERVAL 1 YEAR), NOW(), NOW())
        ");
        $stmt->execute([
            $apiClientId,
            $newToken,
            'Test Token',
            'Auto-created token for testing'
        ]);
        
        $apiToken = $newToken;
        echo "   ✅ Created test token: " . substr($apiToken, 0, 20) . "...\n";
    }
} catch (PDOException $e) {
    echo "   ❌ Error with API token: " . $e->getMessage() . "\n";
    $apiToken = null;
}

// Test 6: Test API Endpoint
echo "\n6. Testing API Endpoint...\n";
if ($apiToken && isset($testEndpoint)) {
    $testUrl = "https://copilotproduction.site/api/post/$testEndpoint";
    $testData = [
        'name' => 'John Doe',
        'email' => 'john@example.com',
        'age' => 30,
        'active' => true,
        'test_timestamp' => date('Y-m-d H:i:s')
    ];
    
    echo "   📡 Testing URL: $testUrl\n";
    echo "   📤 Sending data: " . json_encode($testData) . "\n";
    
    // Test with cURL
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $testUrl,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($testData),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $apiToken,
            'User-Agent: POST-Only-Receiver-Test/1.0'
        ],
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => false
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    if ($curlError) {
        echo "   ❌ cURL Error: $curlError\n";
    } else {
        echo "   📥 HTTP Status: $httpCode\n";
        echo "   📥 Response: " . ($response ?: 'No response body') . "\n";
        
        if ($httpCode == 200) {
            echo "   ✅ API endpoint test successful!\n";
        } else {
            echo "   ⚠️  API endpoint returned status $httpCode\n";
        }
    }
} else {
    echo "   ⚠️  Skipping API test (missing token or endpoint)\n";
}

// Test 7: Check Logs
echo "\n7. Checking Logs...\n";
if (isset($receiverId)) {
    try {
        $stmt = $pdo->prepare("SELECT COUNT(*) as log_count FROM post_only_receive_logs WHERE receiver_id = ?");
        $stmt->execute([$receiverId]);
        $logCount = $stmt->fetch(PDO::FETCH_ASSOC)['log_count'];
        
        echo "   📊 Found $logCount log entries for test receiver\n";
        
        if ($logCount > 0) {
            $stmt = $pdo->prepare("
                SELECT status_code, response_message, token_valid, validation_success, saved_to_table, created_at 
                FROM post_only_receive_logs 
                WHERE receiver_id = ? 
                ORDER BY created_at DESC 
                LIMIT 3
            ");
            $stmt->execute([$receiverId]);
            $logs = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo "   📋 Recent log entries:\n";
            foreach ($logs as $log) {
                $tokenStatus = $log['token_valid'] ? '✅' : '❌';
                $validationStatus = $log['validation_success'] ? '✅' : '❌';
                $saveStatus = $log['saved_to_table'] ? '✅' : '❌';
                
                echo "     - [{$log['created_at']}] HTTP {$log['status_code']} | Token: $tokenStatus | Validation: $validationStatus | Saved: $saveStatus\n";
                if ($log['response_message']) {
                    echo "       Message: {$log['response_message']}\n";
                }
            }
        }
    } catch (PDOException $e) {
        echo "   ❌ Error checking logs: " . $e->getMessage() . "\n";
    }
}

// Test 8: Cleanup (Optional)
echo "\n8. Cleanup Options...\n";
echo "   ℹ️  To clean up test data, run:\n";
if (isset($receiverId)) {
    echo "   🗑️  DELETE FROM post_only_receivers WHERE id = $receiverId;\n";
    echo "   🗑️  DELETE FROM post_only_receive_logs WHERE receiver_id = $receiverId;\n";
}
if (isset($testTableName)) {
    echo "   🗑️  DROP TABLE IF EXISTS $testTableName;\n";
}

echo "\n=== Test Complete ===\n";
echo "📊 Summary:\n";
echo "   - Database connection: ✅\n";
echo "   - Required tables: Check output above\n";
echo "   - API client: ✅\n";
echo "   - Test receiver: ✅\n";
echo "   - API token: ✅\n";
echo "   - Endpoint test: Check output above\n";
echo "   - Logs: Check output above\n";

echo "\n🔗 Test URLs:\n";
echo "   - Admin Panel: https://copilotproduction.site/post-only-receivers\n";
echo "   - Create Form: https://copilotproduction.site/post-only-receivers/create\n";
if (isset($testEndpoint)) {
    echo "   - Test Endpoint: https://copilotproduction.site/api/post/$testEndpoint\n";
}

echo "\n📚 Next Steps:\n";
echo "   1. Visit the admin panel to see the created receiver\n";
echo "   2. Test the endpoint with different data\n";
echo "   3. Check the logs in the admin panel\n";
echo "   4. Try creating receivers through the web form\n";

?> 