<?php
/**
 * Test API Connection and Data Import
 * This script tests the connection to the Logam Mulia API and simulates data import
 */

echo "=== API Connection Test ===\n\n";

// Test 1: Direct API Connection
echo "1. Testing direct API connection...\n";
$apiUrl = 'https://logam-mulia-api.vercel.app/prices/anekalogam';

try {
    $context = stream_context_create([
        'http' => [
            'timeout' => 30,
            'user_agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
        ]
    ]);
    
    $response = file_get_contents($apiUrl, false, $context);
    
    if ($response === false) {
        throw new Exception("Failed to fetch data from API");
    }
    
    $data = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception("Invalid JSON response: " . json_last_error_msg());
    }
    
    echo "✅ API connection successful!\n";
    echo "   Response status: " . (isset($http_response_header[0]) ? $http_response_header[0] : 'Unknown') . "\n";
    echo "   Data type: " . gettype($data) . "\n";
    
    if (is_array($data)) {
        echo "   Data structure:\n";
        if (isset($data['data']) && is_array($data['data'])) {
            echo "   - Has 'data' key with " . count($data['data']) . " records\n";
            if (!empty($data['data'])) {
                echo "   - First record keys: " . implode(', ', array_keys($data['data'][0])) . "\n";
            }
        } elseif (isset($data[0])) {
            echo "   - Array with " . count($data) . " records\n";
            echo "   - First record keys: " . implode(', ', array_keys($data[0])) . "\n";
        } else {
            echo "   - Single record with keys: " . implode(', ', array_keys($data)) . "\n";
        }
    }
    
    // Show sample data
    echo "\n   Sample data:\n";
    if (isset($data['data']) && !empty($data['data'])) {
        $sample = $data['data'][0];
        foreach ($sample as $key => $value) {
            echo "     $key: " . (is_string($value) ? $value : json_encode($value)) . "\n";
        }
    } elseif (isset($data[0])) {
        $sample = $data[0];
        foreach ($sample as $key => $value) {
            echo "     $key: " . (is_string($value) ? $value : json_encode($value)) . "\n";
        }
    } else {
        foreach ($data as $key => $value) {
            echo "     $key: " . (is_string($value) ? $value : json_encode($value)) . "\n";
        }
    }
    
} catch (Exception $e) {
    echo "❌ API connection failed: " . $e->getMessage() . "\n";
    exit(1);
}

echo "\n=== Data Import Simulation ===\n\n";

// Test 2: Simulate data import process
echo "2. Simulating data import process...\n";

try {
    // Simulate the import logic from the controller
    $insertData = [];
    
    if (is_array($data) && isset($data['data']) && is_array($data['data'])) {
        // Handle structured API response with 'data' key
        foreach ($data['data'] as $row) {
            if (is_array($row)) {
                // Add metadata
                $row['raw_data'] = json_encode($data);
                $row['source'] = 'logam_mulia_api';
                $row['created_at'] = date('Y-m-d H:i:s');
                $row['updated_at'] = date('Y-m-d H:i:s');
                $insertData[] = $row;
            }
        }
    } elseif (is_array($data) && isset($data[0])) {
        // Array of records
        foreach ($data as $row) {
            if (is_array($row)) {
                // Add metadata
                $row['raw_data'] = json_encode($data);
                $row['source'] = 'logam_mulia_api';
                $row['created_at'] = date('Y-m-d H:i:s');
                $row['updated_at'] = date('Y-m-d H:i:s');
                $insertData[] = $row;
            }
        }
    } elseif (is_array($data)) {
        // Single record as array
        $data['raw_data'] = json_encode($data);
        $data['source'] = 'logam_mulia_api';
        $data['created_at'] = date('Y-m-d H:i:s');
        $data['updated_at'] = date('Y-m-d H:i:s');
        $insertData[] = $data;
    }
    
    echo "✅ Data processing successful!\n";
    echo "   Records to import: " . count($insertData) . "\n";
    
    if (!empty($insertData)) {
        echo "   Sample processed record:\n";
        $sample = $insertData[0];
        foreach ($sample as $key => $value) {
            if ($key === 'raw_data') {
                echo "     $key: [JSON data - " . strlen($value) . " characters]\n";
            } else {
                echo "     $key: " . (is_string($value) ? $value : json_encode($value)) . "\n";
            }
        }
    }
    
} catch (Exception $e) {
    echo "❌ Data processing failed: " . $e->getMessage() . "\n";
    exit(1);
}

echo "\n=== Test Summary ===\n";
echo "✅ API is accessible and returns valid JSON data\n";
echo "✅ Data structure is compatible with import process\n";
echo "✅ " . count($insertData) . " records would be imported\n";
echo "\nThe API connection and data import simulation completed successfully!\n";
echo "If the database is accessible, the import should work properly.\n"; 