<?php

require_once 'vendor/autoload.php';

$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\ApiTarget;
use App\Models\ApiSendLog;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;

echo "=== TEST API TARGET WITH AUTHENTICATION ===\n\n";

try {
    // Get the test API target
    $apiTarget = ApiTarget::where('name', 'Test API Target')->first();
    
    if (!$apiTarget) {
        echo "❌ Test API Target not found!\n";
        exit(1);
    }
    
    echo "✓ Found API Target: {$apiTarget->name}\n";
    echo "  URL: {$apiTarget->url}\n";
    echo "  Method: {$apiTarget->method}\n";
    echo "  Auth Type: {$apiTarget->auth_type}\n";
    echo "  Source Table: {$apiTarget->source_table}\n\n";
    
    // Get data from source table (users)
    $data = DB::table('users')->get();
    echo "✓ Found {$data->count()} records in users table\n";
    
    // Prepare headers based on authentication type
    $headers = [
        'Content-Type' => 'application/json',
        'Accept' => 'application/json',
        'User-Agent' => 'API8-Test/1.0'
    ];
    
    // Add authentication headers
    switch ($apiTarget->auth_type) {
        case 'bearer_token':
            $headers['Authorization'] = 'Bearer ' . $apiTarget->auth_token;
            echo "✓ Added Bearer Token authentication\n";
            break;
        case 'api_key':
            $headers['X-API-Key'] = $apiTarget->auth_token;
            echo "✓ Added API Key authentication\n";
            break;
        case 'basic_auth':
            $headers['Authorization'] = 'Basic ' . base64_encode($apiTarget->auth_username . ':' . $apiTarget->auth_password);
            echo "✓ Added Basic Auth authentication\n";
            break;
        case 'none':
            echo "✓ No authentication required\n";
            break;
    }
    
    echo "✓ Headers prepared\n";
    
    // Send request
    echo "\n🔄 Sending data to {$apiTarget->url}...\n";
    
    $response = Http::withHeaders($headers)
        ->timeout(30)
        ->post($apiTarget->url, $data->toArray());
    
    echo "✓ Response received!\n";
    echo "  Status: {$response->status()}\n";
    echo "  Success: " . ($response->successful() ? 'Yes' : 'No') . "\n";
    
    // Create log entry
    $log = ApiSendLog::create([
        'api_target_id' => $apiTarget->id,
        'success' => $response->successful(),
        'status_code' => $response->status(),
        'response_time' => $response->handlerStats()['total_time'] ?? 0,
        'payload' => $data->toArray(),
        'records_sent' => $data->count(),
        'error_message' => $response->successful() ? null : $response->body(),
        'response_body' => $response->body(),
        'sent_at' => now()
    ]);
    
    echo "✓ Log entry created (ID: {$log->id})\n";
    
    // Update last_sent_at
    $apiTarget->update(['last_sent_at' => now()]);
    echo "✓ Last sent time updated\n";
    
    // Show response details
    echo "\n📋 Response Details:\n";
    echo "Headers:\n";
    foreach ($response->headers() as $key => $values) {
        echo "  {$key}: " . implode(', ', $values) . "\n";
    }
    
    echo "\nBody (first 500 chars):\n";
    $body = $response->body();
    echo substr($body, 0, 500) . (strlen($body) > 500 ? '...' : '') . "\n";
    
    // Show authentication info
    echo "\n🔐 Authentication Details:\n";
    echo "Type: {$apiTarget->auth_type}\n";
    if ($apiTarget->auth_token) {
        echo "Token: " . substr($apiTarget->auth_token, 0, 20) . "...\n";
    }
    if ($apiTarget->auth_username) {
        echo "Username: {$apiTarget->auth_username}\n";
    }
    
    echo "\n=== SUCCESS! ===\n";
    echo "API target with authentication tested successfully!\n";
    echo "Check the logs at: http://127.0.0.1:8000/api-targets/{$apiTarget->id}/logs\n";
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
} 