<?php
/**
 * Test Database Connection and Table Creation
 */

require_once 'vendor/autoload.php';

// Bootstrap Laravel
$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

echo "=== Database Connection Test ===\n\n";

try {
    // Test database connection
    echo "1. Testing database connection...\n";
    DB::connection()->getPdo();
    echo "✅ Database connection successful!\n";
    echo "   Database: " . DB::connection()->getDatabaseName() . "\n";
    echo "   Driver: " . DB::connection()->getDriverName() . "\n";
    
} catch (Exception $e) {
    echo "❌ Database connection failed: " . $e->getMessage() . "\n";
    echo "Please check your database configuration in .env file\n";
    exit(1);
}

// Check if TEST table exists
echo "\n2. Checking TEST table...\n";
if (Schema::hasTable('TEST')) {
    echo "✅ TEST table exists\n";
    $recordCount = DB::table('TEST')->count();
    echo "   Current records: $recordCount\n";
    
    if ($recordCount > 0) {
        echo "   Sample records:\n";
        $sampleRecords = DB::table('TEST')->limit(3)->get();
        foreach ($sampleRecords as $record) {
            echo "   - ID: {$record->id}, Type: {$record->type}, Sell: {$record->sell}, Buy: {$record->buy}\n";
        }
    }
} else {
    echo "❌ TEST table does not exist\n";
    echo "   Creating TEST table...\n";
    
    try {
        Schema::create('TEST', function ($table) {
            $table->id();
            $table->decimal('sell', 15, 2)->nullable();
            $table->decimal('buy', 15, 2)->nullable();
            $table->string('type')->nullable();
            $table->text('info')->nullable();
            $table->decimal('weight', 10, 2)->nullable();
            $table->string('unit')->nullable();
            $table->string('source')->default('api_import');
            $table->longText('raw_data')->nullable();
            $table->timestamps();
        });
        
        echo "✅ TEST table created successfully!\n";
    } catch (Exception $e) {
        echo "❌ Failed to create TEST table: " . $e->getMessage() . "\n";
        exit(1);
    }
}

// Test data insertion
echo "\n3. Testing data insertion...\n";
try {
    $testData = [
        'sell' => 1885000,
        'buy' => 1770000,
        'type' => 'antam',
        'info' => 'Test data from database connection test',
        'weight' => 100,
        'unit' => 'gram',
        'source' => 'test_script',
        'raw_data' => json_encode(['test' => 'data']),
        'created_at' => now(),
        'updated_at' => now()
    ];
    
    DB::table('TEST')->insert($testData);
    echo "✅ Test data inserted successfully!\n";
    
    // Verify insertion
    $newCount = DB::table('TEST')->count();
    echo "   Total records after insertion: $newCount\n";
    
} catch (Exception $e) {
    echo "❌ Data insertion failed: " . $e->getMessage() . "\n";
    exit(1);
}

echo "\n=== Database Test Summary ===\n";
echo "✅ Database connection is working\n";
echo "✅ TEST table is ready for data import\n";
echo "✅ Data insertion is working\n";
echo "\nThe database is ready for API data import!\n"; 