<?php

require_once 'vendor/autoload.php';

// Bootstrap Laravel
$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\ApiSource;
use App\Services\DataImportService;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

echo "=== TESTING DATA IMPORT SYSTEM ===\n\n";

try {
    // Test 1: Check API Sources
    echo "1. Checking API Sources...\n";
    $sources = ApiSource::all(['id', 'name', 'url', 'table_name', 'status']);
    
    if ($sources->count() > 0) {
        echo "✓ Found {$sources->count()} API sources\n";
        foreach ($sources as $source) {
            echo "   - ID: {$source->id}, Name: {$source->name}, Table: {$source->table_name}, Status: {$source->status}\n";
        }
    } else {
        echo "✗ No API sources found\n";
        exit(1);
    }
    
    echo "\n";
    
    // Test 2: Test API Connection
    echo "2. Testing API Connection...\n";
    $testSource = $sources->first();
    echo "Testing with source: {$testSource->name} (ID: {$testSource->id})\n";
    echo "URL: {$testSource->url}\n";
    
    try {
        $response = Http::timeout(30)->get($testSource->url);
        echo "Response Status: {$response->status()}\n";
        
        if ($response->successful()) {
            $data = $response->json();
            echo "✓ API connection successful\n";
            echo "Response type: " . gettype($data) . "\n";
            
            if (is_array($data)) {
                echo "Response keys: " . implode(', ', array_keys($data)) . "\n";
                if (isset($data['data'])) {
                    echo "Data count: " . count($data['data']) . "\n";
                } elseif (isset($data[0])) {
                    echo "Array count: " . count($data) . "\n";
                }
            }
        } else {
            echo "✗ API connection failed\n";
        }
    } catch (\Exception $e) {
        echo "✗ API connection error: " . $e->getMessage() . "\n";
    }
    
    echo "\n";
    
    // Test 3: Test DataImportService
    echo "3. Testing DataImportService...\n";
    $importService = new DataImportService();
    
    // Test getImportStats
    $stats = $importService->getImportStats($testSource);
    echo "Import Stats:\n";
    echo "  - Total records: {$stats['total_records']}\n";
    echo "  - Today's records: {$stats['today_records']}\n";
    echo "  - Table exists: " . ($stats['table_exists'] ? 'Yes' : 'No') . "\n";
    echo "  - Message: {$stats['message']}\n";
    
    echo "\n";
    
    // Test 4: Test Import Process
    echo "4. Testing Import Process...\n";
    
    // Check if table exists
    if (!Schema::hasTable($testSource->table_name)) {
        echo "✗ Table '{$testSource->table_name}' does not exist\n";
        echo "Creating table...\n";
        
        // Create TEST table if it doesn't exist
        if ($testSource->table_name === 'TEST') {
            Schema::create('TEST', function ($table) {
                $table->id();
                $table->decimal('sell', 15, 2)->nullable();
                $table->decimal('buy', 15, 2)->nullable();
                $table->string('type')->nullable();
                $table->text('info')->nullable();
                $table->decimal('weight', 10, 2)->nullable();
                $table->string('unit')->nullable();
                $table->string('source')->nullable();
                $table->json('raw_data')->nullable();
                $table->timestamps();
            });
            echo "✓ Created TEST table\n";
        }
    } else {
        echo "✓ Table '{$testSource->table_name}' exists\n";
    }
    
    // Test import
    echo "Attempting import...\n";
    $result = $importService->importFromSource($testSource, true);
    
    if ($result['success']) {
        echo "✓ Import successful!\n";
        echo "  - Message: {$result['message']}\n";
        echo "  - Imported count: {$result['imported_count']}\n";
        if (isset($result['mapped_fields'])) {
            echo "  - Mapped fields: " . implode(', ', $result['mapped_fields']) . "\n";
        }
    } else {
        echo "✗ Import failed: {$result['message']}\n";
    }
    
    echo "\n";
    
    // Test 5: Verify Imported Data
    echo "5. Verifying Imported Data...\n";
    $newStats = $importService->getImportStats($testSource);
    echo "After import:\n";
    echo "  - Total records: {$newStats['total_records']}\n";
    echo "  - Today's records: {$newStats['today_records']}\n";
    
    if ($newStats['total_records'] > 0) {
        echo "✓ Data imported successfully!\n";
        
        // Show sample data
        $sampleData = DB::table($testSource->table_name)->limit(3)->get();
        echo "Sample data:\n";
        foreach ($sampleData as $record) {
            echo "  - ID: {$record->id}, Type: {$record->type}, Sell: {$record->sell}, Buy: {$record->buy}\n";
        }
    } else {
        echo "✗ No data was imported\n";
    }
    
    echo "\n";
    
    // Test 6: Test API Logs
    echo "6. Checking API Logs...\n";
    $logs = DB::table('api_logs')->where('api_source_id', $testSource->id)->orderBy('created_at', 'desc')->limit(5)->get();
    echo "Recent API logs: {$logs->count()}\n";
    
    foreach ($logs as $log) {
        echo "  - {$log->created_at}: Status {$log->status_code}, Time {$log->response_time}ms\n";
    }
    
    echo "\n";
    
    // Summary
    echo "=== IMPORT TEST SUMMARY ===\n";
    echo "✓ API Sources: " . $sources->count() . " found\n";
    echo "✓ API Connection: " . ($response->successful() ? 'Working' : 'Failed') . "\n";
    echo "✓ DataImportService: Working\n";
    echo "✓ Import Process: " . ($result['success'] ? 'Successful' : 'Failed') . "\n";
    echo "✓ Data Verification: " . ($newStats['total_records'] > 0 ? 'Data imported' : 'No data') . "\n";
    echo "✓ API Logs: " . $logs->count() . " recent logs\n";
    
    if ($result['success'] && $newStats['total_records'] > 0) {
        echo "\n🎉 SUCCESS: Data import system is working correctly!\n";
        echo "You can now use the web interface to import data automatically.\n";
    } else {
        echo "\n⚠️  WARNING: Some issues detected. Check the logs above.\n";
    }
    
} catch (\Exception $e) {
    echo "✗ Error during testing: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}

echo "\n=== TEST COMPLETED ===\n"; 