<?php
/**
 * Test Response Format Fix
 * 
 * Menguji GET request dengan body data untuk memastikan response format sudah benar
 */

// Konfigurasi
$url = 'https://your-domain.com/api/in/test-receiver'; // Ganti dengan URL Anda

echo "🚀 Testing Response Format Fix\n";
echo "URL: {$url}\n\n";

// Test cases
$testCases = [
    [
        'name' => 'Test 1: Ambil data customer',
        'body' => ['data' => 'customer']
    ],
    [
        'name' => 'Test 2: Ambil data customer dengan limit',
        'body' => ['data' => 'customer', 'limit' => 3]
    ],
    [
        'name' => 'Test 3: Tabel tidak ada (error test)',
        'body' => ['data' => 'table_not_exists']
    ]
];

// Function untuk melakukan request
function makeRequest($url, $body) {
    $ch = curl_init();
    
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'GET');
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    return [
        'http_code' => $httpCode,
        'response' => $response,
        'error' => $error
    ];
}

// Function untuk menampilkan hasil
function displayResult($testName, $result) {
    echo str_repeat("=", 60) . "\n";
    echo "TEST: {$testName}\n";
    echo str_repeat("=", 60) . "\n";
    
    if ($result['error']) {
        echo "❌ CURL ERROR: " . $result['error'] . "\n";
        return;
    }
    
    echo "📡 HTTP Code: " . $result['http_code'] . "\n";
    
    if ($result['http_code'] >= 200 && $result['http_code'] < 300) {
        echo "✅ SUCCESS\n";
    } else {
        echo "❌ FAILED\n";
    }
    
    echo "\n📄 Response:\n";
    $responseData = json_decode($result['response'], true);
    if ($responseData) {
        echo json_encode($responseData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        
        // Check response format
        if (isset($responseData['sukses'])) {
            echo "\n\n✅ FORMAT CHECK: Response menggunakan format 'sukses' ✓\n";
            
            if ($responseData['sukses'] === true) {
                if (isset($responseData['datas'])) {
                    echo "✅ DATA CHECK: Response memiliki field 'datas' ✓\n";
                    echo "📊 Data count: " . count($responseData['datas']) . " records\n";
                } else {
                    echo "❌ DATA CHECK: Response tidak memiliki field 'datas'\n";
                }
            } else {
                if (isset($responseData['error'])) {
                    echo "✅ ERROR CHECK: Response memiliki field 'error' ✓\n";
                } else {
                    echo "❌ ERROR CHECK: Response tidak memiliki field 'error'\n";
                }
            }
        } else {
            echo "\n❌ FORMAT CHECK: Response tidak menggunakan format 'sukses'\n";
        }
    } else {
        echo $result['response'];
    }
    echo "\n\n";
}

// Jalankan semua test
foreach ($testCases as $testCase) {
    $result = makeRequest($url, $testCase['body']);
    displayResult($testCase['name'], $result);
}

echo str_repeat("=", 60) . "\n";
echo "✅ Testing selesai!\n";
echo str_repeat("=", 60) . "\n";

// Contoh curl command
echo "\n📋 Contoh curl command:\n";
echo "curl -X GET '{$url}' \\\n";
echo "  -H 'Content-Type: application/json' \\\n";
echo "  -H 'Accept: application/json' \\\n";
echo "  -d '{\"data\": \"customer\"}'\n";

echo "\n📋 Expected Response Format:\n";
echo "{\n";
echo "  \"sukses\": true,\n";
echo "  \"datas\": [\n";
echo "    {\n";
echo "      \"nama\": \"sulaiman\",\n";
echo "      \"Alamat\": \"Bogor\",\n";
echo "      \"Gender\": \"Male\"\n";
echo "    }\n";
echo "  ]\n";
echo "}\n";
?>
