<?php
/**
 * Test GET Request dengan Response Format Sederhana
 * 
 * Contoh request:
 * GET /api/in/{endpoint} dengan body: {"data": "customer"}
 * 
 * Response yang diharapkan:
 * {
 *   "sukses": true,
 *   "datas": [
 *     {
 *       "nama": "sulaiman",
 *       "Alamat": "Bogor",
 *       "Gender": "Male"
 *     },
 *     {
 *       "nama": "Daud",
 *       "Alamat": "Jakarta",
 *       "Gender": "Male"
 *     }
 *   ]
 * }
 */

// Konfigurasi
$baseUrl = 'https://your-domain.com'; // Ganti dengan domain Anda
$endpoint = 'test-receiver'; // Ganti dengan endpoint receiver Anda
$url = $baseUrl . '/api/in/' . $endpoint;

// Test cases
$testCases = [
    [
        'name' => 'Test 1: Ambil data customer',
        'body' => [
            'data' => 'customer'
        ]
    ],
    [
        'name' => 'Test 2: Ambil data customer dengan limit',
        'body' => [
            'data' => 'customer',
            'limit' => 3
        ]
    ],
    [
        'name' => 'Test 3: Ambil data customer dengan filter',
        'body' => [
            'data' => 'customer',
            'Gender' => 'Male'
        ]
    ],
    [
        'name' => 'Test 4: Ambil data customer dengan multiple filter',
        'body' => [
            'data' => 'customer',
            'Gender' => 'Male',
            'Alamat' => 'Jakarta'
        ]
    ],
    [
        'name' => 'Test 5: Ambil data dari tabel lain',
        'body' => [
            'data' => 'users'
        ]
    ]
];

// Function untuk melakukan request
function makeRequest($url, $body) {
    $ch = curl_init();
    
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'GET');
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    return [
        'http_code' => $httpCode,
        'response' => $response,
        'error' => $error
    ];
}

// Function untuk menampilkan hasil
function displayResult($testName, $result) {
    echo "\n" . str_repeat("=", 80) . "\n";
    echo "TEST: {$testName}\n";
    echo str_repeat("=", 80) . "\n";
    
    if ($result['error']) {
        echo "❌ ERROR: " . $result['error'] . "\n";
        return;
    }
    
    echo "📡 HTTP Code: " . $result['http_code'] . "\n";
    
    if ($result['http_code'] >= 200 && $result['http_code'] < 300) {
        echo "✅ SUCCESS\n";
    } else {
        echo "❌ FAILED\n";
    }
    
    echo "\n📄 Response:\n";
    $responseData = json_decode($result['response'], true);
    if ($responseData) {
        echo json_encode($responseData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    } else {
        echo $result['response'];
    }
    echo "\n";
}

// Jalankan semua test
echo "🚀 Testing GET Request dengan Response Format Sederhana\n";
echo "URL: {$url}\n";

foreach ($testCases as $testCase) {
    $result = makeRequest($url, $testCase['body']);
    displayResult($testCase['name'], $result);
}

echo "\n" . str_repeat("=", 80) . "\n";
echo "✅ Testing selesai!\n";
echo str_repeat("=", 80) . "\n";

// Contoh penggunaan dengan curl command
echo "\n📋 Contoh curl commands:\n";
echo str_repeat("-", 50) . "\n";

foreach ($testCases as $index => $testCase) {
    $curlCommand = "curl -X GET '{$url}' \\\n";
    $curlCommand .= "  -H 'Content-Type: application/json' \\\n";
    $curlCommand .= "  -H 'Accept: application/json' \\\n";
    $curlCommand .= "  -d '" . json_encode($testCase['body']) . "'";
    
    echo "Test " . ($index + 1) . ":\n";
    echo $curlCommand . "\n\n";
}

// Contoh response yang diharapkan
echo "📋 Contoh Response yang Diharapkan:\n";
echo str_repeat("-", 50) . "\n";
echo json_encode([
    'sukses' => true,
    'datas' => [
        [
            'nama' => 'sulaiman',
            'Alamat' => 'Bogor',
            'Gender' => 'Male'
        ],
        [
            'nama' => 'Daud',
            'Alamat' => 'Jakarta',
            'Gender' => 'Male'
        ],
        [
            'nama' => 'Sarah',
            'Alamat' => 'Bandung',
            'Gender' => 'Female'
        ]
    ]
], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

echo "\n\n";

// Contoh penggunaan dengan JavaScript
echo "📋 Contoh JavaScript:\n";
echo str_repeat("-", 50) . "\n";
echo "fetch('{$url}', {\n";
echo "  method: 'GET',\n";
echo "  headers: {\n";
echo "    'Content-Type': 'application/json',\n";
echo "    'Accept': 'application/json'\n";
echo "  },\n";
echo "  body: JSON.stringify({\n";
echo "    data: 'customer'\n";
echo "  })\n";
echo "})\n";
echo ".then(response => response.json())\n";
echo ".then(data => {\n";
echo "  if (data.sukses) {\n";
echo "    console.log('Data customer:', data.datas);\n";
echo "    data.datas.forEach(customer => {\n";
echo "      console.log(customer.nama + ' - ' + customer.Alamat);\n";
echo "    });\n";
echo "  }\n";
echo "});\n";

echo "\n\n";
?>
