<?php
/**
 * Test Response JSON untuk Data Tabel
 * 
 * Menguji GET request dengan body yang berisi nama tabel
 * dan memastikan response JSON menampilkan data tabel dengan benar
 */

// Konfigurasi
$baseUrl = 'https://your-domain.com'; // Ganti dengan domain Anda
$endpoint = 'test-receiver'; // Ganti dengan endpoint receiver Anda
$url = $baseUrl . '/api/in/' . $endpoint;

// Test cases untuk berbagai tabel
$testCases = [
    [
        'name' => 'Test 1: Ambil data dari tabel customer',
        'body' => [
            'data' => 'customer',
            'limit' => 3
        ],
        'expected_structure' => [
            'success' => true,
            'message' => 'Data retrieved successfully',
            'received_at' => 'string',
            'data' => [
                'table' => 'customer',
                'total_records' => 'number',
                'returned_records' => 'number',
                'pagination' => [
                    'limit' => 'number',
                    'offset' => 'number',
                    'sort' => 'string',
                    'order' => 'string'
                ],
                'records' => 'array'
            ]
        ]
    ],
    [
        'name' => 'Test 2: Ambil data dari tabel users',
        'body' => [
            'data' => 'users',
            'limit' => 5,
            'sort' => 'name',
            'order' => 'asc'
        ]
    ],
    [
        'name' => 'Test 3: Ambil data dengan filter',
        'body' => [
            'data' => 'customer',
            'limit' => 2,
            'status' => 'active'
        ]
    ],
    [
        'name' => 'Test 4: Ambil data dengan pagination',
        'body' => [
            'data' => 'customer',
            'limit' => 1,
            'offset' => 1
        ]
    ],
    [
        'name' => 'Test 5: Ambil data dengan debug mode',
        'body' => [
            'data' => 'customer',
            'limit' => 2
        ],
        'headers' => [
            'X-Debug: true'
        ]
    ]
];

// Function untuk melakukan request
function makeRequest($url, $body, $headers = []) {
    $ch = curl_init();
    
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'GET');
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
    curl_setopt($ch, CURLOPT_HTTPHEADER, array_merge([
        'Content-Type: application/json',
        'Accept: application/json'
    ], $headers));
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    return [
        'http_code' => $httpCode,
        'response' => $response,
        'error' => $error
    ];
}

// Function untuk validasi struktur response
function validateResponseStructure($responseData, $expectedStructure) {
    $errors = [];
    
    foreach ($expectedStructure as $key => $expectedType) {
        if (!isset($responseData[$key])) {
            $errors[] = "Missing key: {$key}";
            continue;
        }
        
        $actualValue = $responseData[$key];
        $actualType = gettype($actualValue);
        
        if ($expectedType === 'string' && $actualType !== 'string') {
            $errors[] = "Key '{$key}' should be string, got {$actualType}";
        } elseif ($expectedType === 'number' && !is_numeric($actualValue)) {
            $errors[] = "Key '{$key}' should be number, got {$actualType}";
        } elseif ($expectedType === 'array' && $actualType !== 'array') {
            $errors[] = "Key '{$key}' should be array, got {$actualType}";
        } elseif ($expectedType === 'boolean' && $actualType !== 'boolean') {
            $errors[] = "Key '{$key}' should be boolean, got {$actualType}";
        }
    }
    
    return $errors;
}

// Function untuk menampilkan hasil
function displayResult($testName, $result, $expectedStructure = null) {
    echo "\n" . str_repeat("=", 80) . "\n";
    echo "TEST: {$testName}\n";
    echo str_repeat("=", 80) . "\n";
    
    if ($result['error']) {
        echo "❌ ERROR: " . $result['error'] . "\n";
        return;
    }
    
    echo "📡 HTTP Code: " . $result['http_code'] . "\n";
    
    if ($result['http_code'] >= 200 && $result['http_code'] < 300) {
        echo "✅ SUCCESS\n";
    } else {
        echo "❌ FAILED\n";
    }
    
    echo "\n📄 Response:\n";
    $responseData = json_decode($result['response'], true);
    if ($responseData) {
        echo json_encode($responseData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        
        // Validasi struktur jika ada
        if ($expectedStructure) {
            echo "\n\n🔍 Structure Validation:\n";
            $validationErrors = validateResponseStructure($responseData, $expectedStructure);
            
            if (empty($validationErrors)) {
                echo "✅ Response structure is valid\n";
            } else {
                echo "❌ Structure validation failed:\n";
                foreach ($validationErrors as $error) {
                    echo "   - {$error}\n";
                }
            }
        }
        
        // Analisis data
        if (isset($responseData['data']) && isset($responseData['data']['records'])) {
            echo "\n📊 Data Analysis:\n";
            echo "   - Table: " . ($responseData['data']['table'] ?? 'N/A') . "\n";
            echo "   - Total Records: " . ($responseData['data']['total_records'] ?? 'N/A') . "\n";
            echo "   - Returned Records: " . ($responseData['data']['returned_records'] ?? 'N/A') . "\n";
            echo "   - Records Count: " . count($responseData['data']['records']) . "\n";
            
            if (!empty($responseData['data']['records'])) {
                echo "   - Sample Record Keys: " . implode(', ', array_keys($responseData['data']['records'][0])) . "\n";
            }
        }
        
    } else {
        echo $result['response'];
    }
    echo "\n";
}

// Jalankan semua test
echo "🚀 Testing Response JSON untuk Data Tabel\n";
echo "URL: {$url}\n";

foreach ($testCases as $testCase) {
    $result = makeRequest($url, $testCase['body'], $testCase['headers'] ?? []);
    $expectedStructure = $testCase['expected_structure'] ?? null;
    displayResult($testCase['name'], $result, $expectedStructure);
}

echo "\n" . str_repeat("=", 80) . "\n";
echo "✅ Testing selesai!\n";
echo str_repeat("=", 80) . "\n";

// Contoh response yang diharapkan
echo "\n📋 Contoh Response yang Diharapkan:\n";
echo str_repeat("-", 50) . "\n";
echo json_encode([
    'success' => true,
    'message' => 'Data retrieved successfully',
    'received_at' => '2024-01-15T10:30:45.123456Z',
    'data' => [
        'table' => 'customer',
        'total_records' => 150,
        'returned_records' => 3,
        'pagination' => [
            'limit' => 3,
            'offset' => 0,
            'sort' => 'id',
            'order' => 'desc'
        ],
        'records' => [
            [
                'id' => 1,
                'nama' => 'John Doe',
                'email' => 'john@example.com',
                'status' => 'active',
                'created_at' => '2024-01-15T10:30:45.000000Z',
                'updated_at' => '2024-01-15T10:30:45.000000Z'
            ],
            [
                'id' => 2,
                'nama' => 'Jane Smith',
                'email' => 'jane@example.com',
                'status' => 'active',
                'created_at' => '2024-01-15T10:30:45.000000Z',
                'updated_at' => '2024-01-15T10:30:45.000000Z'
            ],
            [
                'id' => 3,
                'nama' => 'Bob Johnson',
                'email' => 'bob@example.com',
                'status' => 'inactive',
                'created_at' => '2024-01-15T10:30:45.000000Z',
                'updated_at' => '2024-01-15T10:30:45.000000Z'
            ]
        ]
    ]
], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

echo "\n\n";

// Contoh curl commands
echo "📋 Contoh Curl Commands:\n";
echo str_repeat("-", 50) . "\n";

foreach ($testCases as $index => $testCase) {
    $curlCommand = "curl -X GET '{$url}' \\\n";
    $curlCommand .= "  -H 'Content-Type: application/json' \\\n";
    $curlCommand .= "  -H 'Accept: application/json' \\\n";
    
    if (isset($testCase['headers'])) {
        foreach ($testCase['headers'] as $header) {
            $curlCommand .= "  -H '{$header}' \\\n";
        }
    }
    
    $curlCommand .= "  -d '" . json_encode($testCase['body']) . "'";
    
    echo "Test " . ($index + 1) . ":\n";
    echo $curlCommand . "\n\n";
}

echo "🎯 Tips Penggunaan:\n";
echo "1. Pastikan tabel yang diminta ada di database\n";
echo "2. Gunakan parameter 'limit' untuk membatasi jumlah data\n";
echo "3. Gunakan parameter 'offset' untuk pagination\n";
echo "4. Gunakan parameter 'sort' dan 'order' untuk sorting\n";
echo "5. Tambahkan filter berdasarkan kolom tabel\n";
echo "6. Gunakan header 'X-Debug: true' untuk informasi debug\n";
?>
