<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\ApiTarget;
use App\Models\ApiSendLog;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;
use PHPUnit\Framework\Attributes\Test;

class ApiTargetTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    protected $user;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create();
    }

    /** @test */
    public function user_can_view_api_targets_index()
    {
        $this->actingAs($this->user);
        
        $response = $this->get(route('api-targets.index'));
        
        $response->assertStatus(200);
        $response->assertViewIs('api-targets.index');
        $response->assertViewHas('api_targets');
    }

    /** @test */
    public function user_can_view_create_api_target_form()
    {
        $this->actingAs($this->user);
        
        $response = $this->get(route('api-targets.create'));
        
        $response->assertStatus(200);
        $response->assertViewIs('api-targets.create');
        $response->assertViewHas('tables');
    }

    /** @test */
    public function user_can_create_api_target()
    {
        $this->actingAs($this->user);
        
        $data = [
            'name' => 'Test API Target',
            'description' => 'Test description',
            'base_url' => 'https://api.test.com',
            'endpoint' => '/webhook/data',
            'method' => 'POST',
            'source_table' => 'users',
            'auth_type' => 'none',
            'schedule_type' => 'manual',
            'is_active' => true
        ];
        
        $response = $this->post(route('api-targets.store'), $data);
        
        $response->assertRedirect(route('api-targets.index'));
        $response->assertSessionHas('success');
        
        $this->assertDatabaseHas('api_targets', [
            'name' => 'Test API Target',
            'base_url' => 'https://api.test.com',
            'endpoint' => '/webhook/data',
            'method' => 'POST',
            'source_table' => 'users'
        ]);
    }

    /** @test */
    public function api_target_creation_validates_required_fields()
    {
        $this->actingAs($this->user);
        
        $response = $this->post(route('api-targets.store'), []);
        
        $response->assertSessionHasErrors([
            'name',
            'base_url',
            'endpoint',
            'method',
            'source_table',
            'auth_type',
            'schedule_type'
        ]);
    }

    /** @test */
    public function api_target_creation_validates_url_format()
    {
        $this->actingAs($this->user);
        
        $data = [
            'name' => 'Test API Target',
            'base_url' => 'invalid-url',
            'endpoint' => '/webhook/data',
            'method' => 'POST',
            'source_table' => 'users',
            'auth_type' => 'none',
            'schedule_type' => 'manual'
        ];
        
        $response = $this->post(route('api-targets.store'), $data);
        
        $response->assertSessionHasErrors(['base_url']);
    }

    /** @test */
    public function user_can_view_api_target_details()
    {
        $this->actingAs($this->user);
        
        $apiTarget = ApiTarget::factory()->create([
            'created_by' => $this->user->id
        ]);
        
        $response = $this->get(route('api-targets.show', $apiTarget));
        
        $response->assertStatus(200);
        $response->assertViewIs('api-targets.show');
        $response->assertViewHas('apiTarget', $apiTarget);
    }

    /** @test */
    public function user_can_edit_api_target()
    {
        $this->actingAs($this->user);
        
        $apiTarget = ApiTarget::factory()->create([
            'created_by' => $this->user->id
        ]);
        
        $response = $this->get(route('api-targets.edit', $apiTarget));
        
        $response->assertStatus(200);
        $response->assertViewIs('api-targets.edit');
        $response->assertViewHas('apiTarget', $apiTarget);
        $response->assertViewHas('tables');
    }

    /** @test */
    public function user_can_update_api_target()
    {
        $this->actingAs($this->user);
        
        $apiTarget = ApiTarget::factory()->create([
            'created_by' => $this->user->id
        ]);
        
        $updateData = [
            'name' => 'Updated API Target',
            'description' => 'Updated description',
            'base_url' => 'https://api.updated.com',
            'endpoint' => '/updated/endpoint',
            'method' => 'PUT',
            'source_table' => 'api_sources',
            'auth_type' => 'api_key',
            'auth_token' => 'new-api-key',
            'schedule_type' => 'hourly',
            'schedule_frequency' => 2,
            'is_active' => false
        ];
        
        $response = $this->put(route('api-targets.update', $apiTarget), $updateData);
        
        $response->assertRedirect(route('api-targets.index'));
        $response->assertSessionHas('success');
        
        $this->assertDatabaseHas('api_targets', [
            'id' => $apiTarget->id,
            'name' => 'Updated API Target',
            'base_url' => 'https://api.updated.com',
            'endpoint' => '/updated/endpoint',
            'method' => 'PUT'
        ]);
    }

    /** @test */
    public function user_can_delete_api_target()
    {
        $this->actingAs($this->user);
        
        $apiTarget = ApiTarget::factory()->create([
            'created_by' => $this->user->id
        ]);
        
        $response = $this->delete(route('api-targets.destroy', $apiTarget));
        
        $response->assertRedirect(route('api-targets.index'));
        $response->assertSessionHas('success');
        
        $this->assertDatabaseMissing('api_targets', ['id' => $apiTarget->id]);
    }

    /** @test */
    public function api_target_url_attribute_combines_base_url_and_endpoint()
    {
        $apiTarget = ApiTarget::factory()->create([
            'base_url' => 'https://api.example.com',
            'endpoint' => '/webhook/data'
        ]);
        
        $this->assertEquals('https://api.example.com/webhook/data', $apiTarget->url);
    }

    /** @test */
    public function api_target_url_handles_trailing_slashes_correctly()
    {
        $apiTarget = ApiTarget::factory()->create([
            'base_url' => 'https://api.example.com/',
            'endpoint' => '/webhook/data'
        ]);
        
        $this->assertEquals('https://api.example.com/webhook/data', $apiTarget->url);
    }

    /** @test */
    public function user_can_view_api_target_logs()
    {
        $this->actingAs($this->user);
        
        $apiTarget = ApiTarget::factory()->create([
            'created_by' => $this->user->id
        ]);
        
        // Create some test logs
        ApiSendLog::factory()->count(3)->create([
            'api_target_id' => $apiTarget->id
        ]);
        
        $response = $this->get(route('api-targets.logs', $apiTarget));
        
        $response->assertStatus(200);
        $response->assertViewIs('api-targets.logs');
        $response->assertViewHas('apiTarget', $apiTarget);
        $response->assertViewHas('logs');
    }

    /** @test */
    public function user_can_view_api_target_documentation()
    {
        $this->actingAs($this->user);
        
        $apiTarget = ApiTarget::factory()->create([
            'created_by' => $this->user->id,
            'source_table' => 'users'
        ]);
        
        $response = $this->get(route('api-targets.documentation', $apiTarget));
        
        $response->assertStatus(200);
        $response->assertViewIs('api-targets.documentation');
        $response->assertViewHas('apiTarget', $apiTarget);
        $response->assertViewHas('documentation');
    }

    /** @test */
    public function test_connection_endpoint_works()
    {
        $this->actingAs($this->user);
        
        Http::fake([
            'https://api.test.com/*' => Http::response(['status' => 'ok'], 200)
        ]);
        
        $response = $this->post(route('api-targets.test-connection'), [
            'base_url' => 'https://api.test.com'
        ]);
        
        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
    }

    /** @test */
    public function test_connection_fails_for_invalid_url()
    {
        $this->actingAs($this->user);
        
        Http::fake([
            'https://invalid-url.com/*' => Http::response([], 404)
        ]);
        
        $response = $this->post(route('api-targets.test-connection'), [
            'base_url' => 'https://invalid-url.com'
        ]);
        
        $response->assertStatus(200);
        $response->assertJson(['success' => false]);
    }

    /** @test */
    public function table_fields_endpoint_returns_table_structure()
    {
        $this->actingAs($this->user);
        
        $response = $this->post(route('api-targets.table-fields'), [
            'table' => 'users'
        ]);
        
        $response->assertStatus(200);
        $response->assertJsonStructure(['fields']);
    }

    /** @test */
    public function api_target_send_method_creates_log_entry()
    {
        $this->actingAs($this->user);
        
        // Create test data in users table
        $user = User::factory()->create();
        
        $apiTarget = ApiTarget::factory()->create([
            'created_by' => $this->user->id,
            'source_table' => 'users',
            'base_url' => 'https://api.test.com',
            'endpoint' => '/webhook',
            'method' => 'POST'
        ]);
        
        Http::fake([
            'https://api.test.com/webhook' => Http::response(['status' => 'success'], 200)
        ]);
        
        $response = $this->post(route('api-targets.send', $apiTarget));
        
        $response->assertRedirect();
        
        $this->assertDatabaseHas('api_send_logs', [
            'api_target_id' => $apiTarget->id,
            'status' => 'success'
        ]);
    }

    /** @test */
    public function api_target_send_method_handles_errors()
    {
        $this->actingAs($this->user);
        
        $apiTarget = ApiTarget::factory()->create([
            'created_by' => $this->user->id,
            'source_table' => 'non_existent_table',
            'base_url' => 'https://api.test.com',
            'endpoint' => '/webhook',
            'method' => 'POST'
        ]);
        
        $response = $this->post(route('api-targets.send', $apiTarget));
        
        $response->assertRedirect();
        $response->assertSessionHas('warning');
        
        $this->assertDatabaseHas('api_send_logs', [
            'api_target_id' => $apiTarget->id,
            'status' => 'error'
        ]);
    }

    /** @test */
    public function api_target_with_basic_auth_includes_headers()
    {
        $apiTarget = ApiTarget::factory()->create([
            'auth_type' => 'basic_auth',
            'auth_username' => 'testuser',
            'auth_password' => 'testpass'
        ]);
        
        $controller = new \App\Http\Controllers\ApiTargetController();
        $headers = $controller->prepareHeaders($apiTarget);
        
        $this->assertArrayHasKey('Authorization', $headers);
        $this->assertStringStartsWith('Basic ', $headers['Authorization']);
    }

    /** @test */
    public function api_target_with_bearer_token_includes_headers()
    {
        $apiTarget = ApiTarget::factory()->create([
            'auth_type' => 'bearer_token',
            'auth_token' => 'test-token'
        ]);
        
        $controller = new \App\Http\Controllers\ApiTargetController();
        $headers = $controller->prepareHeaders($apiTarget);
        
        $this->assertTrue(array_key_exists('Authorization', $headers));
        $this->assertTrue($headers['Authorization'] === 'Bearer test-token');
    }

    /** @test */
    public function api_target_with_api_key_includes_headers()
    {
        $apiTarget = ApiTarget::factory()->create([
            'auth_type' => 'api_key',
            'auth_token' => 'test-api-key'
        ]);
        
        $controller = new \App\Http\Controllers\ApiTargetController();
        $headers = $controller->prepareHeaders($apiTarget);
        
        $this->assertTrue(array_key_exists('X-API-Key', $headers));
        $this->assertTrue($headers['X-API-Key'] === 'test-api-key');
    }
} 