# 📦 Installation Guide - Laravel License Server

Complete step-by-step installation guide for the Laravel License Server API.

## Prerequisites

Before you begin, ensure you have the following installed:

- ✅ PHP >= 8.2
- ✅ Composer
- ✅ MySQL/MariaDB or PostgreSQL
- ✅ Web server (Apache/Nginx) or use Laravel's built-in server

## Step-by-Step Installation

### Step 1: Clone or Setup Project

If you haven't already, navigate to your project directory:

```bash
cd c:\laragon\www\lisensiapi
```

### Step 2: Install Composer Dependencies

```bash
composer install
```

This will install all required Laravel packages and dependencies.

### Step 3: Environment Configuration

Copy the example environment file:

```bash
# Windows
copy .env.example .env

# Linux/Mac
cp .env.example .env
```

Edit `.env` file and configure your database:

```env
APP_NAME="License Server"
APP_ENV=production
APP_DEBUG=false
APP_URL=http://your-domain.com

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=license_server
DB_USERNAME=root
DB_PASSWORD=yourpassword
```

**Important Settings:**

- Set `APP_DEBUG=false` in production
- Set `APP_ENV=production` in production
- Configure proper `APP_URL`

### Step 4: Generate Application Key

```bash
php artisan key:generate
```

This generates a unique encryption key for your application.

### Step 5: Create Database

Create a new database using MySQL command line or phpMyAdmin:

**Using MySQL Command Line:**

```bash
mysql -u root -p
```

```sql
CREATE DATABASE license_server CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
EXIT;
```

**Or using phpMyAdmin:**

1. Open phpMyAdmin
2. Click "New" to create a database
3. Name it: `license_server`
4. Choose collation: `utf8mb4_unicode_ci`
5. Click "Create"

### Step 6: Run Database Migrations

```bash
php artisan migrate
```

This will create the `licenses` table with all required fields.

**Expected Output:**

```
Migration table created successfully.
Migrating: 2025_11_23_070258_create_licenses_table
Migrated:  2025_11_23_070258_create_licenses_table
```

### Step 7: Generate Sample License Keys (Optional)

If you want to generate sample license keys for testing:

```bash
php artisan db:seed --class=LicenseSeeder
```

This will create 10 sample license keys with:
- Status: `inactive`
- Expiration: 1 year from now

### Step 8: Clear Caches

```bash
php artisan config:clear
php artisan cache:clear
php artisan route:clear
php artisan optimize:clear
```

### Step 9: Start the Server

For development:

```bash
php artisan serve
```

Your API will be available at: `http://localhost:8000`

For production with Laragon:

- The site should be automatically available at `http://lisensiapi.test`

## Verification

### Test the API

**1. Check if server is running:**

```bash
curl http://localhost:8000/up
```

**2. Test the verify endpoint (should return "license key not found"):**

```bash
curl "http://localhost:8000/api/verify?key=TEST&domain=example.com"
```

**Expected Response:**

```json
{
  "status": "invalid",
  "msg": "license key not found"
}
```

**3. If you ran the seeder, check database:**

```bash
php artisan tinker
```

```php
\App\Models\License::all();
exit;
```

## Creating Your First License

### Method 1: Using Tinker (Recommended for Testing)

```bash
php artisan tinker
```

```php
use App\Models\License;

$license = License::create([
    'license_key' => 'DEMO-1234-5678-ABCD',
    'status' => 'inactive',
    'expired_at' => now()->addYear()
]);

echo "License created: " . $license->license_key;
exit;
```

### Method 2: Direct Database Insert

Using phpMyAdmin or MySQL command line:

```sql
INSERT INTO licenses (license_key, status, expired_at, created_at, updated_at)
VALUES ('DEMO-1234-5678-ABCD', 'inactive', DATE_ADD(NOW(), INTERVAL 1 YEAR), NOW(), NOW());
```

## Testing the License Flow

### 1. Activate the License

```bash
curl -X POST "http://localhost:8000/api/activate" \
  -H "Content-Type: application/json" \
  -d "{\"key\":\"DEMO-1234-5678-ABCD\",\"domain\":\"example.com\"}"
```

**Expected Response:**

```json
{
  "status": "activated",
  "msg": "License activated successfully",
  "data": {
    "license_key": "DEMO-1234-5678-ABCD",
    "domain": "example.com",
    "status": "active",
    "activated_at": "2025-11-23 12:00:00"
  }
}
```

### 2. Verify the License

```bash
curl "http://localhost:8000/api/verify?key=DEMO-1234-5678-ABCD&domain=example.com"
```

**Expected Response:**

```json
{
  "status": "valid",
  "msg": "License is valid",
  "data": {
    "license_key": "DEMO-1234-5678-ABCD",
    "domain": "example.com",
    "status": "active",
    "expired_at": "2026-11-23"
  }
}
```

## Production Deployment

### Additional Steps for Production

1. **Set proper permissions:**

```bash
chmod -R 755 storage bootstrap/cache
chown -R www-data:www-data storage bootstrap/cache
```

2. **Optimize for production:**

```bash
php artisan config:cache
php artisan route:cache
php artisan view:cache
composer install --optimize-autoloader --no-dev
```

3. **Setup HTTPS:**

Use Let's Encrypt or your SSL certificate:

```bash
certbot --nginx -d your-domain.com
```

4. **Setup Queue Worker (if needed):**

```bash
php artisan queue:work --daemon
```

5. **Setup Scheduler (if needed):**

Add to crontab:

```
* * * * * cd /path/to/project && php artisan schedule:run >> /dev/null 2>&1
```

### Environment Variables for Production

```env
APP_ENV=production
APP_DEBUG=false
APP_URL=https://your-domain.com

# Session & Cache
CACHE_DRIVER=redis
SESSION_DRIVER=redis
QUEUE_CONNECTION=redis

# Security
SESSION_SECURE_COOKIE=true
```

## Troubleshooting

### Issue: Migration fails

**Error:** `could not find driver`

**Solution:**

Enable PDO extension in `php.ini`:

```ini
extension=pdo_mysql
extension=mysqli
```

Then restart PHP/Apache.

### Issue: Permission denied

**Solution:**

```bash
chmod -R 755 storage bootstrap/cache
```

### Issue: 404 on /api routes

**Solution:**

```bash
php artisan route:clear
php artisan optimize:clear
```

Check if `routes/api.php` exists and `bootstrap/app.php` includes it.

### Issue: CORS errors from client

**Solution:**

Install and configure Laravel CORS:

```bash
composer require fruitcake/laravel-cors
```

Edit `config/cors.php`:

```php
'paths' => ['api/*'],
'allowed_origins' => ['https://your-client-domain.com'],
```

## Backup Strategy

### Database Backup

**Daily backup script:**

```bash
#!/bin/bash
mysqldump -u root -p license_server > backup_$(date +%Y%m%d).sql
```

### File Backup

Important directories to backup:

- `.env` file
- `database/` folder
- `storage/` folder (logs)

## Monitoring

### Check Logs

```bash
tail -f storage/logs/laravel.log
```

### Monitor API Performance

```bash
php artisan pail
```

## Next Steps

1. ✅ [Optional] Install Laravel Filament for admin panel
2. ✅ [Optional] Setup monitoring (Laravel Telescope)
3. ✅ [Optional] Add API authentication (Laravel Sanctum)
4. ✅ Setup automated backups
5. ✅ Configure firewall rules
6. ✅ Setup monitoring/alerting

## Support & Documentation

- 📚 [Main README](README.md)
- 📖 [Laravel Documentation](https://laravel.com/docs)
- 🔧 [API Documentation](README.md#-api-endpoints)

---

**Congratulations!** Your Laravel License Server is now installed and ready to use! 🎉

