<?php

namespace App\Console\Commands;

use App\Models\License;
use Illuminate\Console\Command;
use Illuminate\Support\Str;

class GenerateLicenseKey extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'license:generate {count=1 : Number of licenses to generate} {--expired-days=365 : Days until expiration}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate new license keys';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $count = (int) $this->argument('count');
        $expiredDays = (int) $this->option('expired-days');

        $this->info("Generating {$count} license key(s)...");
        $this->newLine();

        $licenses = [];

        for ($i = 0; $i < $count; $i++) {
            $licenseKey = $this->generateLicenseKey();
            
            $license = License::create([
                'license_key' => $licenseKey,
                'status' => 'inactive',
                'expired_at' => now()->addDays($expiredDays),
            ]);

            $licenses[] = [
                'License Key' => $license->license_key,
                'Status' => $license->status,
                'Expires At' => $license->expired_at->format('Y-m-d'),
            ];
        }

        $this->table(
            ['License Key', 'Status', 'Expires At'],
            $licenses
        );

        $this->newLine();
        $this->info("✅ Successfully generated {$count} license key(s)!");

        return Command::SUCCESS;
    }

    /**
     * Generate a random license key in format: XXXX-XXXX-XXXX-XXXX
     */
    private function generateLicenseKey(): string
    {
        $parts = [];
        for ($i = 0; $i < 4; $i++) {
            $parts[] = strtoupper(Str::random(4));
        }
        return implode('-', $parts);
    }
}
