<?php

namespace App\Http\Controllers;

use App\Models\License;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AnalyticsController extends Controller
{
    public function index()
    {
        // Monthly trend (last 12 months)
        $monthlyTrend = [];
        for ($i = 11; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $count = License::whereYear('created_at', $date->year)
                           ->whereMonth('created_at', $date->month)
                           ->count();
            $monthlyTrend[] = [
                'month' => $date->format('M Y'),
                'count' => $count
            ];
        }

        // Plan distribution
        $planDistribution = License::select('plan_type', DB::raw('count(*) as count'))
            ->groupBy('plan_type')
            ->get();

        // Status distribution
        $statusDistribution = License::select('status', DB::raw('count(*) as count'))
            ->groupBy('status')
            ->get();

        // Top customers
        $topCustomers = License::select(
            'customer_name',
            'customer_email',
            DB::raw('COUNT(*) as license_count'),
            DB::raw('SUM(CASE WHEN status = "active" THEN 1 ELSE 0 END) as active_count')
        )
        ->groupBy('customer_email', 'customer_name')
        ->orderByDesc('license_count')
        ->limit(10)
        ->get();

        // Revenue by plan type
        $revenueByPlan = [
            'basic' => License::where('plan_type', 'basic')->where('status', 'active')->count() * 50,
            'professional' => License::where('plan_type', 'professional')->where('status', 'active')->count() * 50,
            'premium' => License::where('plan_type', 'premium')->where('status', 'active')->count() * 50,
            'enterprise' => License::where('plan_type', 'enterprise')->where('status', 'active')->count() * 50,
            'trial' => 0, // Trial is free
        ];

        // Expiry analysis
        $expiringIn7Days = License::whereBetween('expired_at', [now(), now()->addDays(7)])->count();
        $expiringIn30Days = License::whereBetween('expired_at', [now(), now()->addDays(30)])->count();
        $expiredCount = License::where('expired_at', '<', now())->count();

        return view('analytics.index', compact(
            'monthlyTrend',
            'planDistribution',
            'statusDistribution',
            'topCustomers',
            'revenueByPlan',
            'expiringIn7Days',
            'expiringIn30Days',
            'expiredCount'
        ));
    }
}
