<?php

namespace App\Http\Controllers;

use App\Models\License;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CustomerController extends Controller
{
    /**
     * Display a listing of customers.
     */
    public function index(Request $request)
    {
        // Group licenses by customer
        $query = License::select(
            'customer_name',
            'customer_email',
            DB::raw('COUNT(*) as total_licenses'),
            DB::raw('SUM(CASE WHEN status = "active" THEN 1 ELSE 0 END) as active_licenses'),
            DB::raw('MAX(created_at) as last_license_date')
        )->groupBy('customer_email', 'customer_name');

        // Search
        if ($search = $request->get('search')) {
            $query->where(function($q) use ($search) {
                $q->where('customer_name', 'like', "%{$search}%")
                  ->orWhere('customer_email', 'like', "%{$search}%");
            });
        }

        $customers = $query->paginate(20);

        return view('customers.index', compact('customers'));
    }

    /**
     * Show customer details with their licenses
     */
    public function show($email)
    {
        $customerLicenses = License::where('customer_email', $email)->get();
        
        if ($customerLicenses->isEmpty()) {
            abort(404, 'Customer not found');
        }

        $customer = [
            'name' => $customerLicenses->first()->customer_name,
            'email' => $email,
            'total_licenses' => $customerLicenses->count(),
            'active_licenses' => $customerLicenses->where('status', 'active')->count(),
            'expired_licenses' => $customerLicenses->filter(fn($l) => $l->expired_at && $l->expired_at < now())->count(),
            'total_revenue' => $customerLicenses->where('status', 'active')->count() * 50,
            'first_license' => $customerLicenses->min('created_at'),
        ];

        return view('customers.show', compact('customer', 'customerLicenses'));
    }

    /**
     * Remove the specified customer and all their licenses.
     */
    public function destroy($email)
    {
        License::where('customer_email', $email)->delete();

        return redirect()->route('customers.index')
            ->with('success', 'Customer and all their licenses deleted successfully!');
    }
}
