<?php

namespace App\Http\Controllers;

use App\Models\License;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index()
    {
        // Get statistics
        $totalLicenses = License::count();
        $activeLicenses = License::where('status', 'active')->count();
        $expiredLicenses = License::where('expired_at', '<', now())->count();
        $suspendedLicenses = License::where('status', 'suspended')->count();
        
        // Calculate growth percentage (comparing with last month)
        $lastMonthTotal = License::whereMonth('created_at', now()->subMonth()->month)->count();
        $totalGrowth = $lastMonthTotal > 0 ? (($totalLicenses - $lastMonthTotal) / $lastMonthTotal * 100) : 0;
        
        $lastMonthActive = License::where('status', 'active')
            ->whereMonth('created_at', now()->subMonth()->month)
            ->count();
        $activeGrowth = $lastMonthActive > 0 ? (($activeLicenses - $lastMonthActive) / $lastMonthActive * 100) : 0;
        
        $lastMonthExpired = License::where('expired_at', '<', now()->subMonth())->count();
        $expiredGrowth = $lastMonthExpired > 0 ? (($expiredLicenses - $lastMonthExpired) / $lastMonthExpired * 100) : 0;
        
        // Calculate revenue (dummy calculation: $50 per license)
        $monthlyRevenue = $activeLicenses * 50;
        $lastMonthRevenue = $lastMonthActive * 50;
        $revenueGrowth = $lastMonthRevenue > 0 ? (($monthlyRevenue - $lastMonthRevenue) / $lastMonthRevenue * 100) : 0;
        
        // Get status distribution
        $statusDistribution = [
            'active' => $activeLicenses > 0 ? round(($activeLicenses / $totalLicenses) * 100, 1) : 0,
            'expired' => $expiredLicenses > 0 ? round(($expiredLicenses / $totalLicenses) * 100, 1) : 0,
            'trial' => 0,
            'suspended' => $suspendedLicenses > 0 ? round(($suspendedLicenses / $totalLicenses) * 100, 1) : 0,
        ];
        
        // Get licenses by plan type
        $trialLicenses = License::where('plan_type', 'trial')->count();
        $statusDistribution['trial'] = $trialLicenses > 0 ? round(($trialLicenses / $totalLicenses) * 100, 1) : 0;
        
        // Get recent licenses
        $recentLicenses = License::orderBy('created_at', 'desc')->take(10)->get();
        
        // Get chart data (last 7 days)
        $chartData = [];
        for ($i = 6; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $count = License::whereDate('created_at', $date->format('Y-m-d'))->count();
            $chartData[] = [
                'date' => $date->format('M d'),
                'count' => $count,
                'percentage' => $totalLicenses > 0 ? ($count / $totalLicenses * 100) : 0
            ];
        }
        
        return view('welcome', [
            'stats' => [
                'total' => $totalLicenses,
                'active' => $activeLicenses,
                'expired' => $expiredLicenses,
                'suspended' => $suspendedLicenses,
                'revenue' => $monthlyRevenue,
            ],
            'growth' => [
                'total' => round($totalGrowth, 1),
                'active' => round($activeGrowth, 1),
                'expired' => round($expiredGrowth, 1),
                'revenue' => round($revenueGrowth, 1),
            ],
            'statusDistribution' => $statusDistribution,
            'recentLicenses' => $recentLicenses,
            'chartData' => $chartData,
        ]);
    }
}
