<?php

namespace App\Http\Controllers;

use App\Models\License;
use Illuminate\Http\Request;

class LicenseController extends Controller
{
    /**
     * Verify license key
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function check(Request $request)
    {
        // Validate request
        $validated = $request->validate([
            'key' => 'required|string',
            'domain' => 'required|string',
        ]);

        $key    = $validated['key'];
        $domain = $validated['domain'];
        $ip     = $request->ip();

        $license = License::where('license_key', $key)->first();

        if (!$license) {
            return response()->json([
                'status' => 'invalid',
                'msg' => 'license key not found'
            ], 404);
        }

        if ($license->status !== 'active') {
            return response()->json([
                'status' => 'suspended',
                'msg' => 'License is suspended'
            ], 403);
        }

        if ($license->domain !== $domain) {
            return response()->json([
                'status' => 'invalid',
                'msg' => 'domain mismatch'
            ], 403);
        }

        if ($license->expired_at && $license->expired_at < now()) {
            return response()->json([
                'status' => 'expired',
                'msg' => 'License has expired'
            ], 403);
        }

        return response()->json([
            'status' => 'valid',
            'msg' => 'License is valid',
            'data' => [
                'license_key' => $license->license_key,
                'domain' => $license->domain,
                'status' => $license->status,
                'expired_at' => $license->expired_at?->format('Y-m-d'),
            ]
        ]);
    }

    /**
     * Activate license for a domain
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function activate(Request $request)
    {
        // Validate request
        $validated = $request->validate([
            'key' => 'required|string',
            'domain' => 'required|string',
        ]);

        $key    = $validated['key'];
        $domain = $validated['domain'];
        $ip     = $request->ip();

        $license = License::where('license_key', $key)->first();

        if (!$license) {
            return response()->json([
                'status' => 'invalid',
                'msg' => 'License key not found'
            ], 404);
        }

        if ($license->domain !== null) {
            return response()->json([
                'status' => 'already_assigned',
                'msg' => 'License already assigned to: ' . $license->domain
            ], 400);
        }

        $license->domain = $domain;
        $license->ip_address = $ip;
        $license->status = 'active';
        $license->save();

        return response()->json([
            'status' => 'activated',
            'msg' => 'License activated successfully',
            'data' => [
                'license_key' => $license->license_key,
                'domain' => $license->domain,
                'status' => $license->status,
                'activated_at' => $license->updated_at->format('Y-m-d H:i:s'),
            ]
        ]);
    }
}
