<?php

namespace App\Http\Controllers;

use App\Models\License;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class LicenseCrudController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = License::query();

        // Search
        if ($search = $request->get('search')) {
            $query->where(function($q) use ($search) {
                $q->where('license_key', 'like', "%{$search}%")
                  ->orWhere('customer_name', 'like', "%{$search}%")
                  ->orWhere('customer_email', 'like', "%{$search}%")
                  ->orWhere('domain', 'like', "%{$search}%");
            });
        }

        // Filter by status
        if ($status = $request->get('status')) {
            $query->where('status', $status);
        }

        // Filter by plan type
        if ($planType = $request->get('plan_type')) {
            $query->where('plan_type', $planType);
        }

        // Filter by expiry
        if ($expiry = $request->get('expiry')) {
            if ($expiry === 'expired') {
                $query->where('expired_at', '<', now());
            } elseif ($expiry === 'expiring_soon') {
                $query->whereBetween('expired_at', [now(), now()->addDays(30)]);
            }
        }

        // Sort
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $licenses = $query->paginate(20);

        return view('licenses.index', compact('licenses'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('licenses.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_name' => 'required|string|max:255',
            'customer_email' => 'required|email|max:255',
            'product_name' => 'required|string|max:255',
            'plan_type' => 'required|in:basic,professional,premium,enterprise,trial',
            'status' => 'required|in:active,inactive,suspended',
            'expired_at' => 'required|date',
            'domain' => 'nullable|string|max:255',
            'ip_address' => 'nullable|ip',
        ]);

        $validated['license_key'] = $this->generateLicenseKey();

        License::create($validated);

        return redirect()->route('licenses.index')
            ->with('success', 'License created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(License $license)
    {
        return view('licenses.show', compact('license'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(License $license)
    {
        return view('licenses.edit', compact('license'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, License $license)
    {
        $validated = $request->validate([
            'customer_name' => 'required|string|max:255',
            'customer_email' => 'required|email|max:255',
            'product_name' => 'required|string|max:255',
            'plan_type' => 'required|in:basic,professional,premium,enterprise,trial',
            'status' => 'required|in:active,inactive,suspended',
            'expired_at' => 'required|date',
            'domain' => 'nullable|string|max:255',
            'ip_address' => 'nullable|ip',
        ]);

        $license->update($validated);

        return redirect()->route('licenses.index')
            ->with('success', 'License updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(License $license)
    {
        $license->delete();

        return redirect()->route('licenses.index')
            ->with('success', 'License deleted successfully!');
    }

    /**
     * Generate a new license key
     */
    private function generateLicenseKey(): string
    {
        do {
            $parts = [];
            for ($i = 0; $i < 4; $i++) {
                $parts[] = strtoupper(Str::random(4));
            }
            $licenseKey = implode('-', $parts);
        } while (License::where('license_key', $licenseKey)->exists());

        return $licenseKey;
    }

    /**
     * Export licenses to CSV
     */
    public function export()
    {
        $licenses = License::all();

        $filename = 'licenses_' . date('Y-m-d_His') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($licenses) {
            $file = fopen('php://output', 'w');
            
            // Header
            fputcsv($file, [
                'License Key', 'Customer Name', 'Customer Email', 'Product', 
                'Plan Type', 'Status', 'Domain', 'IP Address', 
                'Expires At', 'Created At'
            ]);

            // Data
            foreach ($licenses as $license) {
                fputcsv($file, [
                    $license->license_key,
                    $license->customer_name,
                    $license->customer_email,
                    $license->product_name,
                    $license->plan_type,
                    $license->status,
                    $license->domain ?? 'N/A',
                    $license->ip_address ?? 'N/A',
                    $license->expired_at?->format('Y-m-d') ?? 'N/A',
                    $license->created_at->format('Y-m-d H:i:s'),
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
