#!/bin/bash

###############################################################################
# License API - Database Backup Script
# 
# This script creates automated backups of the database
# Usage: ./backup-database.sh
# Cron: 0 2 * * * /path/to/backup-database.sh
###############################################################################

# Configuration
BACKUP_DIR="/backups/licenseapi"
PROJECT_DIR="/var/www/licenseapi"
DATE=$(date +%Y%m%d_%H%M%S)
KEEP_DAYS=7

# Colors
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m'

echo "=========================================="
echo "  License API - Database Backup"
echo "=========================================="
echo "Date: $(date '+%Y-%m-%d %H:%M:%S')"
echo ""

# Create backup directory if not exists
mkdir -p $BACKUP_DIR

# Load database credentials from .env
if [ -f "$PROJECT_DIR/.env" ]; then
    source <(grep -v '^#' $PROJECT_DIR/.env | grep -E 'DB_' | sed 's/\r$//' | sed 's/^/export /')
else
    echo -e "${RED}ERROR: .env file not found!${NC}"
    exit 1
fi

# Backup filename
BACKUP_FILE="$BACKUP_DIR/db_backup_$DATE.sql"
BACKUP_GZIP="$BACKUP_FILE.gz"

# Perform backup based on database type
if [ "$DB_CONNECTION" == "mysql" ]; then
    echo "Backing up MySQL database..."
    
    mysqldump \
        --user="$DB_USERNAME" \
        --password="$DB_PASSWORD" \
        --host="$DB_HOST" \
        --port="$DB_PORT" \
        --single-transaction \
        --routines \
        --triggers \
        --events \
        "$DB_DATABASE" > "$BACKUP_FILE"
        
    if [ $? -eq 0 ]; then
        echo -e "${GREEN}✓ MySQL backup created${NC}"
    else
        echo -e "${RED}✗ MySQL backup failed!${NC}"
        exit 1
    fi
    
elif [ "$DB_CONNECTION" == "pgsql" ]; then
    echo "Backing up PostgreSQL database..."
    
    PGPASSWORD="$DB_PASSWORD" pg_dump \
        --username="$DB_USERNAME" \
        --host="$DB_HOST" \
        --port="$DB_PORT" \
        "$DB_DATABASE" > "$BACKUP_FILE"
        
    if [ $? -eq 0 ]; then
        echo -e "${GREEN}✓ PostgreSQL backup created${NC}"
    else
        echo -e "${RED}✗ PostgreSQL backup failed!${NC}"
        exit 1
    fi
else
    echo -e "${RED}ERROR: Unsupported database type: $DB_CONNECTION${NC}"
    exit 1
fi

# Compress backup
echo "Compressing backup..."
gzip "$BACKUP_FILE"

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✓ Backup compressed${NC}"
    BACKUP_SIZE=$(du -h "$BACKUP_GZIP" | cut -f1)
    echo "  Size: $BACKUP_SIZE"
    echo "  File: $BACKUP_GZIP"
else
    echo -e "${RED}✗ Compression failed!${NC}"
    exit 1
fi

# Remove old backups
echo "Cleaning old backups (older than $KEEP_DAYS days)..."
find $BACKUP_DIR -name "db_backup_*.sql.gz" -type f -mtime +$KEEP_DAYS -delete

OLD_COUNT=$(find $BACKUP_DIR -name "db_backup_*.sql.gz" -type f | wc -l)
echo -e "${GREEN}✓ Cleanup completed${NC}"
echo "  Remaining backups: $OLD_COUNT"

# Optional: Backup files too
read -p "Also backup application files? (y/n): " -n 1 -r
echo ""
if [[ $REPLY =~ ^[Yy]$ ]]; then
    echo "Backing up application files..."
    
    BACKUP_TAR="$BACKUP_DIR/files_backup_$DATE.tar.gz"
    
    tar -czf "$BACKUP_TAR" \
        --exclude='node_modules' \
        --exclude='vendor' \
        --exclude='storage/logs' \
        --exclude='storage/framework/cache' \
        --exclude='storage/framework/sessions' \
        --exclude='storage/framework/views' \
        -C /var/www licenseapi
    
    if [ $? -eq 0 ]; then
        FILES_SIZE=$(du -h "$BACKUP_TAR" | cut -f1)
        echo -e "${GREEN}✓ Files backup created${NC}"
        echo "  Size: $FILES_SIZE"
        echo "  File: $BACKUP_TAR"
    else
        echo -e "${RED}✗ Files backup failed!${NC}"
    fi
    
    # Clean old file backups
    find $BACKUP_DIR -name "files_backup_*.tar.gz" -type f -mtime +$KEEP_DAYS -delete
fi

echo ""
echo "=========================================="
echo -e "${GREEN}✓ BACKUP COMPLETED!${NC}"
echo "=========================================="
echo ""
echo "Backup location: $BACKUP_DIR"
echo "Latest backup: $BACKUP_GZIP"
echo ""
echo "To restore this backup:"
echo "  gunzip $BACKUP_GZIP"
echo ""
if [ "$DB_CONNECTION" == "mysql" ]; then
    echo "  mysql -u $DB_USERNAME -p $DB_DATABASE < ${BACKUP_GZIP%.gz}"
elif [ "$DB_CONNECTION" == "pgsql" ]; then
    echo "  psql -U $DB_USERNAME -d $DB_DATABASE < ${BACKUP_GZIP%.gz}"
fi
echo ""

