<?php
/**
 * License Validator Helper Class
 * 
 * Digunakan untuk validasi license di website client
 * 
 * @author License API System
 * @version 1.0.0
 */

class LicenseValidator {
    
    private $apiBaseUrl = 'http://127.0.0.1:8000/api';
    private $licenseFile = 'license.json';
    
    /**
     * Aktivasi license (dijalankan sekali saat instalasi)
     * 
     * @param string $licenseKey License key dari admin
     * @param string $domain Domain website client
     * @return array ['success' => bool, 'message' => string]
     */
    public function activate($licenseKey, $domain) {
        $url = $this->apiBaseUrl . '/activate';
        
        $data = [
            'key' => $licenseKey,
            'domain' => $domain
        ];
        
        $response = $this->sendRequest($url, 'POST', $data);
        
        if ($response && isset($response['status']) && $response['status'] === 'activated') {
            // Simpan license ke file
            $licenseData = [
                'key' => $licenseKey,
                'domain' => $domain,
                'activated_at' => date('Y-m-d H:i:s'),
                'last_check' => date('Y-m-d H:i:s')
            ];
            
            file_put_contents($this->licenseFile, json_encode($licenseData, JSON_PRETTY_PRINT));
            
            return [
                'success' => true,
                'message' => 'License berhasil diaktifkan!',
                'data' => $response['data'] ?? []
            ];
        }
        
        return [
            'success' => false,
            'message' => $response['msg'] ?? 'Aktivasi gagal. Silakan cek license key Anda.'
        ];
    }
    
    /**
     * Verifikasi license (dijalankan setiap kali load aplikasi)
     * 
     * @param bool $forceCheck Force check ke server (skip cache)
     * @return array ['valid' => bool, 'message' => string, 'data' => array]
     */
    public function verify($forceCheck = false) {
        // Cek apakah license sudah disimpan
        if (!file_exists($this->licenseFile)) {
            return [
                'valid' => false,
                'message' => 'License belum diaktifkan. Silakan aktifkan terlebih dahulu.'
            ];
        }
        
        $license = json_decode(file_get_contents($this->licenseFile), true);
        
        // Cache verification selama 5 menit untuk reduce API calls
        if (!$forceCheck && isset($license['last_check'])) {
            $lastCheck = strtotime($license['last_check']);
            if (time() - $lastCheck < 300) { // 5 minutes
                return [
                    'valid' => true,
                    'message' => 'License valid (cached)',
                    'expires_at' => $license['expires_at'] ?? null,
                    'cached' => true
                ];
            }
        }
        
        // Verify dengan server
        $url = $this->apiBaseUrl . '/verify?' . http_build_query([
            'key' => $license['key'],
            'domain' => $license['domain']
        ]);
        
        $response = $this->sendRequest($url, 'GET');
        
        if ($response && isset($response['status']) && $response['status'] === 'valid') {
            // Update last check time
            $license['last_check'] = date('Y-m-d H:i:s');
            $license['expires_at'] = $response['data']['expired_at'] ?? null;
            file_put_contents($this->licenseFile, json_encode($license, JSON_PRETTY_PRINT));
            
            return [
                'valid' => true,
                'message' => 'License valid',
                'expires_at' => $response['data']['expired_at'] ?? null,
                'data' => $response['data'] ?? []
            ];
        }
        
        return [
            'valid' => false,
            'message' => $response['msg'] ?? 'License tidak valid atau sudah expired.'
        ];
    }
    
    /**
     * Get license info dari local storage
     * 
     * @return array|null
     */
    public function getLicenseInfo() {
        if (!file_exists($this->licenseFile)) {
            return null;
        }
        
        return json_decode(file_get_contents($this->licenseFile), true);
    }
    
    /**
     * Delete license (untuk deactivate)
     * 
     * @return bool
     */
    public function deactivate() {
        if (file_exists($this->licenseFile)) {
            return unlink($this->licenseFile);
        }
        return true;
    }
    
    /**
     * Helper untuk send HTTP request
     * 
     * @param string $url URL endpoint
     * @param string $method GET atau POST
     * @param array|null $data Data untuk POST request
     * @return array|null
     */
    private function sendRequest($url, $method = 'GET', $data = null) {
        $ch = curl_init();
        
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Disable for localhost, enable in production
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Accept: application/json'
            ]);
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            error_log("License API Error: " . $error);
            return null;
        }
        
        if ($httpCode >= 200 && $httpCode < 300) {
            return json_decode($response, true);
        }
        
        // Parse error response
        $errorResponse = json_decode($response, true);
        return $errorResponse ?? ['status' => 'error', 'msg' => 'Request failed with code ' . $httpCode];
    }
}

