# 🚀 CLIENT EXAMPLE - License API Integration

Contoh implementasi License API di website client.

---

## 📁 File Structure

```
client-example/
├── LicenseValidator.php   # Helper class untuk validasi license
├── activate.php           # Halaman aktivasi license
├── index.php             # Protected page (dashboard)
├── deactivate.php        # Deactivate license
├── license.json          # License data (auto-generated)
└── README.md             # Dokumentasi ini
```

---

## ⚙️ Setup & Installation

### 1. **Copy Files ke Website Client**

Copy semua file di folder ini ke website client Anda:

```bash
# Contoh struktur:
your-website/
├── LicenseValidator.php
├── activate.php
├── index.php
├── deactivate.php
└── ... (file website lainnya)
```

### 2. **Konfigurasi API URL**

Edit `LicenseValidator.php` line 14:

```php
private $apiBaseUrl = 'http://127.0.0.1:8000/api';  // Ganti dengan URL production
```

**Production:**
```php
private $apiBaseUrl = 'https://yourdomain.com/api';
```

### 3. **Set Permissions** (Linux/Unix)

```bash
chmod 755 *.php
chmod 666 license.json  # Harus writable
```

---

## 🔐 Cara Menggunakan

### **Step 1: Dapatkan License Key**

1. Akses admin panel: `http://127.0.0.1:8000/licenses`
2. Klik "New License"
3. Isi data customer dan buat license
4. Copy license key yang di-generate

### **Step 2: Aktivasi di Client**

1. Akses `activate.php` di website client
2. Paste license key
3. Klik "Aktivasi License"
4. Redirect otomatis ke dashboard jika sukses

### **Step 3: Access Protected Pages**

Setelah license aktif, user bisa akses semua protected pages.

---

## 🛡️ Cara Proteksi Halaman

### Metode 1: Include di setiap file

```php
<?php
require_once 'LicenseValidator.php';

$licenseValidator = new LicenseValidator();
$result = $licenseValidator->verify();

if (!$result['valid']) {
    header('Location: activate.php');
    exit;
}

// Your protected content here
?>
```

### Metode 2: Gunakan file guard (Recommended)

Buat file `license-guard.php`:

```php
<?php
require_once 'LicenseValidator.php';

function checkLicense() {
    $licenseValidator = new LicenseValidator();
    $result = $licenseValidator->verify();
    
    if (!$result['valid']) {
        header('Location: activate.php');
        exit;
    }
    
    return $result;
}

$licenseInfo = checkLicense();
```

Lalu include di setiap protected page:

```php
<?php
require_once 'license-guard.php';
// Page otomatis protected
?>
```

---

## 📋 API Flow

### **Activation Flow:**

```
Client Website → POST /api/activate
    ↓
License API validates
    ↓
Assign domain to license
    ↓
Save to license.json
    ↓
Redirect to dashboard
```

### **Verification Flow:**

```
Every Page Load → Check license.json
    ↓
GET /api/verify
    ↓
License API validates
    ↓
Cache result (5 min)
    ↓
Allow/Deny access
```

---

## 🎯 Features

### ✅ **LicenseValidator.php**

- `activate($key, $domain)` - Aktivasi license
- `verify($forceCheck)` - Verifikasi license
- `getLicenseInfo()` - Get license info dari local
- `deactivate()` - Hapus license data

### ✅ **activate.php**

- Form aktivasi license
- Validasi license key format
- Auto-detect domain
- Error handling
- Success redirect

### ✅ **index.php**

- Protected dashboard
- License info display
- Premium content area
- Feature showcases
- Status indicators

### ✅ **deactivate.php**

- Confirmation dialog
- Remove license data
- Redirect to activate page

---

## 🔄 Caching System

Untuk mengurangi API calls, sistem menggunakan caching:

```php
// Verify dengan cache (default)
$result = $licenseValidator->verify();  // Cache 5 menit

// Force verify (skip cache)
$result = $licenseValidator->verify(true);
```

**Cache Duration:** 5 menit (300 detik)  
**File:** `license.json` dengan `last_check` timestamp

---

## 🧪 Testing

### **Test Aktivasi:**

1. Dapatkan license key dari admin
2. Access: `http://localhost/your-site/activate.php`
3. Input license key
4. Klik "Aktivasi"

### **Test Protected Page:**

1. Access: `http://localhost/your-site/index.php`
2. Jika belum aktif → redirect ke activate.php
3. Jika sudah aktif → tampil dashboard

### **Test Deactivate:**

1. Access: `http://localhost/your-site/deactivate.php`
2. Confirm deactivation
3. License data terhapus
4. Redirect ke activate.php

---

## ⚠️ Troubleshooting

### **Error: "License not found"**

- Pastikan license key benar
- Check di admin panel apakah license ada
- Pastikan API URL benar

### **Error: "Domain mismatch"**

- License sudah diassign ke domain lain
- Contact admin untuk reset
- Atau gunakan license key baru

### **Error: "Connection failed"**

- Check API server status
- Check firewall/network
- Verify API URL configuration

### **Error: "Permission denied" (license.json)**

```bash
chmod 666 license.json
# atau
chmod 777 license.json  # untuk testing
```

---

## 🔒 Security Best Practices

1. **HTTPS Only** - Gunakan SSL di production
2. **Hide license.json** - Jangan expose via web
3. **Rate Limiting** - Limit verify requests
4. **Error Logging** - Log failed attempts
5. **Encrypt Storage** - Encrypt license.json di production

### **.htaccess Protection:**

```apache
<Files "license.json">
    Order Allow,Deny
    Deny from all
</Files>
```

---

## 📱 Mobile/App Integration

Untuk mobile app (iOS/Android), gunakan HTTP request library:

**iOS (Swift):**
```swift
let url = URL(string: "http://api.com/api/activate")!
var request = URLRequest(url: url)
request.httpMethod = "POST"
request.setValue("application/json", forHTTPHeaderField: "Content-Type")
// ... send request
```

**Android (Kotlin):**
```kotlin
val url = "http://api.com/api/activate"
val json = JSONObject()
json.put("key", licenseKey)
json.put("domain", domain)
// ... send request
```

---

## 📞 Support

Jika ada pertanyaan atau issue:

1. Check dokumentasi: `CLIENT_INTEGRATION_GUIDE.md`
2. Check API documentation: `API_DOCUMENTATION.md`
3. Contact admin: admin@yoursite.com

---

## 🎉 Ready to Deploy!

Sistem License API client sudah siap digunakan di production!

**Checklist:**
- [ ] Copy semua files
- [ ] Update API URL
- [ ] Set file permissions
- [ ] Test activation
- [ ] Test verification
- [ ] Setup .htaccess protection
- [ ] Enable HTTPS
- [ ] Deploy to production

---

**Version:** 1.0.0  
**Last Updated:** November 23, 2024  
**License:** MIT

