<?php

namespace Database\Seeders;

use App\Models\License;
use Illuminate\Database\Seeder;
use Illuminate\Support\Str;

class LicenseSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $customerNames = [
            'John Doe', 'Alice Smith', 'Bob Johnson', 'Carol Davis', 'Eve Wilson',
            'Frank Miller', 'Grace Lee', 'Henry Brown', 'Iris Chen', 'Jack Taylor',
            'Kate Anderson', 'Leo Martinez', 'Maria Garcia', 'Nathan White', 'Olivia Moore',
            'Peter Jackson', 'Quinn Harris', 'Rachel Clark', 'Sam Robinson', 'Tina Walker',
            'Uma Patel', 'Victor King', 'Wendy Scott', 'Xavier Green', 'Yara Baker'
        ];

        $productNames = [
            'Premium Software Suite', 'Enterprise Application', 'Professional Tools',
            'Basic Package', 'Advanced Analytics', 'Cloud Platform', 'Development Kit',
            'Business Intelligence', 'Security Suite', 'Marketing Automation'
        ];

        $planTypes = ['basic', 'professional', 'premium', 'enterprise', 'trial'];
        $statuses = ['active', 'active', 'active', 'active', 'inactive', 'suspended']; // More active licenses

        // Generate 50 sample licenses with complete data
        for ($i = 0; $i < 50; $i++) {
            $customerName = $customerNames[array_rand($customerNames)];
            $planType = $planTypes[array_rand($planTypes)];
            $status = $statuses[array_rand($statuses)];
            
            // Generate email from customer name
            $emailName = strtolower(str_replace(' ', '', $customerName));
            $domains = ['example.com', 'company.com', 'startup.io', 'tech.com', 'digital.net', 'business.org'];
            $customerEmail = $emailName . '@' . $domains[array_rand($domains)];
            
            // Determine expiry date based on plan type
            $expiryDate = match($planType) {
                'trial' => now()->addDays(rand(1, 30)),
                'basic' => now()->addMonths(rand(1, 6)),
                'professional' => now()->addMonths(rand(6, 12)),
                'premium' => now()->addYear(),
                'enterprise' => now()->addYears(rand(1, 3)),
            };
            
            // Random chance for expired licenses
            if (rand(1, 100) <= 15) { // 15% chance of expired
                $expiryDate = now()->subDays(rand(1, 60));
                $status = 'inactive';
            }
            
            // Random domain assignment for active licenses
            $domain = null;
            if ($status === 'active' && rand(1, 100) <= 80) { // 80% of active licenses have domain
                $companySlug = strtolower(str_replace(' ', '', explode(' ', $customerName)[1]));
                $domain = $companySlug . '.com';
            }
            
            License::create([
                'license_key' => $this->generateLicenseKey(),
                'customer_name' => $customerName,
                'customer_email' => $customerEmail,
                'product_name' => $productNames[array_rand($productNames)],
                'plan_type' => $planType,
                'domain' => $domain,
                'ip_address' => $domain ? $this->generateRandomIP() : null,
                'status' => $status,
                'expired_at' => $expiryDate,
                'created_at' => now()->subDays(rand(0, 90)), // Spread creation dates
            ]);
        }

        echo "✅ Generated 50 license keys successfully!\n";
    }
    
    /**
     * Generate a random IP address
     */
    private function generateRandomIP(): string
    {
        return rand(1, 255) . '.' . rand(0, 255) . '.' . rand(0, 255) . '.' . rand(1, 255);
    }

    /**
     * Generate a random license key in format: XXXX-XXXX-XXXX-XXXX
     */
    private function generateLicenseKey(): string
    {
        $parts = [];
        for ($i = 0; $i < 4; $i++) {
            $parts[] = strtoupper(Str::random(4));
        }
        return implode('-', $parts);
    }
}
