#!/bin/bash

###############################################################################
# License API - Production Deployment Script
# 
# This script automates the deployment process
# Run this on your production server after uploading files
###############################################################################

echo "=========================================="
echo "  License API - Production Deployment"
echo "=========================================="
echo ""

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Check if running as root
if [ "$EUID" -eq 0 ]; then 
    echo -e "${RED}ERROR: Please do not run as root${NC}"
    exit 1
fi

# Get project directory
PROJECT_DIR=$(pwd)

echo -e "${YELLOW}Project Directory: $PROJECT_DIR${NC}"
echo ""

# Step 1: Check requirements
echo "Step 1: Checking requirements..."

# Check PHP version
PHP_VERSION=$(php -r 'echo PHP_VERSION;')
echo "  ✓ PHP Version: $PHP_VERSION"

# Check Composer
if ! command -v composer &> /dev/null; then
    echo -e "${RED}  ✗ Composer not found!${NC}"
    exit 1
fi
echo "  ✓ Composer found"

# Check if .env exists
if [ ! -f "$PROJECT_DIR/.env" ]; then
    echo -e "${YELLOW}  ! .env file not found${NC}"
    echo "    Creating from env.production.example..."
    cp env.production.example .env
    echo -e "${YELLOW}    Please edit .env with your production settings!${NC}"
    echo ""
    read -p "    Press Enter after editing .env..."
fi

echo ""

# Step 2: Install dependencies
echo "Step 2: Installing dependencies..."
composer install --optimize-autoloader --no-dev
echo "  ✓ Dependencies installed"
echo ""

# Step 3: Generate app key if needed
if grep -q "APP_KEY=base64:YOUR_APP_KEY_HERE" .env; then
    echo "Step 3: Generating application key..."
    php artisan key:generate
    echo "  ✓ App key generated"
else
    echo "Step 3: App key already set"
fi
echo ""

# Step 4: Database migration
echo "Step 4: Running database migrations..."
read -p "  Run migrations? This will modify database (y/n): " -n 1 -r
echo ""
if [[ $REPLY =~ ^[Yy]$ ]]; then
    php artisan migrate --force
    echo "  ✓ Migrations completed"
    
    read -p "  Seed sample data? (y/n): " -n 1 -r
    echo ""
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        php artisan db:seed --class=LicenseSeeder
        echo "  ✓ Sample data seeded"
    fi
else
    echo "  ⊘ Migrations skipped"
fi
echo ""

# Step 5: Cache configuration
echo "Step 5: Caching configuration..."
php artisan config:cache
php artisan route:cache
php artisan view:cache
echo "  ✓ Configuration cached"
echo ""

# Step 6: Set permissions
echo "Step 6: Setting permissions..."
chmod -R 755 $PROJECT_DIR
chmod -R 775 $PROJECT_DIR/storage
chmod -R 775 $PROJECT_DIR/bootstrap/cache
echo "  ✓ Permissions set"
echo ""

# Step 7: Clear old logs
echo "Step 7: Clearing old logs..."
find $PROJECT_DIR/storage/logs -name "*.log" -type f -mtime +30 -delete
echo "  ✓ Old logs cleared"
echo ""

# Step 8: Test configuration
echo "Step 8: Testing configuration..."
php artisan config:show | grep -E "app.env|app.debug|app.url" | head -3
echo ""

# Step 9: Summary
echo "=========================================="
echo "  ✓ DEPLOYMENT COMPLETED!"
echo "=========================================="
echo ""
echo "Next steps:"
echo "  1. Configure web server (Nginx/Apache)"
echo "  2. Setup SSL certificate"
echo "  3. Test API endpoints"
echo "  4. Setup backups"
echo ""
echo "Access your application:"
echo "  Admin Panel: $(php artisan config:show | grep 'app.url' | awk '{print $3}')"
echo "  API Endpoints: $(php artisan config:show | grep 'app.url' | awk '{print $3}')/api"
echo ""
echo -e "${GREEN}Deployment successful!${NC}"

