<?php

namespace App\Http\Controllers;

use App\Models\Complaint;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Carbon\Carbon;

class AnalyticsController extends Controller
{
    /**
     * Display analytics dashboard
     */
    public function index(Request $request)
    {
        $user = $request->user();
        
        // Get date range
        $startDate = $request->input('start_date', Carbon::now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->format('Y-m-d'));

        if (!Schema::hasTable('complaints')) {
            return view('analytics.index', [
                'stats' => [
                    'total' => 0,
                    'emergency' => 0,
                    'non_emergency' => 0,
                    'completed' => 0,
                    'pending' => 0,
                ],
                'heatmapData' => [],
                'categoryBreakdown' => [],
                'hourlyStats' => [],
                'responseTimeStats' => [
                    'avg_response_time' => 0,
                    'avg_resolution_time' => 0,
                    'min_response_time' => 0,
                    'max_response_time' => 0,
                ],
                'startDate' => $startDate,
                'endDate' => $endDate,
            ]);
        }

        // Get statistics
        $stats = $this->getStatistics($user, $startDate, $endDate);
        
        // Get heatmap data
        $heatmapData = $this->getHeatmapData($user, $startDate, $endDate);
        
        // Get category breakdown
        $categoryBreakdown = $this->getCategoryBreakdown($user, $startDate, $endDate);
        
        // Get hourly statistics
        $hourlyStats = $this->getHourlyStatistics($user, $startDate, $endDate);
        
        // Get response time statistics
        $responseTimeStats = $this->getResponseTimeStats($user, $startDate, $endDate);

        return view('analytics.index', compact(
            'stats',
            'heatmapData',
            'categoryBreakdown',
            'hourlyStats',
            'responseTimeStats',
            'startDate',
            'endDate'
        ));
    }

    /**
     * Get statistics
     */
    private function getStatistics(?Authenticatable $user, string $startDate, string $endDate): array
    {
        $query = Complaint::whereBetween('created_at', [$startDate, $endDate]);
        
        // Filter based on user role
        if ($user && $user->role === 'polda' && $user->polda_id) {
            $query->where('polda_id', $user->polda_id);
        } elseif ($user && $user->role === 'polres' && $user->polres_id) {
            $query->where('polres_id', $user->polres_id);
        } elseif ($user && $user->role === 'polsek' && $user->polsek_id) {
            $query->where('polsek_id', $user->polsek_id);
        }

        return [
            'total' => $query->count(),
            'emergency' => (clone $query)->whereIn('category', [
                'emergency_police',
                'emergency_medical',
                'emergency_fire',
                'emergency_disaster'
            ])->count(),
            'non_emergency' => (clone $query)->where('category', 'non_emergency')->count(),
            'completed' => (clone $query)->where('status', 'completed')->count(),
            'pending' => (clone $query)->where('status', 'pending')->count(),
        ];
    }

    /**
     * Get heatmap data
     */
    private function getHeatmapData(?Authenticatable $user, string $startDate, string $endDate): array
    {
        $query = Complaint::select(
            DB::raw('locations.latitude as lat'),
            DB::raw('locations.longitude as lng'),
            DB::raw('COUNT(*) as count'),
            DB::raw('complaints.category')
        )
        ->join('locations', 'complaints.location_id', '=', 'locations.id')
        ->whereBetween('complaints.created_at', [$startDate, $endDate])
        ->whereNotNull('locations.latitude')
        ->whereNotNull('locations.longitude')
        ->groupBy('locations.latitude', 'locations.longitude', 'complaints.category');
        
        // Filter based on user role
        if ($user && $user->role === 'polda' && $user->polda_id) {
            $query->where('complaints.polda_id', $user->polda_id);
        } elseif ($user && $user->role === 'polres' && $user->polres_id) {
            $query->where('complaints.polres_id', $user->polres_id);
        } elseif ($user && $user->role === 'polsek' && $user->polsek_id) {
            $query->where('complaints.polsek_id', $user->polsek_id);
        }

        return $query->get()->toArray();
    }

    /**
     * Get category breakdown
     */
    private function getCategoryBreakdown(?Authenticatable $user, string $startDate, string $endDate): array
    {
        $query = Complaint::select('category', DB::raw('COUNT(*) as count'))
            ->whereBetween('created_at', [$startDate, $endDate])
            ->groupBy('category');
        
        // Filter based on user role
        if ($user && $user->role === 'polda' && $user->polda_id) {
            $query->where('polda_id', $user->polda_id);
        } elseif ($user && $user->role === 'polres' && $user->polres_id) {
            $query->where('polres_id', $user->polres_id);
        } elseif ($user && $user->role === 'polsek' && $user->polsek_id) {
            $query->where('polsek_id', $user->polsek_id);
        }

        return $query->get()->pluck('count', 'category')->toArray();
    }

    /**
     * Get hourly statistics
     */
    private function getHourlyStatistics(?Authenticatable $user, string $startDate, string $endDate): array
    {
        $query = Complaint::select(
            DB::raw('HOUR(created_at) as hour'),
            DB::raw('COUNT(*) as count')
        )
        ->whereBetween('created_at', [$startDate, $endDate])
        ->groupBy(DB::raw('HOUR(created_at)'))
        ->orderBy('hour');
        
        // Filter based on user role
        if ($user && $user->role === 'polda' && $user->polda_id) {
            $query->where('polda_id', $user->polda_id);
        } elseif ($user && $user->role === 'polres' && $user->polres_id) {
            $query->where('polres_id', $user->polres_id);
        } elseif ($user && $user->role === 'polsek' && $user->polsek_id) {
            $query->where('polsek_id', $user->polsek_id);
        }

        return $query->get()->pluck('count', 'hour')->toArray();
    }

    /**
     * Get response time statistics
     */
    private function getResponseTimeStats(?Authenticatable $user, string $startDate, string $endDate): array
    {
        $query = Complaint::whereBetween('created_at', [$startDate, $endDate])
            ->whereNotNull('dispatched_at')
            ->whereNotNull('resolved_at');
        
        // Filter based on user role
        if ($user && $user->role === 'polda' && $user->polda_id) {
            $query->where('polda_id', $user->polda_id);
        } elseif ($user && $user->role === 'polres' && $user->polres_id) {
            $query->where('polres_id', $user->polres_id);
        } elseif ($user && $user->role === 'polsek' && $user->polsek_id) {
            $query->where('polsek_id', $user->polsek_id);
        }

        $complaints = $query->get();
        
        $responseTimes = $complaints->map(function($complaint) {
            return $complaint->dispatched_at->diffInMinutes($complaint->received_at);
        });

        $resolutionTimes = $complaints->map(function($complaint) {
            return $complaint->resolved_at->diffInMinutes($complaint->received_at);
        });

        return [
            'avg_response_time' => $responseTimes->avg(),
            'avg_resolution_time' => $resolutionTimes->avg(),
            'min_response_time' => $responseTimes->min(),
            'max_response_time' => $responseTimes->max(),
        ];
    }
}
