<?php

namespace App\Http\Controllers;

use App\Models\AutomationRule;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;

class AutomationRuleController extends Controller
{
    /**
     * Display a listing of automation rules
     */
    public function index(Request $request)
    {
        if (!Schema::hasTable('automation_rules')) {
            $rules = collect([]);
            return view('automation-rules.index', compact('rules'));
        }

        $query = AutomationRule::orderBy('priority', 'desc')
            ->orderBy('name');

        if ($request->has('is_active')) {
            $query->where('is_active', $request->is_active);
        }

        $rules = $query->paginate(20);

        return view('automation-rules.index', compact('rules'));
    }

    /**
     * Show the form for creating a new rule
     */
    public function create()
    {
        return view('automation-rules.create');
    }

    /**
     * Store a newly created rule
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'trigger_event' => 'required|string',
            'conditions' => 'required|string',
            'actions' => 'required|string',
            'action_params' => 'nullable|string',
            'is_active' => 'nullable|boolean',
            'priority' => 'nullable|integer|min:0|max:100',
        ]);

        // Decode JSON strings
        $validated['conditions'] = json_decode($validated['conditions'], true) ?? [];
        $validated['actions'] = json_decode($validated['actions'], true) ?? [];
        $validated['action_params'] = isset($validated['action_params']) ? (json_decode($validated['action_params'], true) ?? []) : [];
        $validated['is_active'] = $request->has('is_active') ? (bool)$request->is_active : true;
        $validated['priority'] = $validated['priority'] ?? 0;

        AutomationRule::create($validated);

        return redirect()->route('automation-rules.index')
            ->with('success', 'Automation rule created successfully.');
    }

    /**
     * Display the specified rule
     */
    public function show(AutomationRule $automationRule)
    {
        return view('automation-rules.show', compact('automationRule'));
    }

    /**
     * Show the form for editing the rule
     */
    public function edit(AutomationRule $automationRule)
    {
        return view('automation-rules.edit', compact('automationRule'));
    }

    /**
     * Update the specified rule
     */
    public function update(Request $request, AutomationRule $automationRule)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'trigger_event' => 'required|string',
            'conditions' => 'required|string',
            'actions' => 'required|string',
            'action_params' => 'nullable|string',
            'is_active' => 'nullable|boolean',
            'priority' => 'nullable|integer|min:0|max:100',
        ]);

        // Decode JSON strings
        $validated['conditions'] = json_decode($validated['conditions'], true) ?? [];
        $validated['actions'] = json_decode($validated['actions'], true) ?? [];
        $validated['action_params'] = isset($validated['action_params']) ? (json_decode($validated['action_params'], true) ?? []) : [];
        $validated['is_active'] = $request->has('is_active') ? (bool)$request->is_active : true;
        $validated['priority'] = $validated['priority'] ?? 0;

        $automationRule->update($validated);

        return redirect()->route('automation-rules.index')
            ->with('success', 'Automation rule updated successfully.');
    }

    /**
     * Remove the specified rule
     */
    public function destroy(AutomationRule $automationRule)
    {
        $automationRule->delete();

        return redirect()->route('automation-rules.index')
            ->with('success', 'Automation rule deleted successfully.');
    }
}

