<?php

namespace App\Http\Controllers;

use App\Models\Complaint;
use App\Models\Officer;
use App\Services\NotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class BroadcastController extends Controller
{
    public function __construct(
        private NotificationService $notificationService
    ) {}

    /**
     * Display broadcast management
     */
    public function index(Request $request)
    {
        // Get recent broadcasts (stored in notifications table with type=broadcast)
        $broadcasts = \App\Models\Notification::where('type', 'broadcast')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('broadcasts.index', compact('broadcasts'));
    }

    /**
     * Store a new broadcast
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'type' => 'required|in:emergency,disaster,crime_alert,general',
            'priority' => 'required|in:low,medium,high,critical',
            'target_scope' => 'required|in:national,polda,polres,polsek',
            'target_polda_id' => 'nullable|exists:poldas,id',
            'target_polres_id' => 'nullable|exists:polres,id',
            'target_polsek_id' => 'nullable|exists:polsek,id',
            'channels' => 'required|array',
            'channels.*' => 'in:mobile_app,whatsapp,sms,radio,email',
        ]);

        // Create notification record
        $notification = \App\Models\Notification::create([
            'type' => 'broadcast',
            'title' => $validated['title'],
            'message' => $validated['message'],
            'metadata' => [
                'priority' => $validated['priority'],
                'broadcast_type' => $validated['type'],
                'target_scope' => $validated['target_scope'],
                'target_polda_id' => $validated['target_polda_id'] ?? null,
                'target_polres_id' => $validated['target_polres_id'] ?? null,
                'target_polsek_id' => $validated['target_polsek_id'] ?? null,
                'channels' => $validated['channels'],
            ],
            'sent_at' => now(),
        ]);

        // Send broadcast to all target recipients
        $this->sendBroadcast($notification, $validated);

        return response()->json([
            'success' => true,
            'message' => 'Broadcast berhasil dikirim',
            'notification' => $notification,
        ], 201);
    }

    /**
     * Send broadcast to recipients
     */
    private function sendBroadcast($notification, array $validated): void
    {
        // Get target officers/users
        $query = Officer::where('is_active', true);
        
        if ($validated['target_scope'] === 'polda' && $validated['target_polda_id']) {
            $query->where('polda_id', $validated['target_polda_id']);
        } elseif ($validated['target_scope'] === 'polres' && $validated['target_polres_id']) {
            $query->where('polres_id', $validated['target_polres_id']);
        } elseif ($validated['target_scope'] === 'polsek' && $validated['target_polsek_id']) {
            $query->where('polsek_id', $validated['target_polsek_id']);
        }

        $officers = $query->get();

        foreach ($officers as $officer) {
            if (in_array('mobile_app', $validated['channels']) && $officer->user && $officer->user->device_token) {
                $this->notificationService->sendPushNotification(
                    $officer->user->device_token,
                    [
                        'title' => $validated['title'],
                        'body' => $validated['message'],
                    ]
                );
            }

            if (in_array('whatsapp', $validated['channels']) && $officer->phone) {
                $this->notificationService->sendWhatsApp(
                    $officer->phone,
                    $validated['message']
                );
            }

            if (in_array('sms', $validated['channels']) && $officer->phone) {
                $this->notificationService->sendSMS(
                    $officer->phone,
                    $validated['message']
                );
            }

            if (in_array('radio', $validated['channels'])) {
                $this->sendRadioBroadcast($officer, $validated['message']);
            }
        }
    }

    /**
     * Send radio broadcast
     */
    private function sendRadioBroadcast(Officer $officer, string $message): void
    {
        // Integrate with radio gateway
        // This would typically use a radio gateway API
        $radioGatewayUrl = config('services.radio_gateway.api_url');
        $radioGatewayKey = config('services.radio_gateway.api_key');

        if (!$radioGatewayUrl || !$radioGatewayKey) {
            return;
        }

        try {
            \Illuminate\Support\Facades\Http::withHeaders([
                'Authorization' => 'Bearer ' . $radioGatewayKey,
            ])->post($radioGatewayUrl . '/broadcast', [
                'channel' => $officer->radio_channel ?? 'default',
                'message' => $message,
                'officer_id' => $officer->id,
            ]);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Radio broadcast failed: ' . $e->getMessage());
        }
    }
}

