<?php

namespace App\Http\Controllers;

use App\Models\Coordination;
use App\Models\Complaint;
use App\Services\NotificationService;
use Illuminate\Http\Request;

class CoordinationController extends Controller
{
    public function __construct(
        private NotificationService $notificationService
    ) {}

    /**
     * Display a listing of coordinations
     */
    public function index(Request $request)
    {
        $query = Coordination::with(['complaint', 'sender'])
            ->orderBy('created_at', 'desc');

        if ($request->has('complaint_id')) {
            $query->where('complaint_id', $request->complaint_id);
        }

        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        $coordinations = $query->paginate(20);

        return view('coordinations.index', compact('coordinations'));
    }

    /**
     * Store a newly created coordination
     */
    public function store(Request $request, Complaint $complaint)
    {
        $validated = $request->validate([
            'type' => 'required|in:internal,external',
            'internal_type' => 'required_if:type,internal|in:operator_polsek,polda_polres,polres_polsek,chat',
            'external_type' => 'required_if:type,external|in:damkar,rs,bpbd,bnpb,basarnas,dishub,pemda',
            'recipient_name' => 'nullable|string',
            'recipient_phone' => 'nullable|string',
            'recipient_email' => 'nullable|email',
            'message' => 'required|string',
        ]);

        $coordination = Coordination::create([
            'complaint_id' => $complaint->id,
            'type' => $validated['type'],
            'internal_type' => $validated['internal_type'] ?? null,
            'external_type' => $validated['external_type'] ?? null,
            'recipient_name' => $validated['recipient_name'] ?? null,
            'recipient_phone' => $validated['recipient_phone'] ?? null,
            'recipient_email' => $validated['recipient_email'] ?? null,
            'message' => $validated['message'],
            'status' => 'pending',
            'sender_id' => $request->user()->id,
        ]);

        // Send notification
        if ($coordination->type === 'external' && $coordination->recipient_phone) {
            $this->notificationService->sendSMS(
                $coordination->recipient_phone,
                $coordination->message
            );
        }

        $coordination->update([
            'status' => 'sent',
            'sent_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'coordination' => $coordination,
        ], 201);
    }

    /**
     * Acknowledge coordination
     */
    public function acknowledge(Coordination $coordination)
    {
        $coordination->update([
            'status' => 'acknowledged',
            'acknowledged_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Coordination acknowledged',
        ]);
    }
}
