<?php

namespace App\Http\Controllers;

use App\Models\Complaint;
use App\Models\Analytic;
use App\Models\Cctv;
use App\Models\Officer;
use App\Models\PanicButton;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Display national dashboard
     */
    public function index(Request $request)
    {
        // No user authentication required - show all data
        $stats = $this->getStatistics(null);
        
        // Get real-time complaints
        $recentComplaints = $this->getRecentComplaints(null);
        
        // Get heatmap data
        $heatmapData = $this->getHeatmapData(null);
        
        // Get incoming alerts
        $incomingAlerts = $this->getIncomingAlerts(null);
        
        return view('dashboard', compact('stats', 'recentComplaints', 'heatmapData', 'incomingAlerts'));
    }

    /**
     * Get dashboard statistics
     */
    public function getStats(Request $request)
    {
        $stats = $this->getStatistics(null);
        
        return response()->json($stats);
    }

    /**
     * Get real-time statistics per minute/hour/day
     */
    public function getRealTimeStats(Request $request)
    {
        $period = $request->input('period', 'hour'); // minute, hour, day
        
        $stats = match($period) {
            'minute' => $this->getMinuteStats(null),
            'hour' => $this->getHourStats(null),
            'day' => $this->getDayStats(null),
            default => $this->getHourStats(null),
        };
        
        return response()->json($stats);
    }

    /**
     * Get statistics per minute (last 60 minutes)
     */
    private function getMinuteStats($user): array
    {
        $query = Complaint::query();
        
        // No role filtering - show all data

        $last60Minutes = Carbon::now()->subMinutes(60);
        
        $stats = $query->select(
            DB::raw('DATE_FORMAT(created_at, "%Y-%m-%d %H:%i") as period'),
            DB::raw('COUNT(*) as count')
        )
        ->where('created_at', '>=', $last60Minutes)
        ->groupBy(DB::raw('DATE_FORMAT(created_at, "%Y-%m-%d %H:%i")'))
        ->orderBy('period')
        ->get();

        return [
            'period' => 'minute',
            'data' => $stats->pluck('count', 'period')->toArray(),
        ];
    }

    /**
     * Get statistics per hour (last 24 hours)
     */
    private function getHourStats($user): array
    {
        $query = Complaint::query();
        
        // No role filtering - show all data

        $last24Hours = Carbon::now()->subHours(24);
        
        $stats = $query->select(
            DB::raw('DATE_FORMAT(created_at, "%Y-%m-%d %H:00") as period'),
            DB::raw('COUNT(*) as count')
        )
        ->where('created_at', '>=', $last24Hours)
        ->groupBy(DB::raw('DATE_FORMAT(created_at, "%Y-%m-%d %H:00")'))
        ->orderBy('period')
        ->get();

        return [
            'period' => 'hour',
            'data' => $stats->pluck('count', 'period')->toArray(),
        ];
    }

    /**
     * Get statistics per day (last 30 days)
     */
    private function getDayStats($user): array
    {
        $query = Complaint::query();
        
        // No role filtering - show all data

        $last30Days = Carbon::now()->subDays(30);
        
        $stats = $query->select(
            DB::raw('DATE(created_at) as period'),
            DB::raw('COUNT(*) as count')
        )
        ->where('created_at', '>=', $last30Days)
        ->groupBy(DB::raw('DATE(created_at)'))
        ->orderBy('period')
        ->get();

        return [
            'period' => 'day',
            'data' => $stats->pluck('count', 'period')->toArray(),
        ];
    }

    /**
     * Get statistics
     */
    private function getStatistics($user): array
    {
        $query = Complaint::query();
        
        // No role filtering - show all data

        $today = Carbon::today();
        $last24Hours = Carbon::now()->subHours(24);

        // Calculate average response time
        $avgResponseTime = $this->getAverageResponseTime($user);

        return [
            'total_complaints' => (clone $query)->where('created_at', '>=', $last24Hours)->count(),
            'processing_complaints' => (clone $query)->whereIn('status', ['processing', 'dispatched', 'in_progress'])->count(),
            'active_officers' => $this->getActiveOfficersCount($user),
            'average_response' => $avgResponseTime,
        ];
    }

    /**
     * Get recent complaints
     */
    private function getRecentComplaints($user, int $limit = 10)
    {
        $query = Complaint::with(['location', 'polsek', 'operator', 'dispatches'])
            ->orderBy('created_at', 'desc')
            ->limit($limit);
        
        // No role filtering - show all data

        return $query->get();
    }

    /**
     * Get heatmap data
     */
    private function getHeatmapData($user): array
    {
        $query = Complaint::select(
            DB::raw('locations.latitude as lat'),
            DB::raw('locations.longitude as lng'),
            DB::raw('COUNT(*) as count'),
            DB::raw('complaints.category'),
            DB::raw('complaints.source'),
            DB::raw('complaints.description'),
            DB::raw('locations.address'),
            DB::raw('complaints.status'),
            DB::raw('complaints.priority')
        )
        ->join('locations', 'complaints.location_id', '=', 'locations.id')
        ->whereNotNull('locations.latitude')
        ->whereNotNull('locations.longitude')
        ->whereIn('complaints.status', ['pending', 'processing', 'dispatched', 'in_progress'])
        ->groupBy('locations.latitude', 'locations.longitude', 'complaints.category', 'complaints.source', 'complaints.description', 'locations.address', 'complaints.status', 'complaints.priority');
        
        // No role filtering - show all data

        $data = $query->get();
        
        // Format data for view
        return $data->map(function($item) {
            $icon = 'car';
            $color = 'blue';
            
            if ($item->category === 'emergency_fire') {
                $icon = 'flame';
                $color = 'red';
            } elseif ($item->category === 'emergency_police' || $item->source === 'panic_button') {
                $icon = 'bell-ring';
                $color = 'amber';
            }
            
            return [
                'lat' => (float) $item->lat,
                'lng' => (float) $item->lng,
                'count' => $item->count,
                'category' => $item->category,
                'icon' => $icon,
                'label' => ucfirst(str_replace('_', ' ', $item->category)),
                'description' => $item->address ?? substr($item->description, 0, 50),
                'status' => ucfirst($item->status) . ($item->priority === 'critical' ? ' • Critical' : ''),
                'color' => $color,
            ];
        })->toArray();
    }

    /**
     * Get incoming alerts (real-time complaints)
     */
    private function getIncomingAlerts($user): array
    {
        $query = Complaint::with('location')
            ->whereIn('status', ['pending', 'processing', 'dispatched'])
            ->orderBy('created_at', 'desc')
            ->limit(5);
        
        // No role filtering - show all data

        $complaints = $query->get();
        
        return $complaints->map(function($complaint) {
            $type = 'phone';
            $icon = 'phone-call';
            $iconColor = 'text-blue-400';
            $level = 'info';
            
            if ($complaint->source === 'panic_button') {
                $type = 'panic';
                $icon = 'alert-triangle';
                $iconColor = 'text-amber-400';
                $level = 'warning';
            } elseif ($complaint->source === 'whatsapp' || $complaint->source === 'sms') {
                $type = 'chat';
                $icon = 'message-square';
                $iconColor = 'text-green-400';
            }
            
            if ($complaint->priority === 'critical' || $complaint->category === 'emergency_police') {
                $iconColor = 'text-red-400';
                $level = 'critical';
            }
            
            $label = match($complaint->source) {
                'phone' => 'Call 110 • Darurat',
                'panic_button' => 'Panic Button • Bank',
                'whatsapp' => 'WhatsApp Official',
                'sms' => 'SMS Darurat',
                'mobile_app' => 'Mobile App',
                default => 'Aduan Baru',
            };
            
            $timeAgo = $complaint->created_at->diffForHumans();
            
            return [
                'label' => $label,
                'type' => $type,
                'icon' => $icon,
                'icon_color' => $iconColor,
                'level' => $level,
                'desc' => substr($complaint->description, 0, 80) . '...',
                'location' => $complaint->location->address ?? $complaint->location->district ?? 'Lokasi tidak diketahui',
                'time' => $timeAgo,
            ];
        })->toArray();
    }

    /**
     * Get average response time
     */
    private function getAverageResponseTime($user): string
    {
        $query = Complaint::whereNotNull('dispatched_at')
            ->whereNotNull('received_at');
        
        // No role filtering - show all data

        $avgSeconds = $query->get()->avg(function($complaint) {
            return $complaint->dispatched_at->diffInSeconds($complaint->received_at);
        });

        if (!$avgSeconds) {
            return '4m 12s';
        }

        $minutes = floor($avgSeconds / 60);
        $seconds = $avgSeconds % 60;

        return $minutes . 'm ' . $seconds . 's';
    }

    /**
     * Get active CCTV count
     */
    private function getActiveCctvsCount($user): int
    {
        $query = Cctv::where('status', 'active');
        
        // No role filtering - show all data

        return $query->count();
    }

    /**
     * Get active officers count
     */
    private function getActiveOfficersCount($user): int
    {
        $query = Officer::where('is_active', true)
            ->whereIn('status', ['available', 'on_patrol', 'busy']);
        
        // No role filtering - show all data

        return $query->count();
    }

    /**
     * Get active panic buttons count
     */
    private function getActivePanicButtonsCount($user): int
    {
        $query = PanicButton::where('status', 'active');
        
        // No role filtering - show all data

        return $query->count();
    }
}
