<?php

namespace App\Http\Controllers;

use App\Models\KnowledgeCenter;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;

class KnowledgeCenterController extends Controller
{
    /**
     * Display a listing of knowledge center articles
     */
    public function index(Request $request)
    {
        if (!Schema::hasTable('knowledge_centers')) {
            $articles = $this->emptyPaginator(12);
            return view('knowledge-center.index', compact('articles'));
        }

        $query = KnowledgeCenter::with('author')
            ->where('is_published', true)
            ->orderBy('created_at', 'desc');

        if ($request->has('category')) {
            $query->where('category', $request->category);
        }

        if ($request->has('search')) {
            $query->where(function($q) use ($request) {
                $q->where('title', 'like', '%' . $request->search . '%')
                  ->orWhere('content', 'like', '%' . $request->search . '%');
            });
        }

        $articles = $query->paginate(12);

        return view('knowledge-center.index', compact('articles'));
    }

    /**
     * Display the specified article
     */
    public function show(KnowledgeCenter $knowledgeCenter)
    {
        // Increment views
        $knowledgeCenter->increment('views');

        return view('knowledge-center.show', compact('knowledgeCenter'));
    }

    /**
     * Store a newly created article (admin only)
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'category' => 'required|in:safety_guide,digital_safety,emergency_procedure,training,security_article',
            'content' => 'required|string',
            'excerpt' => 'nullable|string',
            'featured_image' => 'nullable|image|max:2048',
            'is_published' => 'boolean',
        ]);

        $article = KnowledgeCenter::create([
            'title' => $validated['title'],
            'slug' => Str::slug($validated['title']),
            'category' => $validated['category'],
            'content' => $validated['content'],
            'excerpt' => $validated['excerpt'] ?? Str::limit(strip_tags($validated['content']), 200),
            'author_id' => $request->user()->id,
            'is_published' => $validated['is_published'] ?? false,
        ]);

        if ($request->hasFile('featured_image')) {
            $article->featured_image = $request->file('featured_image')->store('knowledge-center', 'public');
            $article->save();
        }

        return response()->json([
            'success' => true,
            'article' => $article,
        ], 201);
    }
}
