<?php

namespace App\Http\Controllers;

use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;

class NotificationController extends Controller
{
    /**
     * Display notification and automation management
     */
    public function index(Request $request)
    {
        if (!Schema::hasTable('notifications')) {
            $notifications = collect([]);
            $automationRules = [];
        } else {
            $notifications = Notification::with(['user', 'complaint'])
                ->orderBy('created_at', 'desc')
                ->paginate(20);

            // Get automation rules (stored in config or database)
            $automationRules = $this->getAutomationRules();
        }

        return view('notifications.index', compact('notifications', 'automationRules'));
    }

    /**
     * Get automation rules
     */
    private function getAutomationRules(): array
    {
        // Try to get from database first
        if (Schema::hasTable('automation_rules')) {
            return \App\Models\AutomationRule::where('is_active', true)
                ->orderBy('priority', 'desc')
                ->get()
                ->map(function($rule) {
                    return [
                        'id' => $rule->id,
                        'name' => $rule->name,
                        'description' => $rule->description,
                        'trigger' => $rule->trigger_event,
                        'conditions' => $rule->conditions,
                        'actions' => $rule->actions,
                        'is_active' => $rule->is_active,
                    ];
                })
                ->toArray();
        }
        
        // Fallback to hardcoded rules
        return [
            [
                'id' => 1,
                'name' => 'Auto Dispatch Critical Emergency',
                'description' => 'Otomatis dispatch unit terdekat untuk aduan critical',
                'trigger' => 'complaint.created',
                'conditions' => ['priority' => 'critical'],
                'actions' => ['auto_dispatch', 'notify_supervisor'],
                'is_active' => true,
            ],
            [
                'id' => 2,
                'name' => 'Eskalasi Non-Response',
                'description' => 'Eskalasi ke Polres jika Polsek tidak merespons dalam 5 menit',
                'trigger' => 'dispatch.no_response',
                'conditions' => ['timeout' => 300],
                'actions' => ['escalate', 'notify_polres'],
                'is_active' => true,
            ],
            [
                'id' => 3,
                'name' => 'Auto Capture CCTV',
                'description' => 'Otomatis capture snapshot CCTV saat aduan masuk',
                'trigger' => 'complaint.created',
                'conditions' => ['has_location' => true],
                'actions' => ['capture_cctv'],
                'is_active' => true,
            ],
            [
                'id' => 4,
                'name' => 'Broadcast Bencana',
                'description' => 'Broadcast peringatan ke semua unit saat bencana',
                'trigger' => 'complaint.category',
                'conditions' => ['category' => 'emergency_disaster'],
                'actions' => ['broadcast_alert', 'notify_all_units'],
                'is_active' => true,
            ],
        ];
    }
}

