<?php

namespace App\Http\Controllers;

use App\Models\NotificationTemplate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;

class NotificationTemplateController extends Controller
{
    /**
     * Display a listing of notification templates
     */
    public function index(Request $request)
    {
        if (!Schema::hasTable('notification_templates')) {
            $templates = collect([]);
            return view('notification-templates.index', compact('templates'));
        }

        $query = NotificationTemplate::orderBy('type')
            ->orderBy('name');

        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        if ($request->has('is_active')) {
            $query->where('is_active', $request->is_active);
        }

        $templates = $query->paginate(20);

        return view('notification-templates.index', compact('templates'));
    }

    /**
     * Show the form for creating a new template
     */
    public function create()
    {
        return view('notification-templates.create');
    }

    /**
     * Store a newly created template
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:sms,whatsapp,email,push',
            'subject' => 'nullable|string|max:255',
            'body' => 'required|string',
            'variables' => 'nullable|string',
            'is_active' => 'nullable|boolean',
        ]);

        // Decode JSON string if provided
        if (isset($validated['variables']) && is_string($validated['variables'])) {
            $validated['variables'] = json_decode($validated['variables'], true) ?? [];
        } else {
            $validated['variables'] = [];
        }
        $validated['is_active'] = $request->has('is_active') ? (bool)$request->is_active : true;

        NotificationTemplate::create($validated);

        return redirect()->route('notification-templates.index')
            ->with('success', 'Notification template created successfully.');
    }

    /**
     * Display the specified template
     */
    public function show(NotificationTemplate $notificationTemplate)
    {
        return view('notification-templates.show', compact('notificationTemplate'));
    }

    /**
     * Show the form for editing the template
     */
    public function edit(NotificationTemplate $notificationTemplate)
    {
        return view('notification-templates.edit', compact('notificationTemplate'));
    }

    /**
     * Update the specified template
     */
    public function update(Request $request, NotificationTemplate $notificationTemplate)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:sms,whatsapp,email,push',
            'subject' => 'nullable|string|max:255',
            'body' => 'required|string',
            'variables' => 'nullable|string',
            'is_active' => 'nullable|boolean',
        ]);

        // Decode JSON string if provided
        if (isset($validated['variables']) && is_string($validated['variables'])) {
            $validated['variables'] = json_decode($validated['variables'], true) ?? [];
        } else {
            $validated['variables'] = [];
        }
        $validated['is_active'] = $request->has('is_active') ? (bool)$request->is_active : true;

        $notificationTemplate->update($validated);

        return redirect()->route('notification-templates.index')
            ->with('success', 'Notification template updated successfully.');
    }

    /**
     * Remove the specified template
     */
    public function destroy(NotificationTemplate $notificationTemplate)
    {
        $notificationTemplate->delete();

        return redirect()->route('notification-templates.index')
            ->with('success', 'Notification template deleted successfully.');
    }
}

