<?php

namespace App\Http\Controllers;

use App\Models\PanicButton;
use App\Models\PanicButtonActivation;
use App\Models\Complaint;
use App\Services\LocationTrackingService;
use App\Services\DispatchService;
use App\Services\CctvService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;

class PanicButtonController extends Controller
{
    public function __construct(
        private LocationTrackingService $locationTrackingService,
        private DispatchService $dispatchService,
        private CctvService $cctvService
    ) {}

    /**
     * Display a listing of panic buttons
     */
    public function index(Request $request)
    {
        if (!Schema::hasTable('panic_buttons')) {
            $panicButtons = $this->emptyPaginator();
            return view('panic-buttons.index', compact('panicButtons'));
        }

        $query = PanicButton::with(['polda', 'polres', 'polsek'])
            ->orderBy('name');

        // Filter based on user role
        $user = $request->user();
        if ($user && $user->role === 'polda' && $user->polda_id) {
            $query->where('polda_id', $user->polda_id);
        } elseif ($user && $user->role === 'polres' && $user->polres_id) {
            $query->where('polres_id', $user->polres_id);
        } elseif ($user && $user->role === 'polsek' && $user->polsek_id) {
            $query->where('polsek_id', $user->polsek_id);
        }

        // Apply filters
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        $panicButtons = $query->paginate(20);

        return view('panic-buttons.index', compact('panicButtons'));
    }

    /**
     * Display the specified panic button
     */
    public function show(PanicButton $panicButton)
    {
        $panicButton->load(['polda', 'polres', 'polsek', 'activations.complaint']);
        
        return view('panic-buttons.show', compact('panicButton'));
    }

    /**
     * Activate panic button
     */
    public function activate(Request $request, PanicButton $panicButton)
    {
        // Create location from panic button coordinates
        $location = $this->locationTrackingService->trackFromGps(
            $panicButton->latitude,
            $panicButton->longitude
        );

        // Find responsible Polsek
        $polsek = $this->locationTrackingService->findResponsiblePolsek($location);

        // Create complaint
        $complaint = Complaint::create([
            'ticket_number' => $this->generateTicketNumber(),
            'source' => 'panic_button',
            'description' => "Panic button diaktifkan di {$panicButton->name}",
            'category' => 'emergency_police',
            'priority' => 'critical',
            'status' => 'pending',
            'location_id' => $location->id,
            'polda_id' => $polsek?->polres?->polda_id,
            'polres_id' => $polsek?->polres_id,
            'polsek_id' => $polsek?->id,
            'received_at' => now(),
        ]);

        // Create activation record
        $activation = PanicButtonActivation::create([
            'panic_button_id' => $panicButton->id,
            'complaint_id' => $complaint->id,
            'activated_at' => now(),
            'status' => 'active',
        ]);

        // Update panic button last activated
        $panicButton->update(['last_activated_at' => now()]);

        // Auto-capture CCTV snapshots
        $this->cctvService->autoCaptureForComplaint($complaint);

        // Auto-dispatch
        $dispatch = $this->dispatchService->autoDispatch($complaint);

        return response()->json([
            'success' => true,
            'activation' => $activation,
            'complaint' => $complaint,
            'dispatch' => $dispatch,
        ], 201);
    }

    /**
     * Generate unique ticket number
     */
    private function generateTicketNumber(): string
    {
        $date = now()->format('Ymd');
        $random = strtoupper(substr(md5(uniqid()), 0, 6));
        return "PANIC-{$date}-{$random}";
    }
}
