<?php

namespace App\Jobs;

use App\Models\Notification;
use App\Services\NotificationService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class SendNotificationJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     */
    public function __construct(
        public Notification $notification
    ) {}

    /**
     * Execute the job.
     */
    public function handle(NotificationService $notificationService): void
    {
        $notification = $this->notification;

        try {
            // Send based on channel
            switch ($notification->channel) {
                case 'whatsapp':
                    if ($notification->user && $notification->user->phone) {
                        $notificationService->sendWhatsApp(
                            $notification->user->phone,
                            $notification->message
                        );
                    }
                    break;
                case 'sms':
                    if ($notification->user && $notification->user->phone) {
                        $notificationService->sendSMS(
                            $notification->user->phone,
                            $notification->message
                        );
                    }
                    break;
                case 'mobile_app':
                    if ($notification->user && $notification->user->device_token) {
                        $notificationService->sendPushNotification(
                            $notification->user->device_token,
                            [
                                'title' => $notification->title,
                                'body' => $notification->message,
                                'data' => $notification->metadata ?? [],
                            ]
                        );
                    }
                    break;
            }

            $notification->update([
                'status' => 'sent',
                'sent_at' => now(),
            ]);
        } catch (\Exception $e) {
            $notification->update([
                'status' => 'failed',
            ]);
            throw $e;
        }
    }
}
