<?php

namespace App\Services;

use App\Models\Cctv;
use App\Models\CctvSnapshot;
use App\Models\Complaint;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class CctvService
{
    /**
     * Find nearby CCTV cameras
     */
    public function findNearbyCctvs(float $latitude, float $longitude, int $radiusMeters = 500): \Illuminate\Database\Eloquent\Collection
    {
        return Cctv::where('status', 'active')
            ->selectRaw('*, (
                6371000 * acos(
                    cos(radians(?)) * cos(radians(latitude)) *
                    cos(radians(longitude) - radians(?)) +
                    sin(radians(?)) * sin(radians(latitude))
                )
            ) AS distance', [
                $latitude,
                $longitude,
                $latitude
            ])
            ->having('distance', '<=', $radiusMeters)
            ->orderBy('distance')
            ->get();
    }

    /**
     * Capture snapshot from CCTV
     */
    public function captureSnapshot(Cctv $cctv, ?Complaint $complaint = null): ?CctvSnapshot
    {
        try {
            // Capture snapshot from RTSP stream
            $snapshotPath = $this->captureFromRtsp($cctv->rtsp_url);
            
            if (!$snapshotPath) {
                return null;
            }

            // Store snapshot
            $storedPath = Storage::disk('public')->put('cctv/snapshots', file_get_contents($snapshotPath));
            
            return CctvSnapshot::create([
                'cctv_id' => $cctv->id,
                'complaint_id' => $complaint?->id,
                'snapshot_path' => $storedPath,
                'snapshot_url' => Storage::disk('public')->url($storedPath),
                'type' => $complaint ? 'event_triggered' : 'realtime',
                'captured_at' => now(),
            ]);
        } catch (\Exception $e) {
            Log::error("CCTV snapshot failed: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Auto-capture snapshots for complaint location
     */
    public function autoCaptureForComplaint(Complaint $complaint): array
    {
        if (!$complaint->location) {
            return [];
        }

        $cctvs = $this->findNearbyCctvs(
            $complaint->location->latitude,
            $complaint->location->longitude,
            300 // 300 meters radius
        );

        $snapshots = [];
        foreach ($cctvs as $cctv) {
            $snapshot = $this->captureSnapshot($cctv, $complaint);
            if ($snapshot) {
                $snapshots[] = $snapshot;
            }
        }

        return $snapshots;
    }

    /**
     * Get live stream URL
     */
    public function getLiveStreamUrl(Cctv $cctv): ?string
    {
        // Convert RTSP to HLS or WebRTC for web streaming
        // This would typically use a media server like Wowza, Kurento, or similar
        return $cctv->stream_url ?? $cctv->rtsp_url;
    }

    /**
     * Capture snapshot from RTSP stream
     */
    private function captureFromRtsp(string $rtspUrl): ?string
    {
        // Use ffmpeg to capture snapshot from RTSP
        // This requires ffmpeg to be installed on the server
        try {
            $outputPath = storage_path('app/temp/' . uniqid() . '.jpg');
            
            $command = sprintf(
                'ffmpeg -rtsp_transport tcp -i "%s" -frames:v 1 -q:v 2 "%s" -y 2>&1',
                escapeshellarg($rtspUrl),
                escapeshellarg($outputPath)
            );

            exec($command, $output, $returnCode);

            if ($returnCode === 0 && file_exists($outputPath)) {
                return $outputPath;
            }
        } catch (\Exception $e) {
            Log::error("RTSP capture failed: " . $e->getMessage());
        }

        return null;
    }
}
