<?php

namespace App\Services;

use App\Models\Location;
use App\Models\Polsek;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class LocationTrackingService
{
    /**
     * Track location from GPS coordinates
     */
    public function trackFromGps(float $latitude, float $longitude, ?float $accuracy = null): Location
    {
        $address = $this->reverseGeocode($latitude, $longitude);
        
        return Location::create([
            'latitude' => $latitude,
            'longitude' => $longitude,
            'tracking_method' => 'gps',
            'accuracy' => $accuracy,
            'address' => $address['address'] ?? null,
            'village' => $address['village'] ?? null,
            'district' => $address['district'] ?? null,
            'regency' => $address['regency'] ?? null,
            'province' => $address['province'] ?? null,
            'postal_code' => $address['postal_code'] ?? null,
        ]);
    }

    /**
     * Track location from Cell-ID
     */
    public function trackFromCellId(string $cellId, ?string $towerId = null): ?Location
    {
        // Integrate with provider API for Cell-ID tracking
        // This is a placeholder - actual implementation requires provider API
        try {
            $location = $this->getLocationFromCellId($cellId, $towerId);
            
            if ($location) {
                return Location::create([
                    'latitude' => $location['latitude'],
                    'longitude' => $location['longitude'],
                    'tracking_method' => 'cell_id',
                    'cell_id' => $cellId,
                    'tower_id' => $towerId,
                    'accuracy' => $location['accuracy'] ?? null,
                    'address' => $location['address'] ?? null,
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Cell-ID tracking failed: ' . $e->getMessage());
        }

        return null;
    }

    /**
     * Track location from phone number area code
     */
    public function trackFromAreaCode(string $phoneNumber): ?Location
    {
        // Extract area code from phone number
        $areaCode = $this->extractAreaCode($phoneNumber);
        
        if (!$areaCode) {
            return null;
        }

        // Get approximate location from area code database
        $location = $this->getLocationFromAreaCode($areaCode);
        
        if ($location) {
            return Location::create([
                'latitude' => $location['latitude'],
                'longitude' => $location['longitude'],
                'tracking_method' => 'area_code',
                'address' => $location['address'] ?? null,
                'district' => $location['district'] ?? null,
                'regency' => $location['regency'] ?? null,
                'province' => $location['province'] ?? null,
            ]);
        }

        return null;
    }

    /**
     * Auto-track location based on available data
     */
    public function autoTrack(?float $latitude = null, ?float $longitude = null, ?string $cellId = null, ?string $phoneNumber = null): ?Location
    {
        // Priority: GPS > Cell-ID > Area Code
        if ($latitude && $longitude) {
            return $this->trackFromGps($latitude, $longitude);
        }

        if ($cellId) {
            $location = $this->trackFromCellId($cellId);
            if ($location) {
                return $location;
            }
        }

        if ($phoneNumber) {
            return $this->trackFromAreaCode($phoneNumber);
        }

        return null;
    }

    /**
     * Find responsible Polsek based on location
     */
    public function findResponsiblePolsek(Location $location): ?Polsek
    {
        // Find nearest Polsek based on coordinates
        $polsek = Polsek::where('is_active', true)
            ->selectRaw('*, (
                6371 * acos(
                    cos(radians(?)) * cos(radians(latitude)) *
                    cos(radians(longitude) - radians(?)) +
                    sin(radians(?)) * sin(radians(latitude))
                )
            ) AS distance', [
                $location->latitude,
                $location->longitude,
                $location->latitude
            ])
            ->orderBy('distance')
            ->first();

        return $polsek;
    }

    /**
     * Reverse geocode coordinates to address
     */
    private function reverseGeocode(float $latitude, float $longitude): array
    {
        // Use OpenStreetMap Nominatim or Google Geocoding API
        try {
            $response = Http::get('https://nominatim.openstreetmap.org/reverse', [
                'lat' => $latitude,
                'lon' => $longitude,
                'format' => 'json',
                'addressdetails' => 1,
            ]);

            if ($response->successful()) {
                $data = $response->json();
                $address = $data['address'] ?? [];

                return [
                    'address' => $data['display_name'] ?? null,
                    'village' => $address['village'] ?? $address['suburb'] ?? null,
                    'district' => $address['county'] ?? null,
                    'regency' => $address['city'] ?? $address['town'] ?? null,
                    'province' => $address['state'] ?? null,
                    'postal_code' => $address['postcode'] ?? null,
                ];
            }
        } catch (\Exception $e) {
            Log::error('Reverse geocoding failed: ' . $e->getMessage());
        }

        return [];
    }

    /**
     * Get location from Cell-ID (requires provider API integration)
     */
    private function getLocationFromCellId(string $cellId, ?string $towerId = null): ?array
    {
        // Placeholder - integrate with provider API
        // This would typically call Telkomsel, Indosat, XL, etc. API
        return null;
    }

    /**
     * Extract area code from phone number
     */
    private function extractAreaCode(string $phoneNumber): ?string
    {
        // Remove non-numeric characters
        $phoneNumber = preg_replace('/[^0-9]/', '', $phoneNumber);
        
        // Indonesian phone number format
        if (strlen($phoneNumber) >= 10) {
            // Mobile: 08xx -> area code is first 4 digits
            if (substr($phoneNumber, 0, 2) === '08') {
                return substr($phoneNumber, 0, 4);
            }
            // Landline: 0xx -> area code is first 3-4 digits
            if (substr($phoneNumber, 0, 1) === '0') {
                return substr($phoneNumber, 0, 3);
            }
        }

        return null;
    }

    /**
     * Get location from area code (requires area code database)
     */
    private function getLocationFromAreaCode(string $areaCode): ?array
    {
        // Placeholder - would query area code database
        // This maps area codes to approximate locations
        return null;
    }
}
