<?php

namespace App\Services;

use App\Models\Dispatch;
use App\Models\Notification;
use App\Models\Officer;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class NotificationService
{
    /**
     * Send dispatch notification
     */
    public function sendDispatchNotification(Dispatch $dispatch): void
    {
        $polsek = $dispatch->polsek;
        
        // Notify Polsek
        $this->createNotification([
            'type' => 'dispatch_new',
            'title' => 'Dispatch Baru',
            'message' => "Aduan baru telah di-dispatch ke {$polsek->name}",
            'channel' => 'system',
            'dispatch_id' => $dispatch->id,
        ]);

        // Send to Polsek users
        foreach ($polsek->users as $user) {
            $this->sendToUser($user, [
                'type' => 'dispatch_new',
                'title' => 'Dispatch Baru',
                'message' => "Aduan baru: {$dispatch->complaint->description}",
                'dispatch_id' => $dispatch->id,
            ]);
        }
    }

    /**
     * Send notification to officer
     */
    public function sendToOfficer(Officer $officer, Dispatch $dispatch): void
    {
        $user = $officer->user;
        
        if (!$user) {
            return;
        }

        // Create system notification
        $this->createNotification([
            'user_id' => $user->id,
            'type' => 'dispatch_assigned',
            'title' => 'Tugas Baru',
            'message' => "Anda ditugaskan untuk menangani aduan: {$dispatch->complaint->description}",
            'channel' => 'system',
            'dispatch_id' => $dispatch->id,
        ]);

        // Send via mobile app (push notification)
        if ($user->device_token) {
            $this->sendPushNotification($user->device_token, [
                'title' => 'Tugas Baru',
                'body' => "Anda ditugaskan untuk menangani aduan",
                'data' => ['dispatch_id' => $dispatch->id],
            ]);
        }

        // Send via WhatsApp
        if ($officer->phone) {
            $this->sendWhatsApp($officer->phone, "Tugas Baru: {$dispatch->complaint->description}");
        }

        // Send via SMS
        if ($officer->phone) {
            $this->sendSMS($officer->phone, "Tugas Baru: {$dispatch->complaint->description}");
        }
    }

    /**
     * Send escalation notification
     */
    public function sendEscalationNotification(Dispatch $dispatch): void
    {
        $polres = $dispatch->polsek->polres;
        
        // Notify Polres
        foreach ($polres->users as $user) {
            $this->sendToUser($user, [
                'type' => 'dispatch_escalated',
                'title' => 'Dispatch Di-escalate',
                'message' => "Dispatch di-escalate dari {$dispatch->polsek->name}",
                'dispatch_id' => $dispatch->id,
            ]);
        }
    }

    /**
     * Create notification record
     */
    private function createNotification(array $data): Notification
    {
        return Notification::create(array_merge([
            'status' => 'pending',
        ], $data));
    }

    /**
     * Send notification to user
     */
    private function sendToUser($user, array $data): void
    {
        $notification = $this->createNotification(array_merge($data, [
            'user_id' => $user->id,
        ]));

        // Mark as sent
        $notification->update([
            'status' => 'sent',
            'sent_at' => now(),
        ]);
    }

    /**
     * Send push notification
     */
    private function sendPushNotification(string $deviceToken, array $data): void
    {
        // Integrate with FCM or similar push notification service
        try {
            // Placeholder for FCM integration
            Log::info("Push notification sent to {$deviceToken}");
        } catch (\Exception $e) {
            Log::error("Push notification failed: " . $e->getMessage());
        }
    }

    /**
     * Send WhatsApp message
     */
    private function sendWhatsApp(string $phone, string $message): void
    {
        // Integrate with WhatsApp Gateway (Twilio, WhatsApp Business API, etc.)
        try {
            // Placeholder for WhatsApp integration
            Log::info("WhatsApp sent to {$phone}: {$message}");
        } catch (\Exception $e) {
            Log::error("WhatsApp send failed: " . $e->getMessage());
        }
    }

    /**
     * Send SMS
     */
    private function sendSMS(string $phone, string $message): void
    {
        // Integrate with SMS Gateway
        try {
            // Placeholder for SMS integration
            Log::info("SMS sent to {$phone}: {$message}");
        } catch (\Exception $e) {
            Log::error("SMS send failed: " . $e->getMessage());
        }
    }
}
