<?php

namespace App\Services;

use App\Models\Complaint;
use App\Models\ComplaintRecording;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class RecordingService
{
    /**
     * Record call from voice logger
     */
    public function recordCall(Complaint $complaint, array $recordingData): ?ComplaintRecording
    {
        try {
            $recordingType = config('services.voice_logger.type', 'panasonic');
            
            // Download recording from voice logger
            $filePath = $this->downloadRecording($recordingData, $recordingType);
            
            if (!$filePath) {
                return null;
            }

            // Store recording
            $storedPath = Storage::disk('public')->put(
                'recordings/' . $complaint->id,
                file_get_contents($filePath)
            );

            // Encrypt if needed
            if (config('services.voice_logger.encrypt', true)) {
                $storedPath = $this->encryptRecording($storedPath);
            }

            return ComplaintRecording::create([
                'complaint_id' => $complaint->id,
                'file_path' => $storedPath,
                'file_name' => $recordingData['file_name'] ?? basename($storedPath),
                'file_size' => Storage::disk('public')->size($storedPath),
                'duration' => $recordingData['duration'] ?? null,
                'source' => $recordingType,
                'recording_id' => $recordingData['recording_id'] ?? null,
                'is_encrypted' => config('services.voice_logger.encrypt', true),
                'recorded_at' => $recordingData['recorded_at'] ?? now(),
            ]);
        } catch (\Exception $e) {
            Log::error("Recording failed: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Download recording from voice logger
     */
    private function downloadRecording(array $recordingData, string $type): ?string
    {
        $apiUrl = config('services.voice_logger.api_url');
        $apiKey = config('services.voice_logger.api_key');

        try {
            switch ($type) {
                case 'panasonic':
                    return $this->downloadFromPanasonic($apiUrl, $apiKey, $recordingData);
                case 'sony':
                    return $this->downloadFromSony($apiUrl, $apiKey, $recordingData);
                case 'ibm':
                    return $this->downloadFromIBM($apiUrl, $apiKey, $recordingData);
                default:
                    return null;
            }
        } catch (\Exception $e) {
            Log::error("Download recording failed: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Download from Panasonic voice logger
     */
    private function downloadFromPanasonic(string $apiUrl, string $apiKey, array $recordingData): ?string
    {
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $apiKey,
        ])->get($apiUrl . '/recordings/' . $recordingData['recording_id']);

        if ($response->successful()) {
            $tempPath = storage_path('app/temp/' . uniqid() . '.wav');
            file_put_contents($tempPath, $response->body());
            return $tempPath;
        }

        return null;
    }

    /**
     * Download from Sony voice logger
     */
    private function downloadFromSony(string $apiUrl, string $apiKey, array $recordingData): ?string
    {
        // Similar implementation for Sony
        return $this->downloadFromPanasonic($apiUrl, $apiKey, $recordingData);
    }

    /**
     * Download from IBM voice logger
     */
    private function downloadFromIBM(string $apiUrl, string $apiKey, array $recordingData): ?string
    {
        // Similar implementation for IBM
        return $this->downloadFromPanasonic($apiUrl, $apiKey, $recordingData);
    }

    /**
     * Encrypt recording file
     */
    private function encryptRecording(string $filePath): string
    {
        // Use Laravel's encryption or custom encryption
        $content = Storage::disk('public')->get($filePath);
        $encrypted = encrypt($content);
        
        $encryptedPath = $filePath . '.encrypted';
        Storage::disk('public')->put($encryptedPath, $encrypted);
        
        // Delete original
        Storage::disk('public')->delete($filePath);
        
        return $encryptedPath;
    }

    /**
     * Get recording URL for playback
     */
    public function getRecordingUrl(ComplaintRecording $recording): string
    {
        if ($recording->is_encrypted) {
            // Decrypt on-the-fly or return decrypted URL
            return route('recordings.play', $recording->id);
        }

        return Storage::disk('public')->url($recording->file_path);
    }
}
